define('atlassian-nps-plugin/js/nps/survey-inline-dialog', [
    'atlassian-nps-plugin/js/nps/util',
    'atlassian-nps-plugin/js/nps/product',
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/amd-shims/templates',
    'jquery',
    'skate',
    'aui/inline-dialog2'
], function (
    Util,
    Product,
    AJS,
    Templates,
    $,
    skate
) {
    "use strict";

    var _npsInlineDialogId = 'nps-survey-inline-dialog';

    var SurveyInlineDialog = {};
    SurveyInlineDialog.eventAttrs = {};

    SurveyInlineDialog.show = function(showOptOutOption) {
        var $surveyEl = generateTemplate({
            id: _npsInlineDialogId,
            trigger: Product.getSurveyTrigger(),
            productName: Product.getProductName(),
            roles: getRoles(),
            showOptOutOption: showOptOutOption
        });

        // NPS-168 In AUI 5.9.x, Inline Dialog is initiated with skate
        // The element might not be ready in the DOM at the time these methods called
        // The ideal solution is defining skate callbacks which is impossible here
        // Skate 0.12.0 is consumed from the product.
        // If the products are upgraded to use Skate 0.14, the issue can be solve in a better way with skate.ready
        setTimeout(function() {
            addSurveyListeners($surveyEl, false);
            safeDialog($surveyEl);

            // buttons should not be focused at beginning.
            preventRankButtonFocused($surveyEl);

            openDialog($surveyEl);
        }, 0);
    };

    SurveyInlineDialog.preview = function() {
        var $surveyEl = generateTemplate({
            id: _npsInlineDialogId,
            trigger: '#nps-preview',
            productName: Product.getProductName(),
            roles: getRoles(),
            showOptOutOption: false
        });

        // NPS-168 In AUI 5.9.x, Inline Dialog is initiated with skate
        // The element might not be ready in the DOM at the time these methods called
        // The ideal solution is defining skate callbacks which is impossible here
        // Skate 0.12.0 is consumed from the product.
        // If the products are upgraded to use Skate 0.14, the issue can be solve in a better way with skate.ready
        setTimeout(function() {
            addSurveyListeners($surveyEl, true);
            safeDialog($surveyEl);
            $surveyEl.find("#nps-send-button").prop('disabled', true);

            // buttons should not be focused at beginning.
            preventRankButtonFocused($surveyEl);

            openDialog($surveyEl);
        }, 0);
    };

    function generateTemplate(options) {
        // Setup target
        var $target = $(options.trigger);
        $target.attr({
            'aria-controls': _npsInlineDialogId
        });

        var $surveyEl = $(Templates.surveyInlineDialog({
            data: options
        }));
        $surveyEl.appendTo('body');

        skate.init($surveyEl[0]);

        return $surveyEl;
    }

    function addSurveyListeners($surveyEl, isPreviewMode) {
        // TODO: More investigation as to why it was done / has to be done like this. It is not ideal.
        $surveyEl.find('.nps-close-icon').on("click", function() {
            $surveyEl.trigger('aui-layer-hide');
        });

        $surveyEl.on('aui-layer-hide', function () {
            if (isPreviewMode === false) {
                completeSurvey();
            }
        });

        // Rank selection
        var $rankButtons = $surveyEl.find('#nps-rank-buttons > .rank-button');
        $rankButtons.click(function (event) {
            $rankButtons.removeClass("nps-rank-selected");

            var $buttonClicked = $(event.target);
            $buttonClicked.addClass("nps-rank-selected");
            $buttonClicked.blur();

            if (isPreviewMode === false) {
                SurveyInlineDialog.eventAttrs.rank = parseInt($buttonClicked.text(), 10);
                window.onbeforeunload = completeSurvey;
            }

            $surveyEl.find('.comment-session').show();
        });

        if (isPreviewMode === false) {
            // Send button
            var $sendSurveyButton = $surveyEl.find("#nps-send-button");
            $sendSurveyButton.on("click", onSendButtonClicked);

            // Done button
            var $doneSurveyButton = $surveyEl.find("#nps-done-button");
            $doneSurveyButton.on("click", onDoneButtonClicked);

            // Dismissed Forever
            $surveyEl.find(".nps-dismiss-forever").on("click", onDismissedForeverClicked);
        }
    }

    function getRoles() {
        var prefix = [
            {
                value: "",
                text: AJS.I18n.getText("nps.survey.question.role.choose"),
                disabled: true,
                selected: true
            }
        ];

        var roles = [
            {
                value: "software-engineer",
                text: "Software Engineering"
            },
            {
                value: "product-manager",
                text: "Product Management"
            },
            {
                value: "qa",
                text: "Quality Assurance"
            },
            {
                value: "design",
                text: "Design"
            },
            {
                value: "management",
                text: "Management"
            },
            {
                value: "sys-admin",
                text: "Systems Administration"
            }
        ];

        var suffix = [
            {
                value: "other",
                text: "Other"
            }
        ];

        return prefix.concat(Util.kfyShuffle(roles)).concat(suffix);
    }

    function completeSurvey() {
        var needSave = !$.isEmptyObject(SurveyInlineDialog.eventAttrs);

        if (needSave) {
            AJS.trigger("/nps/surveyCompleted", SurveyInlineDialog.eventAttrs);
            delete SurveyInlineDialog.eventAttrs;
        } else {
            // for now try to re-use "/nps/cancelSurvey" with context is "/nps/cancelSurvey"
            // This context should be remove When remove notification flag.
            // "/nps/cancelSurvey" function should also refactored to remove checking context
            AJS.trigger("/nps/cancelSurvey", "notification");
        }

        $('#nps-survey-inline-dialog').remove();
    }

    function safeDialog($surveyEl) {
        AJS.dialog2.on("show", function() {
            $surveyEl.hide();
        });

        AJS.dialog2.on("hide", function() {
            $surveyEl.show();
        });

        AJS.bind("show.dialog", function() {
            $surveyEl.hide();
        });

        AJS.bind("hide.dialog", function() {
            $surveyEl.show();
        });

        AJS.bind("remove.dialog", function() {
            $surveyEl.show();
        });
    }

    function onSendButtonClicked(e) {
        e.preventDefault();

        $(this).prop('disabled', true);

        var comment = $.trim($("#nps-comment").val());
        if (comment) {
            SurveyInlineDialog.eventAttrs.comment = comment;
        }

        var role = $.trim($('#nps-role').val());
        if (role) {
            SurveyInlineDialog.eventAttrs.role = role;
        }

        if (Product.isServerMode()) {
            completeSurvey();
            $('#nps-survey-inline-dialog').remove();
        } else {
            $('.survey-screen').hide();
            $('.survey-screen.screen-2').show();
        }
    }

    function onDoneButtonClicked(e) {
        e.preventDefault();

        var allowContact = $('#nps-allow-contact').is(':checked');
        if (allowContact) {
            SurveyInlineDialog.eventAttrs.allowContact = allowContact;
        }

        completeSurvey();
        $('#nps-survey-inline-dialog').remove();
    }

    function preventRankButtonFocused($surveyEl) {
        setTimeout(function() {
            $surveyEl.find(".aui-button").blur();
        }, 50);
    }

    function onDismissedForeverClicked() {
        AJS.trigger("/nps/dismissSurvey", "dismiss");
        $('#nps-survey-inline-dialog').remove();
    }

    function closeDialog($surveyEl) {
        // This check is necessary because of a change in API that occured for AUI 5.9.
        if ($surveyEl[0].hide) {
            $surveyEl[0].hide();
        } else {
            $surveyEl[0].removeAttribute('open');
        }
    }

    function openDialog($surveyEl) {
        // This check is necessary because of a change in API that occured for AUI 5.9.
        if ($surveyEl[0].show) {
            $surveyEl[0].show();
        } else {
            $surveyEl[0].open = true;
        }
    }

    return SurveyInlineDialog;
});