AJS.namespace("JIRA.Dashboard.Entities");

JIRA.Dashboard.Entities.DirectoryViewModel = Backbone.Model.extend({
    initialize: function(attributes,options) {
        this.directoryItems = options.directoryItems;
        this.initializeStates();

        this.listenTo(this.directoryItems, {
            "batchRetrieving": this.handleBatchRetrievalStart,
            "retrieve": this.handleRetrieval,
            "retrieveFailure": this.handleRetrievalFailure,
            "reset": this.initializeStates
        });
    },

    initializeStates: function () {
        this.set({
            retrievalInProgress: false,
            retrievalFailed: false,
            hasUnrenderedData: false,
            directoryIsVisible: false,
            searchText: "",
            selectedCategory: JIRA.Dashboard.Directory.Constants.getDefaultCategoryName(),
            categorisedItems: {},
            filteredCategorisedItems: {}
        });
    },

    handleBatchRetrievalStart: function() {
        this.set("retrievalInProgress", true);
        this.trigger("messageUpdate");
    },

    handleRetrieval: function(data) {
        this.checkRetrievalCompleted();
        if(this.get("directoryIsVisible") && data.length > 0)
        {
            this.set("hasUnrenderedData", true);
        }

        if(!this.get("retrievalInProgress") || this.get("hasUnrenderedData"))
        {
            this.trigger("messageUpdate");
        }

        if(!this.get("directoryIsVisible") && data.length > 0)
        {
            this.loadData();
        }
    },

    loadData: function() {
        this.updateCategorisedItems();
        this.trigger("dataUpdate");
    },

    loadUnrenderedData: function() {
        this.loadData();

        this.set("hasUnrenderedData", false);
        this.trigger("messageUpdate");
    },

    handleRetrievalFailure: function() {
        this.checkRetrievalCompleted();
        if(!this.get("retrievalInProgress"))
        {
            this.trigger("messageUpdate");
        }
    },

    checkRetrievalCompleted: function() {
        if(this.directoryItems.retrievalCompleted())
        {
            this.set("retrievalInProgress", false);
            this.set("retrievalFailed", this.directoryItems.hasSomeFailure());
        }
    },

    setDirectoryVisible: function() {
        this.set("directoryIsVisible", true);
    },

    setDirectoryNotVisible: function() {
        this.set("directoryIsVisible", false);
    },

    setSearchText: function(searchText) {
        this.set("searchText", searchText.toLowerCase());
        this.updateFilteredCategorisedItems();
    },

    getSearchText: function() {
        return this.get("searchText");
    },

    setSelectedCategory: function(category) {
        this.set("selectedCategory", category);
    },

    getSelectedCategory: function() {
        return this.get("selectedCategory");
    },

    updateCategorisedItems: function() {
        this.set("categorisedItems", this.directoryItems.getCategorisedItems());
        this.updateFilteredCategorisedItems();
    },

    getCategorisedItems: function() {
        return this.get("categorisedItems");
    },

    updateFilteredCategorisedItems: function() {
        this.set("filteredCategorisedItems", this.filterCategorisedItems(this.getSearchText()));
    },

    getFilteredCategorisedItems: function() {
        return this.get("filteredCategorisedItems");
    },

    filterCategorisedItems: function(searchText) {
        if(!searchText) {
            return this.getCategorisedItems();
        }

        var filtered = {};
        _.each(this.getCategorisedItems(), function(items, category){
            filtered[category] = _.filter(items, function(item){
                return this.matchesSearchText(searchText, item);
            }, this);
        }, this);
        return filtered;
    },

    matchesSearchText: function(searchText, item) {
        searchText = searchText.toLowerCase();
        return item.get("id").toLowerCase().indexOf(searchText) > -1
            || item.get("titleLabel").toLowerCase().indexOf(searchText) > -1
            || item.get("authorLabel").toLowerCase().indexOf(searchText) > -1
            || item.get("description").toLowerCase().indexOf(searchText) > -1;
    },

    getCategoryCount: function() {
        return this.transformCategoryCountForDisplay(this.getCategorisedItems());
    },

    getFilteredCategoryCount: function() {
        return this.transformCategoryCountForDisplay(this.getFilteredCategorisedItems());
    },

    transformCategoryCountForDisplay: function (categorisedItems) {
        var categoryCount = [];
        _.each(categorisedItems, function (items, category) {
            categoryCount.push({
                label: category,
                count: items.length
            });
        });

        return _.sortBy(categoryCount, "label");
    },

    getItemsForCategory: function(category) {
        return this.getCategorisedItems()[category];
    },

    getVisibleItemsForCategory: function(category) {
        return this.getFilteredCategorisedItems()[category];
    }
});