define("servicedesk/internal/agent/settings/automation/view/rule-component-summary-view", [
    "jquery",
    "automation/backbone-brace"
], function (
    $,
    Brace
) {

    return Brace.View.extend({
        className: "component-summary-container",

        template: ServiceDesk.Templates.Agent.Settings.componentSummary,
        tooltipItemTemplate: ServiceDesk.Templates.Agent.Settings.componentSummaryTooltipItem,

        events: {
            "click .delete-component-button": "_deleteComponent",
            "click .component-move-down": "_moveDownComponent",
            "click .component-move-up": "_moveUpComponent",
            "click .component-summary": "selectComponent",
            "keydown .delete-component-button": "_keydownDelete",
            "keydown .component-move-down": "_keydownDown",
            "keydown .component-move-up": "_keydownUp",
            "keydown": "_keydownContainer"
        },

        namedEvents: [
            "componentSelected",
            "componentDelete",
            "componentMoveUp",
            "componentMoveDown"
        ],

        initialize: function (opts) {
            this.isFirst = opts.isFirst;
            this.isLast = opts.isLast;

            this.listenTo(this.model, "modelChanged", _.bind(this._reload, this));
        },

        render: function () {
            var title = this._getTitle();
            var canDelete = this.isFirst === false;
            var canMoveDown = this.isLast === false;
            var canMoveUp = this.isFirst === false;
            this.$el.html(this.template({
                title: title,
                canDelete: canDelete,
                canMoveDown: canMoveDown,
                canMoveUp: canMoveUp
            }));

            var componentModelCollection = this.model.getCollection();
            if (componentModelCollection.length <= 0) {
                var validation = this.model.getValidation();

                var emptyComponentBody = ServiceDesk.Templates.Agent.Settings.emptyComponentSummaryBody({
                    defaultMessage: this._getEmptyComponentMessage(),
                    hasError: !!validation && validation.hasError()
                });

                // update rule component tooltip text
                this._updateTooltipText(this._getEmptyComponentMessage());

                this.$('.component-summary-configurations ol').append($(emptyComponentBody));
            } else {
                componentModelCollection.each(_.bind(function(model) {
                    var validation = model.getValidation();
                    var componentSummaryBody = ServiceDesk.Templates.Agent.Settings.componentSummaryBody({
                        config: model.toJSON(),
                        hasError: !!validation && validation.hasError()
                    });

                    this.$('.component-summary-configurations ol').append($(componentSummaryBody));

                    // update rule component tooltip text
                    var tooltipText = this._computeTooltipText(model);
                    this._updateTooltipText(tooltipText);

                }, this));
            }

            // apply tooltips on rule component boxes
            this._applyTooltips();

            this.$el.on('focus', '*', _.bind(this._onFocus, this));
            this.$el.on('blur', '*', _.bind(this._onBlur, this));

            return this;
        },

        select: function () {
            this.$(".component-summary").addClass("selected");
        },

        unselect: function () {
            this.$(".component-summary").removeClass("selected");
        },

        isSelected: function() {
            return this.$('.component-summary').hasClass("selected");
        },

        focus: function () {
            this.$('.component-summary').focus();
        },

        _onFocus: function() {
            this.$('.component-summary').addClass('focus');
        },

        _onBlur: function() {
            this.$('.component-summary').removeClass('focus');
        },

        selectComponent: function (e) {
            // We will trigger the event only if it's called manually (no event)
            // or if the event is already not handled already. This is necessary
            // so that clicking on the delete button does not also trigger this.
            if (!e || !e.isDefaultPrevented()) {
                e && e.preventDefault();
                this.triggerComponentSelected(this._buildEventData());
            }
        },

        _reload: function() {
            var isSelected = this.isSelected();
            this.render();
            if (isSelected) {
                _.defer(_.bind(this.select, this));
            }
        },

        _deleteComponent: function(e) {
            e && e.preventDefault();
            this.triggerComponentDelete(this._buildEventData());
        },

        _moveUpComponent: function(e) {
            e && e.preventDefault();
            this.triggerComponentMoveUp(this._buildEventData());
        },

        _moveDownComponent: function(e) {
            e && e.preventDefault();
            this.triggerComponentMoveDown(this._buildEventData());
        },

        _keydownContainer: function (e) {
            if (e.keyCode == 13) {
                this.selectComponent(e);
            }
        },

        _keydownUp: function (e) {
            if (e.keyCode == 13) {
                this._moveUpComponent(e);
            }
        },

        _keydownDown: function (e) {
            if (e.keyCode == 13)
            {
                this._moveDownComponent(e);
            }
        },

        _keydownDelete: function (e) {
            if (e.keyCode == 13)
            {
                this._deleteComponent(e);
            }
        },

        /* Other methods */

        _getTitle: function() {
            switch (this.model.getType()) {
                case "whenHandler":
                    return AJS.I18n.getText("sd.automation.component.title.trigger");
                case "ifCondition":
                    if (this.isFirst)
                    {
                        return AJS.I18n.getText("sd.automation.component.title.matcher.condition.if");
                    }
                    else if (this.isLast && this.model.getCollection().length == 0)
                    {
                        return AJS.I18n.getText("sd.automation.component.title.matcher.condition.else");
                    }
                    else
                    {
                        return AJS.I18n.getText("sd.automation.component.title.matcher.condition.else.if");
                    }
                case "thenAction":
                    return AJS.I18n.getText("sd.automation.component.title.action");
            }
        },

        _getEmptyComponentMessage: function() {
            switch (this.model.getType()) {
                case "whenHandler":
                    return AJS.I18n.getText("sd.automation.component.empty.message.when");
                case "ifCondition":
                    if (this.isFirst)
                    {
                        return AJS.I18n.getText("sd.automation.component.empty.message.if");
                    }
                    else
                    {
                        return AJS.I18n.getText("sd.automation.component.empty.message.else");
                    }
                case "thenAction":
                    return AJS.I18n.getText("sd.automation.component.empty.message.then");
            }

            return "";
        },

        _buildEventData: function() {
            return {
                model: this.model,
                view: this
            }
        },

        _applyTooltips: function() {
            this.$('.automation-tooltip').tooltip({
                html: true,
                gravity: 'n',
                title: 'data-tooltip',
                delayIn:0
            });
        },

        _computeTooltipText: function(model) {
            var tooltipText = model.getName() ? model.getName() : model.getModuleKey();
            if (model.getLabel())
            {
                tooltipText += ": " + model.getLabel();
            }
            return tooltipText;
        },

        _updateTooltipText: function(tooltipText) {
            var currentTitle = this.$('.component-summary').attr('data-tooltip') || "";
            this.$('.component-summary').attr('data-tooltip', currentTitle + this.tooltipItemTemplate({tooltipText: tooltipText}));
        }
    });

});