define("servicedesk/internal/agent/settings/automation/view/rule-component-details-panel-form-view", [
    "jquery",
    "automation/underscore",
    "automation/backbone",
    "automation/backbone-brace",
    "servicedesk/internal/agent/settings/automation/util/trace/trace",
    "servicedesk/internal/agent/settings/automation/util/form-mixin/form-mixin",
    "servicedesk/internal/agent/settings/automation/util/analytics/analytics"
], function (
    $,
    _,
    Backbone,
    Brace,
    Trace,
    FormMixin,
    tracker
    ) {

    return Brace.View.extend({
        template: ServiceDesk.Templates.Agent.Settings.editConfigurationForm,
        tagName: "div",
        className: "component-details-configurations",
        mixins: [FormMixin],
        events: {
            "submit": "_submit",
            "click .cancel": "_cancel"
        },

        namedEvents: [
            "cancel",
            "submitSuccess",
            "webFormRenderError",
            "webFormSerializeError",
            "webFormValidateError"
        ],

        initialize: function (models) {
            this.dispatcher = _.clone(Backbone.Events);
            this.ruleModel = models.ruleModel;
            this.isExistingRule = models.isExistingRule;
        },

        render: function () {
            this.dispatcher.trigger("destroy");
            this.$el.html(this.template({
                module: this.model.toJSON(),
                hasError: !!this.model.getValidation() && this.model.getValidation().hasError()
            }));

            this.showSpinner();

            try {
                var webFormFn = require(this.model.getWebFormModule());
                this.webForm = webFormFn(this._controller());
                var validation = this.model.getValidation();

                var context = {
                    ruleModel: this.ruleModel,
                    componentModel: this.model,
                    isExistingRule: this.isExistingRule
                };

                var result = _.bind(this.webForm.render, this.webForm, this.model.getData(), this._getErrors(validation), context)();

                // We support both deferred and non-deferred results from the web form.
                $.when(result).done(_.bind(function() {
                    this.hideSpinner();
                    this.$(".js-status").remove();
                    Trace("automation.component.webform.render.complete")
                }, this));
            } catch(err) {

                tracker.trackEvent("automation.ruleset.edit.component.configuration.load.error");

                var type = this.model.getType();
                var message;
                switch (type) {
                    case "whenHandler":
                        message = AJS.I18n.getText('sd.automation.error.webform.render.when');
                        break;
                    case "ifCondition":
                        message = AJS.I18n.getText('sd.automation.error.webform.render.if');
                        break;
                    case "thenAction":
                        message = AJS.I18n.getText('sd.automation.error.webform.render.then');
                }

                this.$("button").prop("disabled", true);
                this.triggerWebFormRenderError(this.model, err);
                this._renderRenderError(message);
                console.log("Error rendering form", this.model, err);
            }

            return this;
        },

        hide: function () {
            this.$el.addClass("hidden");
        },

        dispose: function() {
            this.remove();
            this.webForm && this.webForm.dispose && this.webForm.dispose();
        },

        focus: function() {
            if (this.webForm) {
                // Attempt to delegate the focus to the webform otherwise focus on the first possible element.
                if (this.webForm.focus) {
                    this.webForm.focus();
                } else {
                    // There is always at least one focusable element due to the confirm and delete elements
                    this.$(':focusable')[0].focus();
                }
            }
        },

        _renderRenderError: function(errorMessage) {
            $(this._controller().el).html(ServiceDesk.Templates.Agent.Settings.error({
                error: errorMessage
            }));
        },

        _renderError: function(errorMessage) {
            this.$('.errors').append(ServiceDesk.Templates.Agent.Settings.error({
                error: errorMessage
            }));
        },

        _submit: function (e) {
            e.preventDefault();
            this.$('.errors').empty();
            this.disableSubmit();
            this._validateSafely().done(_.bind(function () {
                this._serializeSafely().done(_.bind(function (serializedData) {
                    this.serializedData = serializedData;
                    this.model.validateConfiguration(serializedData)
                        .done(_.bind(function(data) {
                            tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".configuration.edit.save.succeeded");
                            this.model.set(data);
                            this.triggerSubmitSuccess(this.model);
                            // only fire event if Include linked issue's details checkbox displays
                            if(this.serializedData.data.includeLinkedIssueDetails === true) {
                                tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".includeLinkedIssueDetails.checked.confirm");
                            } else if(this.serializedData.data.includeLinkedIssueDetails === false) {
                                tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".includeLinkedIssueDetails.unchecked.confirm");
                            }
                            this.enableSubmit();
                        }, this))
                        .fail(_.bind(function(xhr) {

                            tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".configuration.edit.save.failed");

                            var responseText = xhr.responseText;
                            var componentResponse = JSON.parse(responseText);
                            var errors = this._getErrors(componentResponse.validation);
                            this.dispatcher.trigger("error", errors);
                            this.enableSubmit();
                        }, this));
                }, this));
            }, this)).fail(_.bind(this.enableSubmit,this));
        },

        _cancel: function (e) {
            e.preventDefault();

            tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".configuration.edit.cancel");

            this.triggerCancel();
        },

        /*
         * Calls the webform's validate function safely by catching any exception thrown by the function.
         * If the validate function throws an exception the validation will fail.
         * Will fire a webFormValidateError if the validate function throws an exception.
         */
        _validateSafely: function () {
            var deferred = $.Deferred();

            if (!this.webForm.validate) {
                return deferred.resolve();
            }

            try {
                this.webForm.validate(deferred, this.ruleModel);

                return deferred;
            } catch(err) {

                var type = this.model.getType();
                var message;
                switch (type) {
                    case "whenHandler":
                        message = AJS.I18n.getText('sd.automation.error.webform.validate.when');
                        break;
                    case "ifCondition":
                        message = AJS.I18n.getText('sd.automation.error.webform.validate.if');
                        break;
                    case "thenAction":
                        message = AJS.I18n.getText('sd.automation.error.webform.validate.then');
                }

                this._renderError(message);
                console.log("Error validating form", this.model, err);
                this.triggerWebFormValidateError(this.model, err);
                return deferred.reject(err);
            }
        },

        /*
         * Calls the webform's serialize function safely by catching any exception thrown by the function.
         * If the serialize function throws an exception the serialization will fail.
         * Will fire a webFormSerializeError if the serialize function throws an exception.
         */
        _serializeSafely: function () {
            var deferred = $.Deferred();
            try {
                var serializedModel = {
                    moduleKey: this.model.getModuleKey(),
                    effectiveProvides: this.ruleModel.getEffectiveProvides()
                };

                var serializeDeferred = this.webForm.serialize();

                // We support both deferred and non-deferred results from the serialize().
                $.when(serializeDeferred).done(function(data) {
                    serializedModel.data = data;
                    deferred.resolve(serializedModel);
                });
                return deferred;
            } catch (err) {

                var type = this.model.getType();
                var message;
                switch (type) {
                    case "whenHandler":
                        message = AJS.I18n.getText('sd.automation.error.webform.serialize.when');
                        break;
                    case "ifCondition":
                        message = AJS.I18n.getText('sd.automation.error.webform.serialize.if');
                        break;
                    case "thenAction":
                        message = AJS.I18n.getText('sd.automation.error.webform.serialize.then');
                }

                this._renderError(message);
                console.log("Error serializing form", this.model, err);
                this.triggerWebFormSerializeError(this.model, err);
                return deferred.reject(err);
            }
        },

        _on: function (event, callback, context) {
            this.dispatcher.on(event, callback, context)
        },

        _off: function (event, callback, context) {
            this.dispatcher.off(event, callback, context)
        },

        _renderGlobalError: function(errorMessage) {
            AJS.messages.error(this.$('.error-bar'), {
                title: 'An error occurred',
                body: '<p>' + errorMessage + '</p>'
            });
        },

        _renderFieldError: function(message, field) {
            var $input = this.$('[name="' + field + '"]');
            var $error = $('<div>').text(message).addClass('error');
            $input.after($error);
        },

        _renderJqlFieldError: function(message, field) {
            var $input = this.$('[name="' + field + '"]');
            var $error = $('<div>').addClass('aui-message aui-message-error');

            if(_.isArray(message) && message.length > 1) {
                var $errorList = $('<ul>');
                jQuery.each( message, function( i, val ) {
                    var $errorItem = $('<li>');
                    $errorItem.text(val);
                    $errorList.append($errorItem);
                });
                $error.append($errorList);
            } else {
                $error.text(message);
            }

            $input.after($error);
        },

        _controller: function () {
            return {
                el: this.$('.configuration-form-content')[0],
                formData: this.model.toJSON().data,
                on: _.bind(this._on, this),
                off: _.bind(this._off, this),
                renderFieldError: _.bind(this._renderFieldError, this),
                renderJqlFieldError: _.bind(this._renderJqlFieldError, this),
                renderGlobalError: _.bind(this._renderGlobalError, this)
            }
        },

        _getErrors: function(validationModel) {
            if (!validationModel) {
                return {
                    fieldErrors: {},
                    globalErrors: []
                }
            }

            var validationModelAsJson = validationModel.toJSON ? validationModel.toJSON() : validationModel;
            var fieldErrors = validationModelAsJson.fieldErrors;
            var globalErrors = validationModelAsJson.globalErrors;

            return {
                fieldErrors: fieldErrors || {},
                globalErrors: globalErrors || []
            }
        }
    });
});