define("servicedesk/internal/agent/settings/automation/page/automation-rule-set-page-view", [
    "jquery",
    "automation/underscore",
    "automation/backbone-brace",
    "servicedesk/internal/agent/settings/automation/feature/confirm-dialog/confirm-dialog",
    "servicedesk/internal/agent/settings/automation/util/trace/trace",
    "servicedesk/internal/agent/settings/automation/view/rule-view",
    "servicedesk/internal/agent/settings/automation/view/rule-component-details-panel-view",
    "servicedesk/internal/agent/settings/automation/view/rule-settings-dialog-view",
    "servicedesk/internal/agent/settings/automation/util/position-utils/position-utils",
    "servicedesk/internal/agent/settings/automation/model/metadata/project-run-as-user-settings-model",
    "servicedesk/internal/agent/settings/automation/view/help/rule-set-help",
    "servicedesk/internal/agent/settings/automation/model/help-model",
    "servicedesk/internal/agent/settings/automation/internal-automation-service",
    "servicedesk/internal/agent/settings/automation/util/html-utils/html-utils",
    "servicedesk/internal/agent/settings/automation/util/async-utils/async-utils",
    "servicedesk/internal/agent/settings/automation/util/analytics/analytics",
    "servicedesk/internal/agent/settings/automation/util/form-mixin/form-mixin",
    "servicedesk/internal/agent/settings/automation/util/navigator/navigator",
    "servicedesk/internal/agent/settings/automation/util/url-utils/url-utils"
], function (
    $,
    _,
    Brace,
    ConfirmationDialog,
    Trace,
    RuleView,
    RuleComponentDetailsView,
    RuleSettingsDialog,
    PositionUtils,
    RunAsUserContextContainerModel,
    RuleSetHelp,
    HelpModel,
    AutomationService,
    HtmlUtils,
    AsyncUtils,
    tracker,
    FormMixin,
    Navigator,
    UrlUtils
) {
    return Brace.View.extend({
        mixins: [FormMixin],
        template: ServiceDesk.Templates.Agent.Settings.ruleSet,
        headerErrorTemplate: ServiceDesk.Templates.Agent.Settings.headerError,
        bodyErrorTemplate: ServiceDesk.Templates.Agent.Settings.bodyError,
        preSaveDialog: ServiceDesk.Templates.Agent.Settings.preSaveDialog,
        defaultToCurrentUserFlagTemplates: {
            success: {
                title: ServiceDesk.Templates.Agent.Settings.defaultToCurrentUserSuccessTitle,
                body: ServiceDesk.Templates.Agent.Settings.defaultToCurrentUserSuccessBody
            },
            error: {
                title: ServiceDesk.Templates.Agent.Settings.defaultToCurrentUserErrorTitle,
                body: ServiceDesk.Templates.Agent.Settings.defaultToCurrentUserErrorBody
            }
        },
        events: {
            "click": "_componentUnselected",
            "click .ruleset-save-button": "save",
            "click .automation-rule-set-cancel": "cancel",
            "click .delete-rule-set-link": "_deleteRuleSetClicked",

            "keyup .automation-ruleset-name": "_onNameChange",
            "change .automation-ruleset-name": "_onNameChange",
            "paste .automation-ruleset-name": "_onNameChange",

            "keyup .automation-ruleset-description": "_onDescriptionChange",
            "change .automation-ruleset-description": "_onDescriptionChange",
            "paste .automation-ruleset-description": "_onDescriptionChange",

            "submit .automation-settings-header-container form": "_onSubmit",

            "change #enable-rule-checkbox": "_onEnableRuleSetChange"
        },

        initialize: function() {
            this.listenTo(this.model, "fetchedFromServer", _.bind(function() {
                // We only attempt to disable the save button if the model isn't a blueprint
                if (!this.model.isBlueprint()) {
                    var $saveButton = this.$(".ruleset-save-button");

                    this.originalModel = this.model.getRuleSet().serialize();

                    // we disable the save button on fetch because the model is reset
                    this._disableElement($saveButton)
                }
            }, this));

            this.listenTo(this.model, "ruleSetChanged", _.bind(function() {
                this.refreshSaveButtonState();
                this.refreshRuleSettingsState();
                this._showErrors();
            }, this));

            // If there is a refresh link in the error, we trigger a refresh of the model
            // we will do a full page refresh if the request fails for some reason
            // We use rfrsh instead of refresh so it doesn't get accidentally translated
            this.$el.on('click', '.automation.error a.rfrsh', _.bind(function(e) {
                e.preventDefault();
                this.model.fetch()
                    .done(_.bind(function() {
                        this.render();
                    }, this))
                    .fail(function() {
                        window.location.reload();
                    });
            }, this));
        },

        hasChanged: function() {
            // if this.model.getRuleSet() is null then we always assume nothing has changed as you can't edit the rule set.
            return !!this.model.getRuleSet() && !_.isEqual(this.originalModel, this.model.getRuleSet().serialize());
        },

        render: function () {
            // This flag needs to be reset in case it was set to false by a previous edit of a rule.
            this.ignoreDirtySaveCheck = false;

            // This flag is to communicate to child views on the life cycle of the automation rule set edit view.
            // Certain events for the help view should only be fire once per instance of the automation rule set edit view.
            // This is also necessary so that expand/collapse information for the help text is displayed properly if localstorage is not available
            var helpViewIsInitialized = !!this.helpView;
            var helpViewIsExpanded = this.helpView && this.helpView.isExpanded();

            var isBlueprint = this.model.isBlueprint();

            // For blueprints the placeholder is the name of the blueprint.
            this.placeholder = isBlueprint && this.model.getBlueprint().getName() && this.model.getBlueprint().getName() !== "" ?
                AJS.I18n.getText("sd.automation.rule.set.name.placeholder", this.model.getBlueprint().getName()) :
                AJS.I18n.getText("sd.automation.rule.set.name.placeholder.default");


            var ruleSetModel = this.model.getRuleSet();

            var helpModel;
            if (this.model.isBlueprint()) {
                helpModel = this.model.getBlueprint().getHelp();
            } else {
                helpModel = new HelpModel({
                    title: AJS.I18n.getText("sd.automation.blueprint.help.title.default"),
                    description: AJS.I18n.getText("sd.automation.blueprint.help.description.default"),
                    detailAsHtml: ServiceDesk.Templates.Agent.Settings.Blueprints.Help.Default.defaultHelp()
                });
            }

            var isRuleSetEnabled = this.model.getRuleSet().getRules().map(
                function (rule) {
                    return rule.getEnabled();
                }).every(Boolean);
            var isExistingRuleSet = !this.model.getRuleSet().isNew();

            this.$el.html(this.template({
                ruleSet: ruleSetModel.toJSON(),
                projectKey: this.model.getProjectKey(),
                namePlaceholder: this.placeholder,
                isRuleSetEnabled: isRuleSetEnabled,
                isExistingRuleSet: isExistingRuleSet
            }));

            // Render the help view
            this.helpView = new RuleSetHelp({
                model: helpModel,
                hasInitializedBefore: helpViewIsInitialized,
                forceCollapsed: !helpViewIsExpanded // This is only used when hasInitializedBefore is true
            }).render();


            this.helpView.$el.detach().appendTo(this.$('.rule-set-help-wrapper'));

            var ruleSet = this.$('.rule-set-container');
            this.ruleViews = [];

            // We always assume there are always at least 1 rule. The user cannot create a second rule in the UI at the moment.
            this.model.getRuleSet().getRules().each(_.bind(function (rule, index) {
                var ruleView = new RuleView({
                    model: rule
                }).render();
                ruleSet.append(ruleView.el);
                this.listenTo(ruleView, "componentSelected", _.bind(this._componentSelected, this));
                this.listenTo(ruleView, "componentDeleted", _.bind(this._componentDeleted, this));
                this.ruleViews.push(ruleView);
            }, this));

            this._initSettingsDialog();

            this._showErrors();

            // We defer refreshing save button until after skate detects the class mutation.
            _.defer(_.bind(this.refreshSaveButtonState, this));

            $(document).trigger("automationPageRendered", "ruleSetPage");

            this.$('.automation-ruleset-name').focus();

            Trace("automation.edit.page.render.complete");

            return this;
        },

        save: function () {

            // disable the save button so they can't click it again triggering 2 save operations
            var $saveButton = this.$(".ruleset-save-button");
            this._disableElement($saveButton);
            this.showSpinner();
            var isEnabled = $("#enable-rule-checkbox").is(":checked");
            this._enableAllRules(isEnabled);

            // validate the ruleset
            AsyncUtils.execDeferredFunctionQueue([
                function () {
                    return this._validateRuleSet();
                },
                function () {
                    return AutomationService.resolvePreconditions(this.model.toJSON());
                },
                function () {
                    return this._defaultToCurrentUserIfEmpty();
                }
            ], [], this).done(function () {
                tracker.trackEvent("automation.ruleset.edit.save.succeeded", {
                    ruleSetId: this.model.attributes.id
                });
                // and save the ruleset
                // This will re-enable the save button for us by calling this.render() if the save fails
                this._saveRuleSet();
            }).fail(function () {
                // Re-enable the save button
                this._enableElement($saveButton);
                this.hideSpinner();
            });
        },

        cancel: function() {
            tracker.trackEvent("automation.ruleset.edit.cancel", {
                ruleSetId: this.model.attributes.id
            });

            this.ignoreDirtySaveCheck = true;

            this._goBackToPreviousPage();
        },

        _deleteRuleSetClicked: function(e) {
            e.preventDefault();
            var dialog = ConfirmationDialog.create({
                title: AJS.I18n.getText("sd.automation.rule.set.delete.confirmation.dialog.title"),
                explanation: AJS.I18n.getText("sd.automation.rule.set.delete.confirmation.dialog.explanation"),
                action: _.bind(this._handleDelete, this),
                mode: ConfirmationDialog.type.DELETE
            });

            dialog.on("cancel", _.bind(function(e) {
                tracker.trackEvent("automation.list.ruleset.delete.cancel", {
                    ruleSetId: this.model.getId()
                });
            }, this));

            dialog.on("escape", _.bind(function(e) {
                tracker.trackEvent("automation.list.ruleset.delete.cancel", {
                    ruleSetId: this.model.getId()
                });
            }, this));

            tracker.trackEvent("automation.list.ruleset.delete", {
                ruleSetId: this.model.getId()
            });

            this.listenTo(dialog, "submitCompleted", _.bind(function() {
                this._goBackToPreviousPage();
            }, this));

            dialog.show();
        },

        _handleDelete: function() {
            tracker.trackEvent("automation.list.ruleset.delete.confirm", {
                ruleSetId: this.model.getId()
            });
            return this.model.getRuleSet().del();
        },

        _onDescriptionChange: function(e) {
            _.defer(_.bind(function() {
                // We need the defer here because the paste event is fired before the screen is updated
                var description = $(e.target).val();

                if (!_.isEqual(description, this.model.getRuleSet().getDescription())) {
                    this.model.getRuleSet().setDescription(description);
                }
            }, this));
        },

        _onNameChange: function(e) {
            _.defer(_.bind(function() {
                // We need the defer here because the paste event is fired before the screen is updated
                var name = $(e.target).val();

                if (!_.isEqual(name, this.model.getRuleSet().getName())) {
                    this.model.getRuleSet().setName(name);
                }
            }, this));
        },

        /**
         * Disabling default form submission. We want to handle it with the {@link #save} method.
         * (we have to do this for Safari, and IE 9/10/11 event handling)
         */
        _onSubmit: function(e) {
            e.preventDefault();
        },


        _onHelpViewCollapsed: function() {
            var $ruleSetHelpExpandButton = this.$('.automation-rule-set-help-expand-button');
            $ruleSetHelpExpandButton.show();
        },

        _validateRuleSet: function() {
            var $deferred = $.Deferred();

            this.model.getRuleSet().validate()
                .done(_.bind(function () {
                    $deferred.resolve();
                }, this))
                .fail(_.bind(function (xhr) {
                    $deferred.reject();

                    tracker.trackEvent("automation.ruleset.edit.save.failed.validation", {
                        ruleSetId: this.model.attributes.id
                    });

                    // If save fails we render the validation errors
                    var data = JSON.parse(xhr.responseText);
                    this.model.getRuleSet().set(data);
                    this.render();
                }, this));

            return $deferred;
        },

        _defaultToCurrentUserIfEmpty: function() {
            var $deferred = $.Deferred();
            var projectKey = AJS.Meta.get("projectKey");

            var projectRunAsUserSettings = new RunAsUserContextContainerModel();
            projectRunAsUserSettings.defaultToCurrentUserIfEmpty(projectKey)
                .done(_.bind(function(response) {
                    $deferred.resolve();

                    // if the setting has just been auto-updated because of its emptiness
                    if (response.isUpdated) {
                        this._showDefaultToCurrentUserIfEmptySuccess(response);
                    }

                }, this))
                .fail(_.bind(function() {
                    this._showDefaultToCurrentUserIfEmptyError();
                }, this));

            return $deferred;
        },

        _showDefaultToCurrentUserIfEmptySuccess: function(response) {
            var title = this.defaultToCurrentUserFlagTemplates.success.title({user: response.config.user});
            var body = this.defaultToCurrentUserFlagTemplates.success.body();

            try {
                var flag = require("aui/flag");
                flag({
                    type: "info",
                    title: title,
                    body: body,
                    close: "auto"
                });

            } catch(err) {

                AJS.messages.info(".sd-alerts-container", {
                    title: title,
                    body: body,
                    fadeout: true
                });
            }
        },

        _showDefaultToCurrentUserIfEmptyError: function() {
            var title = this.defaultToCurrentUserFlagTemplates.error.title();
            var body = this.defaultToCurrentUserFlagTemplates.error.body();

            try {
                var flag = require("aui/flag");
                flag({
                    type: "error",
                    title: title,
                    body: body,
                    close: "auto"
                });

            } catch(err) {

                AJS.messages.error(".sd-alerts-container", {
                    title: title,
                    body: body,
                    fadeout: true
                });
            }
        },

        _saveRuleSet: function() {
            var $deferred = $.Deferred();

            // ... then save the rule set
            this.model.getRuleSet().save()
                .done(_.bind(function() {
                    $deferred.resolve();

                    // Update the original model so that the dirty check will not detect any changes
                    this.ignoreDirtySaveCheck = true;

                    // If save is successful we redirect the user back to the list screen
                    this._goBackToPreviousPage();

                }, this))
                .fail(_.bind(function(xhr) {
                    $deferred.reject();

                    // If save fails we render the validation errors
                    var data = JSON.parse(xhr.responseText);
                    this.model.getRuleSet().set(data);
                    this.render();
                }, this));

            return $deferred;
        },

        beforeNavigate: function() {
            return this._dirtySaveCheck();
        },

        _dirtySaveCheck: function() {
            var $deferred = $.Deferred();

            if (this.ignoreDirtySaveCheck) {
                return $deferred.resolve();
            }

            if (this.hasChanged()) {
                var self = this;
                var dialog = new JIRA.FormDialog({
                    id: "automation-pre-save-dialog",
                    width:700,
                    content: _.bind(function (ready) {
                        ready(this.preSaveDialog());
                    }, this),
                    onContentRefresh: function () {
                        var popup = this.get$popup();
                        var $progress = popup.find(".js-status");
                        var $form = popup.find("form");
                        var $buttons = popup.find(".buttons button");

                        popup.find(".js-confirm-action-button").click(function (e) {
                            $form.addClass("sd-submitting");
                            $progress.addClass("loading");
                            self._disableElement($buttons);

                            self.model.getRuleSet().save()
                                .fail(function (xhr) {
                                    var data = JSON.parse(xhr.responseText);
                                    self.model.getRuleSet().set(data);
                                    self.render();

                                    tracker.trackEvent("automation.ruleset.edit.save.failed.validation", {
                                        ruleSetId: self.model.attributes.id
                                    });

                                    dialog.hide();
                                    $deferred.reject();
                                })
                                .done(function() {
                                    $progress.removeClass("loading").addClass("success");
                                    _.delay(function() {
                                        $deferred.resolve();
                                        dialog.hide();
                                    }, 800);

                                    tracker.trackEvent("automation.ruleset.edit.save.succeeded", {
                                        ruleSetId: self.model.attributes.id
                                    });
                                })
                                .always(function() {
                                    $progress.removeClass("loading");
                                });

                            tracker.trackEvent("automation.ruleset.edit.dirtypage.save");

                            e.preventDefault();
                        });

                        popup.find(".js-discard-action-button").click(function(e) {
                            tracker.trackEvent("automation.ruleset.edit.dirtypage.discard", {
                                ruleSetId: self.model.attributes.id
                            });

                            dialog.hide();
                            $deferred.resolve();
                            e.preventDefault();
                        });

                        popup.find(".cancel").click(function(e) {
                            tracker.trackEvent("automation.ruleset.edit.dirtypage.cancel");
                        });
                    }
                });

                tracker.trackEvent("automation.ruleset.edit.dirtypage");

                dialog.show();
            } else {
                $deferred.resolve();
            }

            return $deferred;
        },

        // Event handlers
        _componentSelected: function (data) {
            this._unselectAll();
            this.selectedView = data.view;
            this.selectedView.select();

            this.editPanel && this.editPanel.remove();

            var rules = this.model.getRuleSet().getRules();
            var parentRule = rules.find(function(rule) {
                var type = data.model.getType();
                var isMatch = false;

                switch (type) {
                    case "whenHandler":
                        isMatch = _.isEqual(rule.getWhen(), data.model);
                        break;
                    case "ifCondition":
                        isMatch = rule.getIfThens().some(function(ifThen) {
                            return ifThen.getIf() === data.model;
                        });
                        break;
                    case "thenAction":
                        isMatch = rule.getIfThens().some(function(ifThen) {
                            return ifThen.getThen() === data.model;
                        });
                }

                return isMatch;
            });

            if (parentRule) {
                this.editPanel = new RuleComponentDetailsView({
                    model: data.model,
                    ruleModel: parentRule,
                    isExistingRule: _.isNumber(this.model.getId())
                }).render();

                tracker.trackEvent("automation.ruleset.edit." + data.model.getType() + ".select");

                this.listenTo(this.editPanel, "closed", _.bind(this._editPanelClosed, this));
                this.listenTo(this.editPanel, "webFormClosed", _.bind(this.refreshSaveButtonState, this));
                this.listenTo(this.editPanel, "webFormOpened", _.bind(this.disableSaveButton, this));
                this.$(".component-details-panel-container").append(this.editPanel.el);

                Trace("automation.detail.panel.visible");

                PositionUtils.fixPosition(this.editPanel.$el.find('.component-details'));

                this.editPanel.focus();
            }
        },

        _componentDeleted: function(data) {
            this._unselectAll();

            if (this.editPanel) {
                this.editPanel && this.editPanel.remove();
                this.editPanel = null;
            }
        },

        _componentUnselected: function (e) {
            // If we have a edit panel currently in edit mode, we ignore unselect requests
            if (this.editPanel && this.editPanel.isInEditMode()) {
                return;
            }

            var target = $(e.target);
            if (target.closest(".component-summary").length == 0
                && target.closest(".component-details-panel").length == 0) {

                // This ensures that the target is not detached from the DOM.
                // If it is we assume that some other part of the code has handled the click event
                // and we will not do anything here.
                // We need this here because there may be code in the web form that switch to another
                // state and by the time we get this event the target of the click event has become detached.
                if ($.contains(document.documentElement, e.target)) {
                    this._unselectAll();
                    this.editPanel && this.editPanel.remove();
                    this.editPanel = null;
                }
            }
        },

        _unselectAll: function () {
            _.each(this.ruleViews, function (view) {
                view.unselectAll();
            });
        },

        _editPanelClosed: function () {
            this.refreshSaveButtonState();
            if (this.selectedView)
            {
                tracker.trackEvent("automation.ruleset.edit." + this.selectedView.model.getType() + ".close");

                // Focus back on the component summary
                this.selectedView.unselect();
                this.selectedView.focus();
                this.selectedView = null;
            }
        },

        _initSettingsDialog: function() {
            // cleanup first
            $("[id$=rule-set-settings-dialog]").remove();
            var rulesetModel = this.model.getRuleSet();
            this.settingsInlineDialog = AJS.InlineDialog(this.$('.rule-set-settings-button'), 'rule-set-settings-dialog', function(content, trigger, showPopup) {

                var dialog = new RuleSettingsDialog({
                    model: rulesetModel
                });
                content.empty().append(dialog.render().$el);

                // prevent the dialog from closing when user click on the dialog's content
                content.click(function (e) {
                    e.stopPropagation();
                });

                // If we are in an invalid state - try and fix it up
                // so the next time the dialog is opened it will be correct.
                if (!rulesetModel.isUserContextCompatibleWithWhenHandlers()) {
                    rulesetModel.getMetadata().getRunAsUserContext().unset("type");
                }

                showPopup();
                return false;
            }, {
                width: 385
            });
        },

        _showErrors: function() {
            var $headerContainer = this.$('.automation-settings-header-container .error-container');
            var $descriptionContainer = this.$('.automation-settings-description-container');

            $headerContainer.find('.automation.error').remove();
            $descriptionContainer.find('.automation.error').remove();

            var validation = this.model.getRuleSet().getValidation();

            if (validation) {
                var fieldErrors = validation.getFieldErrors();
                var globalErrors = validation.getGlobalErrors() || [];

                // Only the last header error is displayed, at the moment the
                // field errors have a higher priority than global errors.
                var headerErrors = globalErrors.concat(fieldErrors.name || []);

                // As these errors are coming from us, we're assuming that the errors are properly escaped if needed.
                // TODO: clean up once 7.0 only

                var htmlToSanitize = _.last(headerErrors);
                var lastHeaderError = HtmlUtils.sanitizeMarkup(htmlToSanitize);

                if (headerErrors.length > 0) {
                    $headerContainer.append(this.headerErrorTemplate({
                        error: lastHeaderError
                    }));
                }

                if (fieldErrors.description) {
                    $descriptionContainer.append(this.bodyErrorTemplate({
                        errors: fieldErrors.description
                    }));
                }
            }
        },

        // Updates the rule settings based on the state of the entire ruleset
        refreshRuleSettingsState: function() {
            // If we are in an invalid state - tell the user
            if (!this.model.getRuleSet().isUserContextCompatibleWithWhenHandlers())
            {
                // Pop the dialog
                this.settingsInlineDialog.show();
            }
        },

        // Refreshes the save button state and
        // updates the tooltip message if the save button becomes disabled
        refreshSaveButtonState: function() {
            var $saveButton = this.$(".ruleset-save-button");
            if (this.hasChanged()) {
                this._enableElement($saveButton);
            } else {
                this._disableElement($saveButton);
            }
        },

        disableSaveButton: function() {
            var $saveButton = this.$(".ruleset-save-button");
            this._disableElement($saveButton);
        },

        _disableElement: function(el) {
            $(el).attr({
                "aria-disabled": "aria-disabled",
                "disabled": "disabled"
            });
        },

        _enableElement: function(el) {
            $(el).removeAttr("aria-disabled disabled");
        },

        _onEnableRuleSetChange: function (el) {
            var isEnabled = el.target.checked;
            if (isEnabled) {
                tracker.trackEvent("automation.ruleset.edit.rule.enabled", {
                    ruleSetId: this.model.attributes.id
                });
            } else {
                tracker.trackEvent("automation.ruleset.edit.rule.disabled", {
                    ruleSetId: this.model.attributes.id
                });
            }
            // call this to trigger a modification
            this._enableAllRules(isEnabled);
            this.refreshSaveButtonState();
        },

        _enableAllRules: function (isEnabled) {
            this.model.getRuleSet().getRules().each(function (rule) {
                rule.setEnabled(isEnabled);
            });
        },

        _goBackToPreviousPage: function () {
            var originParam = UrlUtils.getQueryParamValue("origin");
            if (originParam) {
                Navigator.navigateToExternalFragment(originParam);
            } else {
                Navigator.navigateToFragment("");
            }
        }
    });
});
