define("servicedesk/internal/agent/settings/automation/page/automation-list-page-view", [
    "jquery",
    "automation/underscore",
    "automation/backbone-brace",
    "servicedesk/internal/agent/settings/automation/page/automation-list-page-list-entry-view",
    "servicedesk/internal/agent/settings/automation/util/html-utils/html-utils",
    "servicedesk/internal/agent/settings/automation/blueprint/select-blueprint-dialog",
    "servicedesk/internal/agent/settings/automation/util/analytics/analytics",
    "servicedesk/internal/agent/settings/automation/util/trace/trace"
], function (
    $,
    _,
    Brace,
    AutomationListPageListEntry,
    HtmlUtils,
    NewBluePrintDialog,
    tracker,
    Trace
    ) {
    return Brace.View.extend({
        template: ServiceDesk.Templates.Agent.Settings.automationSettings,

        events: {
            'click .create-rule-button':                    'createRuleSet',
            'click .sd-run-as-user-for-project-button':     'showRunAsSpecificUserPickerDialog',
            'change #rule-set-sort':                       '_ruleSetSortOptionChanged'
        },

        namedEvents: ["saveRunAsUserClicked"],

        initialize: function() {
            this.listenTo(this.model, "modelChanged", _.bind(this.render, this));
        },

        render: function () {
            var blueprints = this.model.getBlueprints() ? this.model.getBlueprints().toJSON() : [];
            var hasRuleSets = this.model.hasRuleSets();

            this.$el.html(this.template({
                hasRuleSets: hasRuleSets,
                blueprints: blueprints,
                runAsUser: this._computeRunAsUser()
            }));

            this.ruleSetViews = [];

            if (hasRuleSets) {
                this.model.getSortedRuleSets().each(_.bind(function (ruleSet) {
                    var ruleSetView = new AutomationListPageListEntry({
                        model: ruleSet,
                        projectRunAsUserSettings: this.model.getProjectRunAsUserSettings()
                    }).render();

                    this.ruleSetViews.push(ruleSetView);

                    this.$('.rule-set-list').append(ruleSetView.el);
                }, this));
            }

            var $ruleSortSelect = this.$('#rule-set-sort');

            this.sortSelector = AJS.$($ruleSortSelect).auiSelect2({
                minimumResultsForSearch: -1
            });

            if (this.model.getSort()) {
                this.sortSelector.select2('val', this.model.getSort());
            }

            // if the "Run as" user setting is configured but not valid we auto-show the setting dialog
            var projectRunAsUserSettings = this.model.getProjectRunAsUserSettings();
            if (projectRunAsUserSettings.getIsSet() && !projectRunAsUserSettings.getConfig().getUser().getIsValid())
            {
                this.showRunAsSpecificUserPickerDialog();
            }

            var ruleSetIdList = _.map(this.model.getRuleSets().models, function(ruleSetModel) {
                return ruleSetModel.getId();
            });

            Trace("automation.list.page.render.complete", {key: "ruleSetIdList", value: ruleSetIdList});

            return this;
        },

        _computeRunAsUser: function() {
            var configContainer = this.model.getProjectRunAsUserSettings();
            if (!configContainer.getIsSet()) {
                return null;
            }

            var config = configContainer.getConfig();
            if (config && config.getUser()) {

                var user = config.getUser();
                if (!user.getIsValid()) {
                    return null;
                } else {
                    return user.getUserDisplayName();
                }
            }
        },

        createRuleSet: function(e){
            tracker.trackEvent("automation.list.ruleset.create");

            NewBluePrintDialog.create({
                model: this.model
            });
            e.preventDefault();
        },

        _ruleSetSortOptionChanged: function(e) {
            var target = $(e.target);
            var selected = target.find(':selected');
            this.model.setSort(selected.val());

            tracker.trackEvent("automation.list.sortby." + selected.val());

            this.render();
        },

        showRunAsSpecificUserPickerDialog: function() {
            var self = this;
            this.runAsUserDialog = new JIRA.FormDialog({
                width: 400,
                id: "sd-run-as-setting-dialog",
                content: function (ready) {

                    // set the dialog content
                    ready(ServiceDesk.Templates.Agent.Settings.runAsSpecificUserDialogContent({
                        runAsUser: self._computeRunAsUser()
                    }));
                },
                onContentRefresh: function () {
                    var createdDialog = this;
                    var projectRunAsUserSettings = self.model.getProjectRunAsUserSettings();

                    // init the user search select input
                    self._initUserSearchField(createdDialog, projectRunAsUserSettings);

                    // init "Save" button submit
                    self._initSaveButton(createdDialog);
                },
                submitHandler: function(e, hideFooterLoadingIndicatorCallBack) {
                    e.preventDefault();
                    var $form = $(e.target);

                    // if the field if the field is empty, don't do anything but hiding the JIRA.FormDialog added loading spinner
                    if ($form.find("#sd-run-as-setting-user-select :selected").length == 0) {
                        hideFooterLoadingIndicatorCallBack();
                        return;
                    }

                    // disable the "Save" button
                    $form.find(".sd-run-as-setting-save-button").attr({
                        "aria-disabled": "aria-disabled",
                        "disabled": "disabled"
                    });

                    // serialise form data and trigger the saving of the selected user
                    var data = self._prepareRequest($form);
                    var projectKey = self.model.getProjectKey();
                    self.model.getProjectRunAsUserSettings().save(projectKey, data)
                        .done(_.bind(function (newData) {

                            // if the setting was invalid, then the precondition error panel is shown. We want to reload the page in that case.
                            var projectRunAsUserWasInvalid = self._wasProjectRunAsUserInvalid();
                            if (projectRunAsUserWasInvalid) {
                                window.location.reload();
                            }

                            tracker.trackEvent("automation.runrulesas.save.succeeded");

                            self.model.getProjectRunAsUserSettings().set(newData);
                            self._saveRunAsUserSuccess();
                        }, this))
                        .fail(_.bind(function(xhr) {
                            var response = JSON.parse(xhr.responseText);
                            self._saveRunAsUserError(response.errors);

                            tracker.trackEvent("automation.runrulesas.save.failed");
                        }, this));
                }
            });

            $(this.runAsUserDialog).one("Dialog.hide", function(e, el, reason) {
                if (reason == JIRA.Dialog.HIDE_REASON.cancel || reason == JIRA.Dialog.HIDE_REASON.escape) {
                    tracker.trackEvent("automation.runrulesas.cancel");
                }
            });

            tracker.trackEvent("automation.list.runrulesas.edit");

            this.runAsUserDialog.show();
        },

        _wasProjectRunAsUserInvalid: function() {
            var projectRunAsUserSettings = this.model.getProjectRunAsUserSettings();
            return projectRunAsUserSettings.getIsSet() && !projectRunAsUserSettings.getConfig().getUser().getIsValid();
        },

        _prepareRequest: function($form) {
            return {
                type: "specificUser",
                user: $form.serializeObject()
            }
        },

        _initSaveButton: function(dialog) {
            dialog.get$popupContent().find(".sd-run-as-setting-save-button").click(_.bind(function () {
                dialog.get$popupContent().find("form").submit();
            }, dialog));
        },

        _initUserSearchField: function(dialog, projectRunAsUserSettings) {
            var $element = dialog.get$popupContent().find("#sd-run-as-setting-user-select");
            var singleSelectOptions = {
                element: $element,
                itemAttrDisplayed: "label", // show full text, not value, in lozenges. (e.g "Oyster Bay" instead of "5")
                errorMessage: AJS.I18n.getText("sd.automation.run.as.user.for.project.dialog.error.user.invalid"),
                revertOnInvalid: false,
                ajaxOptions: {
                    url: AJS.contextPath() + "/rest/servicedesk/1/servicedesk/sd-user-search/" + this.model.getProjectKey() + "/usersWithBrowseAndEditPermissions",
                    query: true, // keep going back to the sever for each keystroke,
                    data: function (query) { return {query: query}; }, // define the GET parameter name sent when querying the users resource
                    formatResponse: function (response) {
                        var availableUsers = [];

                        _.each(response, function (user) {
                            availableUsers.push(new AJS.ItemDescriptor({
                                value: user.key, // value of item added to select
                                label: user.displayName, // title of lozenge
                                html: HtmlUtils.escapeMarkup(user.displayName) // html used in suggestion
                            }));
                        });
                        return availableUsers;
                    }
                }
            };

            // init the AJS.SingleSelect field
            var runAsUserSelect = new AJS.SingleSelect(singleSelectOptions);

            // fill it with the existing value if any and valid
            if (this._isUserContextConfigSetAndValid(projectRunAsUserSettings)) {

                var userConfig = projectRunAsUserSettings.getConfig().getUser();
                this._initSelectedValue(runAsUserSelect, userConfig.getUserKey(), userConfig.getUserDisplayName());

            } else if (this._isUserContextConfigNotSet(projectRunAsUserSettings)) {

                var currentUserKey = AJS.Meta.get("remote-user");
                var currentUserDisplayName = AJS.Meta.get("remote-user-fullname");
                this._initSelectedValue(runAsUserSelect, currentUserKey, currentUserDisplayName);

            } else if (this._isUserContextConfigSetAndInvalid(projectRunAsUserSettings)) {

                var userConfig = projectRunAsUserSettings.getConfig().getUser();
                this._initSelectedValue(runAsUserSelect, userConfig.getUserKey(), userConfig.getUserDisplayName() ? userConfig.getUserDisplayName() : "");
                runAsUserSelect.showErrorMessage();
            }

            $element.focus(_.bind(function() {
                runAsUserSelect._handleCharacterInput(true);
            }, this));
        },

        _initSelectedValue: function(runAsUserSelect, userKey, userDisplayName) {
            runAsUserSelect.setSelection(new AJS.ItemDescriptor({
                value: userKey, // value of item added to select
                label: userDisplayName, // title of lozenge
                html: HtmlUtils.escapeMarkup(userDisplayName) // html used in suggestion
            }));
        },

        _isUserContextConfigSetAndValid: function(configContainer) {
            var config = configContainer.getConfig();

            return configContainer
                && configContainer.getIsSet()
                && config.getUser()
                && config.getUser().getIsValid();
        },

        _isUserContextConfigNotSet: function(configContainer) {
            return configContainer
                && !configContainer.getIsSet();
        },

        _isUserContextConfigSetAndInvalid: function(configContainer) {
            var config = configContainer.getConfig();

            return configContainer
                && configContainer.getIsSet()
                && config.getUser()
                && !config.getUser().getIsValid();
        },

        _saveRunAsUserSuccess: function() {
            _.delay(_.bind(function() {
                this.runAsUserDialog.hide();
                this.render();
            }, this), 800);
        },

        _saveRunAsUserError: function(errors) {
            var $popupContent = this.runAsUserDialog.get$popupContent();

            // re-enable the "Save" button
            var saveButton = $popupContent.find(".sd-run-as-setting-save-button");
            saveButton.removeAttr('aria-disabled');
            saveButton.removeAttr('disabled');

            // remove the JIRA.FormDialog spinner
            $popupContent.find(".icon.throbber.loading").remove();

            // apply error on the user search field
            $popupContent.find(".error").remove();
            var userSearchInput = $popupContent.find('#sd-run-as-setting-user-select');
            var errorBox = $("<div />").addClass("error").text(errors[0].errorMessage);
            errorBox.insertAfter(userSearchInput);
        }
    });
});
