define("servicedesk/internal/agent/settings/automation/model/rule-set-model", [
    "automation/backbone-brace",
    "servicedesk/internal/agent/settings/automation/util/ajax/ajax",
    "servicedesk/internal/agent/settings/automation/model/link-model",
    "servicedesk/internal/agent/settings/automation/model/rule-collection",
    "servicedesk/internal/agent/settings/automation/model/validation-model",
    "servicedesk/internal/agent/settings/automation/model/metadata/metadata-model"
], function (
    Brace,
    Ajax,
    LinkModel,
    RuleCollection,
    ValidationModel,
    MetadataModel
    ) {

    return Brace.Model.extend({
        namedEvents: [
            "modelChanged",
            "savedOnServer",
            "validatedOnServer"
        ],
        namedAttributes: {
            id: Number,
            name: String,
            description: String,
            link: LinkModel,
            rules: RuleCollection,
            metadata: MetadataModel,
            validation: ValidationModel,
            activeRevisionId: Number
        },

        defaults: {
            metadata: new MetadataModel()
        },

        getSaveUrl: function () {
            return AJS.contextPath() + "/rest/servicedesk/automation/1/ruleset";
        },

        getUpdateUrl: function () {
            return AJS.contextPath() + "/rest/servicedesk/automation/1/ruleset/" + this.getId() + "/expectedRevisionId/" + this.getActiveRevisionId();
        },

        getValidateUrl: function () {
            if (this.isNew()) {
                return AJS.contextPath() + "/rest/servicedesk/automation/1/ruleset/validate";
            } else {
                return AJS.contextPath() + "/rest/servicedesk/automation/1/ruleset/" + this.getId() + "/expectedRevisionId/" + this.getActiveRevisionId() + "/validate";
            }
        },
        isNew: function() {
            return !this.getId();
        },

        getDeleteUrl: function() {
            return AJS.contextPath() + "/rest/servicedesk/automation/1/ruleset/" + this.getId();
        },

        del: function() {
            return Ajax.del(this.getDeleteUrl());
        },

        initialize: function() {
            this._bindEvents();
        },

        serialize: function() {
            var metadata = this.getMetadata() && this.getMetadata().serialize();
            return {
                name: this.getName(),
                description: this.getDescription(),
                metadata: metadata,
                rules: this.getRules().map(function(rule) {
                    return rule.serialize();
                })
            }
        },

        save: function() {
            var serializedData = this.serialize();
            var ajax = !this.isNew() ?
                Ajax.put(this.getUpdateUrl(), serializedData) :
                Ajax.post(this.getSaveUrl(), serializedData);

            return ajax.done(_.bind(function (data) {
                    this.triggerSavedOnServer(data);
                }, this));
        },

        validate: function() {
            var serializedData = this.serialize();
            return Ajax
                .post(this.getValidateUrl(), serializedData)
                .done(_.bind(function (data) {
                    this.triggerValidatedOnServer(data);
                }, this));
        },

        validateStructure: function() {
            // This logic here must be the same as the structure check done on the server (see RuleSetValidator#validateSelf)
            var validation = this.getValidation() || new ValidationModel();

            // Field errors
            var missingNameError = AJS.I18n.getText("sd.automation.rule.set.error.rule.set.name.missing");
            if (!this.getName() || this.getName().trim() === "") {
                validation.addFieldError("name", missingNameError);
            } else {
                validation.removeFieldError("name", missingNameError);
            }

            // rule errors
            var missingRuleError = AJS.I18n.getText("sd.automation.rule.set.error.rule.set.must.contain.a.rule");
            if (this.getRules().isEmpty()) {
                validation.addGlobalError(missingRuleError);
            } else {
                validation.removeGlobalError(missingRuleError);
            }

            // user context errors
            var userContextError = AJS.I18n.getText("sd.automation.rule.set.error.rule.set.user.context.invalid");
            if(!this._isUserContextValid()) {
                validation.addGlobalError(userContextError);
            } else {
                validation.removeGlobalError(userContextError);
            }

            this.getRules().each(function(rule) {
                rule.validateStructure();
            })
        },

        isValid: function() {
            var isThisValid = !this.getValidation() || !this.getValidation().hasError();

            return isThisValid && this.getRules().every(function (rule) {
                return rule.isValid();
            });
        },

        isUserContextCompatibleWithWhenHandlers: function() {
            // Assume only one rule in a rule set.
            var whenHandlers = this.getRules().at(0).getWhen().getWhenHandlers();

            // If we have no when handlers, then we give them the benefit of the doubt and say it is valid
            if (whenHandlers.isEmpty()) {
                return true;
            }
            // Otherwise see if any of the WHENs are not providing 'user'
            var invalidRule = whenHandlers.find(function(whenHandler) {
                var providers = whenHandler.getProvides();
                return !_.contains(providers, "user");
            });

            // If there are no invalid rules, then we are compatible.
            return !invalidRule;
        },

        _isUserContextValid: function() {
            // if we have a run as user context
            if (this.getMetadata().getRunAsUserContext().getType()) {
                var invalidRule = this.getRules().find(function(rule) {
                    var invalidWhenHandler = rule.getWhen().getWhenHandlers().find(function(whenHandler) {
                        return !_.contains(whenHandler.getProvides(), "user");
                    });
                    return !!invalidWhenHandler;
                });
                return !invalidRule;
            }
            return true;
        },

        _checkStructureAndFireChangeEvent: function() {
            this.validateStructure();
            this.triggerModelChanged();
        },

        _bindEvents: function() {
            this.stopListening();

            this.listenTo(this, "change", _.bind(this._bindEvents, this));

            this.listenTo(this.getRules(), "add remove", _.bind(this._checkStructureAndFireChangeEvent, this));
            this.listenTo(this, "change:name change:description", _.bind(this._checkStructureAndFireChangeEvent, this));

            this.listenTo(this.getRules(), "modelChanged", this.triggerModelChanged);
            this.listenTo(this.getMetadata(), "modelChanged", _.bind(this._checkStructureAndFireChangeEvent, this));
        }
});

});