
define("JEP/shared/components/Pagination", [
    "jquery",
    "underscore",
    "backbone"
], function (
    $,
    _,
    Backbone
) {

    return Backbone.View.extend({

        className: "jep-pagination-div",

        template: JEP.Templates.Components.Pagination.content,

        events: {
            renderCompleted: "renderCompleted",

            "click li": "_onSelectPage"
        },

        initialize: function (options) {
            options = options || {};
            this.currentPage = options.currentPage || 1;
            this.totalPages = options.totalPages || 1;
            this.threshold = options.threshold || 2;     // number of pages to render before and after the current page

            this._render(this.currentPage, this.totalPages);
        },

        gotoPage: function (page) {
            if (page > 0 && page <= this.totalPages) {
                this.currentPage = page;
                this._render(this.currentPage, this.totalPages);

                // fire renderCompleted event separately with the render method
                this.trigger(this.events.renderCompleted, this.currentPage, this);
            }
        },

        prev: function () {
            if (this.currentPage > 1) {
                this.currentPage--;
                this.gotoPage(this.currentPage);
            }
        },

        next: function () {
            if (this.currentPage < this.totalPages) {
                this.currentPage++;
                this.gotoPage(this.currentPage);
            }
        },

        /**
         * Show spinner on the selected element. This is desirable when consumer is waiting for a response from rest call
         */
        showSpinner: function () {
            var $select = this.$("li.aui-nav-selected");
            $select.children(0).css("visibility", "hidden");
            $select.spin();

            // disable links
            this.$("li a").parent().each(function () {
                var $e = $(this);
                var content = $e.text();
                $e.empty().append("<span class='nav-a'>" + content + "</span>");
            });

        },

        stopSpinner: function () {
            var $select = this.$("li.aui-nav-selected");
            $select.spinStop();
            $select.children(0).css("visibility", "");

            // enabled links
            this.$("li span.nav-a").parent().each(function () {
                var $e = $(this);
                var content = $e.text();
                $e.empty().append("<a href='#'>" + content + "</a>");
            });
        },

        _render: function (currentPage, totalPages) {
            var startPage = 1;
            var endPage = totalPages;
            if (currentPage > this.threshold && totalPages > 2 * this.threshold + 3) {
                startPage = Math.min(currentPage - this.threshold, totalPages - 2 * this.threshold);
            }
            if (startPage <= 2) {
                endPage = Math.min(totalPages, 2 * this.threshold + 3);
            } else if (currentPage < totalPages - this.threshold) {
                endPage = Math.max(currentPage + this.threshold, startPage + 2 * this.threshold);
            }

            this.$el.html(this.template({
                currentPage: currentPage,
                totalPages: totalPages,
                startPage: startPage,
                endPage: endPage
            }));
        },

        _onSelectPage: function (e) {
            e.preventDefault();

            var $e = $(e.target);
            var $parent = $e.parent();

            // check if this li element has an enabled anchor child
            if ($e.is("a") && !$parent.attr("aria-disabled")) {
                if ($parent.hasClass("aui-nav-previous")) {
                    this.prev();
                } else if ($parent.hasClass("aui-nav-next")) {
                    this.next();
                } else {
                    this.gotoPage(parseInt($e.text()));
                }
            }
        }
    });
});