/* globals define, AJS */
define("JEP/globalmail/GlobalMailSettings", [
    "underscore",
    "backbone",
    "JEP/shared/Tracker",
    "JEP/globalmail/GlobalMailSettingsModel",
    "JEP/shared/components/ConfirmationDialog",
    "JEP/Jira/trace"
], function (
    _,
    Backbone,
    Tracker,
    GlobalMailModel,
    ConfirmationDialog,
    trace
) {

    /**
     * The outgoing mail enable/disable toggle.
     */
    return Backbone.View.extend({

        template: {
            content: JEP.Templates.Admin.GlobalMailSettings.content,
            puller: JEP.Templates.Admin.GlobalMailSettings.puller,
            processor: JEP.Templates.Admin.GlobalMailSettings.processor
        },

        ui: {
            pullerBtn: ".jep-puller-button",
            pullDisabledLozenge: ".jep-disabled-pull-lozenge",
            processorBtn: ".jep-processor-button",
            pullerError: ".jep-puller-content .jep-error",
            processorError: ".jep-processor-content .jep-error",
            pullerSpinner: ".jep-puller-button .spinner",
            processorSpinner: ".jep-processor-button .spinner",
            cleanerAgeDropdown: ".jep-cleaner-age-dropdown-value",
            cleanerAgeError: ".jep-cleaner-content .jep-error"
        },

        events: {
            "click .jep-puller-button": "_pullerButtonClicked",
            "click .jep-processor-button": "_processorButtonClicked",
            "keypress .jep-on-off-btn:focus": "_enter"
        },

        initialize: function() {
            this.model = new GlobalMailModel({
                isPullerOn: this.$(this.ui.pullerBtn).hasClass("on"),
                isProcessorOn: this.$(this.ui.processorBtn).hasClass("on")
            });
            // bind the dropdown items to function
            AJS.$("body").on("click", ".jep-cleaner-age-dropdown-item", _.bind(this._processCleanerAgeClicked, this));
        },

        _enter: function (e) {
            if(e.keyCode == 13){
                e.target.click();
            }
        },

        /**
         *  Event handler when Turn on/off button of Global Mail Puller is clicked
         */
        _pullerButtonClicked: function() {
            if (this.model.get("isPullerOn")) {
                Tracker.trackPageview("/admin/globalmailsettings/puller/off");

                var dialog = ConfirmationDialog.create({
                    title: AJS.I18n.getText("jep.global.mail.puller.confirm.dialog.title"),
                    explanation: AJS.I18n.getText("jep.global.mail.puller.confirm.dialog.message"),
                    button: AJS.I18n.getText("jep.common.word.turn.it.off"),
                    action: _.bind(function () {
                        return this.model.turnPullerOff();
                    }, this)
                }).show();

                this.listenTo(dialog, "confirmedSuccess", _.bind(this._renderPuller, this));

            } else {
                Tracker.trackPageview("/admin/globalmailsettings/puller/on");

                this.$(this.ui.pullerBtn+" *").attr("disabled", "disabled");
                this.$(this.ui.pullerSpinner).spin();

                this.model.turnPullerOn()
                    .done(_.bind(this._renderPuller, this))
                    .fail(_.bind(this._showPullerError, this));
            }
        },

        /**
         *  Event handler when Turn on/off button of Global Mail Processor is clicked
         */
        _processorButtonClicked: function() {
            if (this.model.get("isProcessorOn")) {
                Tracker.trackPageview("/admin/globalmailsettings/processor/off");

                var dialog = ConfirmationDialog.create({
                    title: AJS.I18n.getText("jep.global.mail.processor.confirm.dialog.title"),
                    explanation: AJS.I18n.getText("jep.global.mail.processor.confirm.dialog.message"),
                    button: AJS.I18n.getText("jep.common.word.turn.it.off"),
                    action: _.bind(function () {
                        return this.model.turnProcessorOff();
                    }, this)
                }).show();

                this.listenTo(dialog, "confirmedSuccess", _.bind(this._renderProcessor, this));
            } else {
                Tracker.trackPageview("/admin/globalmailsettings/processor/on");

                this.$(this.ui.processorBtn+" *").attr("disabled", "disabled");
                this.$(this.ui.processorSpinner).spin();

                this.model.turnProcessorOn()
                    .done(_.bind(this._renderProcessor, this))
                    .fail(_.bind(this._showProcessorError, this));
            }
        },

        _processCleanerAgeClicked: function (e) {
            e.preventDefault();
            this.$(this.ui.cleanerAgeError).text("");
            this.model.setCleanerThreshold(AJS.$(e.target).data("value"))
                .done(_.bind(this._renderThreshold, this))
                .fail(_.bind(this._showThresholdError, this));
        },

        _renderPuller: function (isOn) {
            this.$(".jep-puller-content").html(this.template.puller({
                isSystemFlagOn: this.$(this.ui.pullDisabledLozenge).length === 0,
                isOn: isOn
            }));
        },

        _renderProcessor: function (isOn) {
            this.$(".jep-processor-content").html(this.template.processor({
                isOn: isOn
            }));
        },

        _renderThreshold: function (age) {
            this.$(this.ui.cleanerAgeDropdown).text(age);
            trace("jepp.email.cleaner.threshold.updated");
        },

        _showPullerError: function (errorMessage) {
            this.$(this.ui.pullerBtn+" *").removeAttr("disabled");
            this.$(this.ui.pullerSpinner).spinStop();

            this.$(this.ui.pullerError).text(errorMessage);
        },

        _showProcessorError: function (errorMessage) {
            this.$(this.ui.processorBtn+" *").removeAttr("disabled");
            this.$(this.ui.processorSpinner).spinStop();

            this.$(this.ui.processorError).text(errorMessage);
        },

        _showThresholdError: function (errorMessage) {
            this.$(this.ui.cleanerAgeError).text(errorMessage);
            trace("jepp.email.cleaner.threshold.error");
        }

    });

});
