define("servicedesk/internal/agent/settings/automation/view/rule-component-details-panel-view", [
    "jquery",
    "automation/underscore",
    "automation/backbone-brace",
    "servicedesk/internal/agent/settings/automation/view/rule-component-details-panel-list-view",
    "servicedesk/internal/agent/settings/automation/view/rule-component-details-panel-form-view",
    "servicedesk/internal/agent/settings/automation/view/picker/rule-component-picker",
    "servicedesk/internal/agent/settings/automation/util/analytics/analytics"
], function (
    $,
    _,
    Brace,
    RuleComponentPanelListView,
    RuleComponentPanelFormView,
    Picker,
    tracker
    ) {

    return Brace.View.extend({
        className: "component-details-panel",
        template: ServiceDesk.Templates.Agent.Settings.componentDetailsPanel,
        events: {
            "click .close-panel-button": "_closeButtonClicked",
            "keydown": "_escapePressed"
        },
        namedEvents: [
            "closed",
            "webFormClosed",
            "webFormOpened"
        ],

        initialize: function(opt) {
            this.ruleModel = opt.ruleModel;
            this.isExistingRule = opt.isExistingRule;
        },

        render: function () {
            this._disposeOfChildren();

            this.$el.html(this.template({
                title: this._getTitle()
            }));

            this.listView = new RuleComponentPanelListView({
                model: this.model.getCollection()
            }).render();

            this.listenTo(this.listView, "editClicked", _.bind(this._onEditClicked, this));
            this.listenTo(this.listView, "componentRemoved", _.bind(this._onRemoveClicked, this));

            this.picker = new Picker({
                type: this.model.getType(),
                ruleModel: this.ruleModel
            }).render();

            this.listenTo(this.picker, "addModule", _.bind(this._onModuleAdded, this));

            // Append the new components to the DOM
            this.$('.component-details').append(this.listView.el);
            this.$('.component-details').append(this.picker.el);

            return this;
        },

        close: function() {
            this.triggerClosed(this._buildEventModel());
            this.dispose();
        },

        focus: function () {
            this.picker.focus();

            // Ask the list view for focus, if it doesn't have any list elements it will leave the focus on the picker
            this.listView.focus();
        },

        dispose: function() {
            this._disposeOfChildren();
            this.remove();
        },

        isInEditMode: function() {
            return !!this.webFormView;
        },

        /* Event Handlers */

        _onRemoveClicked: function() {
            // This is needed so that if the list view has nothing to focus we ensure that
            // the picker is focused instead.
            this.focus();
            this.picker.collapse();
        },

        _onEditClicked: function(data) {
            tracker.trackEvent("automation.ruleset.edit." + this.model.getType() + ".configuration.edit", {
                moduleKey: data.model.attributes.moduleKey
            });

            this._showWebFormView(data.model);
        },

        _onModuleAdded: function(data) {
            var model = data.model;

            if (!model.getWebFormModule()) {
                // We add the module automatically if there's no web form.
                this.picker.collapse();
                this.model.getCollection().add(model.toJSON());
                this.listView.render();
                this.listView.focus();
            } else {
                this._showWebFormView(model);
            }
        },

        _closeButtonClicked: function (e) {
            e.preventDefault();
            this.close();
        },

        _escapePressed: function (e) {
            if (e.keyCode == 27)
            {
                e.preventDefault();

                // Don't close if currently in edit mode. Instead we close the form view.
                if (this.isInEditMode()) {
                    this._closeWebFormView();
                } else {
                    this.close();
                }
            }
        },

        _showWebFormView: function(model) {
            if (this.picker) {
                this.picker.hide();
            }

            if (this.listView) {
                this.listView.hide();
            }

            this.webFormView = new RuleComponentPanelFormView({
                model: model,
                ruleModel: this.ruleModel,
                isExistingRule: this.isExistingRule
            }).render();

            this.listenTo(this.webFormView, "submitSuccess", _.bind(function(data) {
                var changedModel = data;
                if (!this.model.getCollection().contains(changedModel)) {
                    this.model.getCollection().add(changedModel.toJSON());
                    this.listView.render();
                }

                this._closeWebFormView();
            }, this));

            this.$('.component-details').append(this.webFormView.el);
            this.listenTo(this.webFormView, "cancel", _.bind(this._closeWebFormView, this));

            this.triggerWebFormOpened();
            this.webFormView.focus();

            return this.webFormView;
        },

        _closeWebFormView: function() {
            if (this.webFormView) {
                this.webFormView.stopListening();
                this.webFormView.undelegateEvents();
                this.webFormView.$el.remove();
                this.webFormView.dispose && this.webFormView.dispose();
                delete this.webFormView;
            }

            this.triggerWebFormClosed();

            if (this.picker) {
                this.picker.show();
            }

            if (this.listView) {
                this.listView.show();
            }

            this.focus();
        },

        _getTitle: function () {
            switch (this.model.getType()) {
                case "whenHandler":
                    return AJS.I18n.getText("sd.automation.component.edit.title.trigger");
                case "ifCondition":
                    return AJS.I18n.getText("sd.automation.component.edit.title.matcher.condition");
                case "thenAction":
                    return AJS.I18n.getText("sd.automation.component.edit.title.action");
            }
        },

        _disposeOfChildren: function() {
            if (this.listView) {
                this.listView.stopListening();
                this.listView.undelegateEvents();
                this.listView.dispose && this.listView.dispose();
            }

            if (this.picker) {
                this.picker.stopListening();
                this.picker.undelegateEvents();
                this.picker.dispose && this.picker.dispose();
            }

            if (this.webFormView) {
                this.webFormView.stopListening();
                this.webFormView.undelegateEvents();
                this.webFormView.dispose && this.webFormView.dispose();
            }
        },

        _buildEventModel: function() {
            return {
                view: this,
                model: this.model
            }
        }
    });

});