define("servicedesk/internal/agent/settings/automation/model/rule-model", [
    "automation/backbone-brace",
    "automation/underscore",
    "servicedesk/internal/agent/settings/automation/model/when-model",
    "servicedesk/internal/agent/settings/automation/model/if-then-collection",
    "servicedesk/internal/agent/settings/automation/model/validation-model"
], function (
    Brace,
    _,
    WhenModel,
    IfThenCollection,
    ValidationModel
    ) {

    return Brace.Model.extend({
        namedEvents: [
            "modelChanged"
        ],
        namedAttributes: {
            when: WhenModel,
            ifThens: IfThenCollection,
            validation: ValidationModel,
            enabled: Boolean
        },

        initialize: function() {
            this._bindEvents();
        },

        serialize: function() {
            var when = this.getWhen() || new WhenModel();
            var ifThens = this.getIfThens() || new IfThenCollection();

            return {
                when: when.serialize(),
                ifThens: ifThens.map(function(ifThen) {
                    return ifThen.serialize();
                }),
                enabled: this.getEnabled()
            }
        },

        getEffectiveProvides: function() {
            // get all when handlers in the rule
            var whens = this.getWhen().attributes.whenHandlers.models;
            var provides = [];

            // get provided contexts of each when handler
            _.each(whens, function(when){
                provides.push(when.attributes.provides)
            });

            // intersection of provided contexts
            return _.intersection.apply(_, provides);
        },

        validateStructure: function() {
            // This logic here must be the same as the structure check done on the server (see RuleValidator#validateData)
            var validation = this.getValidation() || new ValidationModel();

            var missingIfThenError = AJS.I18n.getText("sd.automation.rule.set.error.rule.must.contain.an.if.then");
            if (!this.getIfThens() || this.getIfThens().isEmpty()) {
                validation.addGlobalError(missingIfThenError);
            } else {
                validation.removeGlobalError(missingIfThenError);
            }

            this.getWhen().validateStructure();

            // There are additional structural checks here based on RuleValidator#validateIfThens
            var lastIndex = this.getIfThens().length - 1;
            this.getIfThens().each(function(ifThen, index) {
                ifThen.validateStructure(index === lastIndex);
            });
        },

        isValid: function() {
            var isThisValid = !this.getValidation() || !this.getValidation().hasError();

            return isThisValid &&
                this.getWhen().isValid() &&
                this.getIfThens().every(function (ifThen) {
                    return ifThen.isValid();
                });
        },

        _checkStructureAndFireChangeEvent: function() {
            this.validateStructure();
            this.triggerModelChanged();
        },

        // This is quite hacky but is required because the if model has no concept of where it belongs
        // in a collection, it must ask the rule for the index.
        _validateIfModel: function(model) {
            var ifThens = this.getIfThens();
            if (ifThens) {
                var modelIndex = ifThens.indexOf(model);

                if (modelIndex !== -1) {
                    var lastIndex = this.getIfThens().length - 1;
                    model.validateStructure(modelIndex === lastIndex)
                }
            }
        },

        _bindEvents: function() {
            this.stopListening();

            this.listenTo(this, "change", _.bind(this._bindEvents, this));

            this.listenTo(this.getIfThens(), "add remove", _.bind(this._checkStructureAndFireChangeEvent, this));
            this.listenTo(this.getIfThens(), "requestValidateStructure", _.bind(this._validateIfModel, this));
            this.listenTo(this.getIfThens(), "modelChanged", this.triggerModelChanged);

            this.listenTo(this.getWhen(), "modelChanged", this.triggerModelChanged);
        }
    });

});