/* global define */

/**
 * Change after creating mail log component
 *    .setChannelKey()   // set which mail channel should be displayed the log
 *    .setCurrentPage()
 *    .setPageLimit()
 *    .showWaitingPage() // show loading page while waiting for data from server
 *
 * Events:
 *    - renderCompleted ( isError : Boolean )
 *    // if isError return true, it means the component has just rendered an error.
 *
 */

define("JEP/MailLog",[
    "jquery",
    "underscore",
    "backbone",
    "JEP/shared/components/Pagination"
], function (
    $,
    _,
    Backbone,
    Pagination
) {

    return Backbone.View.extend({

        events: {
            renderCompleted: "renderCompleted"
        },

        initialize: function (options) {
            if (options) {
                this.channelKey = options.channelKey;
                this.pageLimit = options.pageLimit || 50;
                this.currentPage = options.currentPage || 1;
            }
        },

        setChannelKey: function (channelKey) {
            this.channelKey = channelKey;
            return this;
        },

        setPageLimit: function (pageLimit) {
            this.pageLimit = pageLimit;
            return this;
        },

        setCurrentPage: function (currentPage) {
            this.currentPage = currentPage;
            return this;
        },

        show: function () {
            // abstract function, put your implementation here to show content of the log
        },

        showWaitingPage: function () {
            this._resetElement(this.options.el);
            this.$el.html(JEP.Templates.Components.MailLog.waiting());
            return this;
        },

        _showPage: function (page, paginationView) {
            this.currentPage = page;
            paginationView.showSpinner();

            // update the other pagination and show spinner
            if (paginationView.cid === this.topPagination.cid) {
                this.bottomPagination.currentPage = paginationView.currentPage;
                this.bottomPagination._render(paginationView.currentPage, paginationView.totalPages);
                this.bottomPagination.showSpinner();
            } else {
                this.topPagination.currentPage = paginationView.currentPage;
                this.topPagination._render(paginationView.currentPage, paginationView.totalPages);
                this.topPagination.showSpinner();
            }

            this.show();
        },

        /**
         * in some cases, contain element is not available at initialize time
         * so we must attempt to find it at render time
         */
        _resetElement: function (el) {
            if (!this.el) {
                this.setElement(el);
            }
        },

        _renderPagination: function (currentPage, totalPages) {
            // only render pagination if total page > 1
            if (totalPages > 1) {
                this.topPagination = new Pagination({
                    currentPage: currentPage,
                    totalPages: totalPages
                });
                this.listenTo(this.topPagination, this.topPagination.events.renderCompleted, _.bind(this._showPage, this));

                this.bottomPagination = new Pagination({
                    currentPage: currentPage,
                    totalPages: totalPages
                });
                this.listenTo(this.bottomPagination, this.bottomPagination.events.renderCompleted, _.bind(this._showPage, this));

                this.$(".js-mail-log").prepend(this.topPagination.$el);
                this.$(".js-mail-log").append(this.bottomPagination.$el);
            }
        },

        _renderError: function (errorMessage) {
            // if init spinner exists, remove it first
            if (this.$(".jep-spinner").length > 0) {
                this.$el.empty();
            }

            // clear spinning state of pagination
            this.topPagination && this.topPagination.stopSpinner();
            this.bottomPagination && this.bottomPagination.stopSpinner();

            // clear and show error message
            this._clearError();
            this._showError(errorMessage);

            this.trigger(this.events.renderCompleted, true);
        },

        _showError: function (errorMessage) {
            var $error = $(JEP.Templates.Components.MailLog.error({"error": errorMessage}));
            this._clearError();
            this.$el.prepend($error);
        },

        _clearError: function () {
            this.$(".aui-message.error.shadowed.closeable").remove();
        }
    });
});