/* globals define, JIRA */

/**
 * Show a simple confirm dialog before processing an action.
 *
 * Usage:
 *
 *  var ConfirmationDialog = require('JEP/shared/components/ConfirmationDialog');
 *  var dialog = ConfirmationDialog.create({
 *      title: '',
 *      explanation: '',
 *      button: 'Confirm',      // (optional) text in confirm button
 *      action: function(){}    // (optional) confirm action
 *  });
 *  dialog.show();
 *
 * Events:
 *  confirmedSuccess - fire after handling action successfully
 *  confirmedFailure - fire after handling action failed
 *  confirmed - fire if there's no action defined and confirm button is clicked
 */

define("JEP/shared/components/ConfirmationDialog", [
    "jquery",
    "underscore",
    "backbone",
    "JEP/shared/ajax",
    "exports"
], function (
    $,
    _,
    Backbone,
    Ajax,
    Exports
) {

    var View = Backbone.View.extend({
        tagName: "section",
        className: "aui-layer aui-dialog2 aui-dialog2-medium confirm-dialog",
        attributes: {
            "role": "dialog",
            "data-aui-remove-on-hide": "true"
        },
        template: JEP.Templates.Components.ConfirmationDialog.dialog,

        events: {
            "click .js-confirm-button": "handleAction",
            "click .js-cancel-button": "onCancel",
            "keyup": "onKeyUp"
        },

        initialize: function (options) {
            this.title = options.title;
            this.explanation = options.explanation;
            this.button = options.button || AJS.I18n.getText("jep.common.word.confirm");
            this.action = options.action;
        },

        handleAction: function (e) {
            e.preventDefault();

            if (this.action && _.isFunction(this.action)) {
                this._startLoading();

                this.action()
                    .done(_.bind(function (response) {
                        this._showSuccess(_.bind(function () {
                            this.hide();
                            this.trigger("confirmedSuccess", response);
                        }, this));
                    }, this))
                    .fail(_.bind(function (xhr) {
                        this._stopLoading();
                        var errorMessage = Ajax.getErrorMessage(xhr);
                        this._showError(errorMessage);
                        this.trigger("confirmedFailure", xhr);
                    }, this));
            } else {
                this.trigger("confirmed");
            }
        },

        onCancel: function () {
            this.hide();
        },

        onKeyUp: function (e) {
            // if ESC was pressed
            if (e.keyCode == 27) {
                this.hide();
            }
        },

        show: function () {
            if (this.dialog) {
                this.dialog.show();
            }
            return this;
        },

        hide: function () {
            if (this.dialog) {
                this.dialog.hide();
            }
            return this;
        },

        _getFunctionOrValue: function (value) {
            return _.isFunction(value) ? value() : value;
        },

        render: function () {
            var title = this._getFunctionOrValue(this.title);
            var explanation = this._getFunctionOrValue(this.explanation);
            var buttonText = this._getFunctionOrValue(this.button);

            this.$el.html(this.template({
                title: title,
                explanation: explanation,
                button: buttonText
            }));

            return this.$el;
        },

        _startLoading: function () {
            this.$(":input[type='submit']").attr({
                "aria-disabled": "aria-disabled",
                "disabled": "disabled"
            });
            this.$(".js-status").spin();
        },

        _stopLoading: function () {
            this.$(":input[type='submit']").removeAttr("aria-disabled disabled");
            this.$(".js-status").spinStop();
        },

        _showSuccess: function (callback, ctx) {
            this.$(".js-status").spinStop().addClass("success");
            if (callback) {
                _.delay(_.bind(callback, ctx), 800);
            }
        },

        _showError: function (errorMessage) {
            var $error = $(JEP.Templates.Components.ConfirmationDialog.error({"error": errorMessage}));

            // clear previous error
            this.$(".js-confirm-content .aui-message.error.shadowed.closeable").remove();

            this._bindRemoveErrorOnClick($error);
            this.$(".js-confirm-content").prepend($error);
        },

        _bindRemoveErrorOnClick: function ($error) {
            var $closeButton = $error.find(".icon-close");
            if ($closeButton) {
                $closeButton.click(this._removeError);
            }
            else {
                $error.click(this._removeError)
            }
        },

        _removeError: function (e) {
            $(e.currentTarget).closest(".error").remove();
            e.preventDefault();
        }
    });

    Exports.create = function (options) {
        var view = new View(options);
        view.dialog = new AJS.dialog2(view.render());
        return view;
    };
});
