(function () {

    /**
     * Registers a new hotspot tour. Hotspot tours are little pulsating orange dots that are placed on elements,
     * allowing you to guide a user through a path or just call out an item on the page.
     *
     * @param {Object} opts
     * ... {Array[Object]} steps - array of steps in the tour. Check the StepModel for object properties
     * @param deps
     * ... {Object} brace - backbone brace library reference
     * ... {Object} jQuery - jQuery brace library reference
     * ... {Object} _ - underscore library reference
     */
    Chaperone.registerHotspotTour = function (opts, deps) {
        deps = deps || {};

        var _jQuery = deps.jQuery || window.jQuery;
        var defaultLibs = {
            underscore: window._,
            brace: window.Brace
        };

        deps = _jQuery.extend(defaultLibs, deps);

        var __ = deps.underscore;
        var _Brace = deps.brace;

        var StepModel = _Brace.Model.extend({
            namedAttributes: {
                selector: undefined,
                dismissingSelectors: undefined,
                tooltipText: String,
                tooltipOptions: Object,
                styleClass: String,
                dismissed: Boolean,
                onDismiss: Function,
                onShow: Function,
                offsetRight: Number,
                offsetTop: Number
            },
            /**
             * Resolves the css selector string for target. The target being the element the callout will be positioned against.
             * @returns {String} selector
             */
            resolveSelector: function () {
                var selector = this.getSelector();
                if (__.isFunction(selector)) {
                    selector = selector();
                }
                return selector;
            },
            /**
             * Resolves the css selector string for dismissing elements. The elements that when clicked will dismiss the step.
             * @returns {String} selector
             */
            resolveDismissingSelectors: function () {
                if (this.getDismissingSelectors()) {
                    var selector = this.getDismissingSelectors();
                    if (__.isFunction(selector)) {
                        selector = selector();
                    }
                    return selector;
                } else {
                    return this.resolveSelector();
                }
            }
        });

        var StepCollection = _Brace.Collection.extend({
            model: StepModel
        });

        var StepView = _Brace.View.extend({
            template: chaperone.hotspot.spot,

            render: function($elem) {
                var $html = _jQuery(this.template(this.model.toJSON()));
                if (this.model.getOffsetRight() !== undefined) {
                    $html.css("right", this.model.getOffsetRight());
                }
                if (this.model.getOffsetTop() !== undefined) {
                    $html.css("top", this.model.getOffsetTop());
                }
                var tooltipDefaults = {
                    gravity: 's',
                    delayIn: 0
                };
                $html.tooltip(_jQuery.extend(tooltipDefaults, this.model.getTooltipOptions()));
                $html.click(function(event) {
                    event.preventDefault();
                    event.stopPropagation();
                    $elem.click();
                });
                $elem.addClass("chaperone-hotspot-anchor");
                $elem.append($html);
                this.$elem = $elem;
                this.$el = $html;
            },

            dismiss: function() {
                this.trigger("dismissed");
                if (this.$el) {
                    this.$el.remove();
                }
                if (this.$elem) {
                    this.$elem.removeClass("chaperone-hotspot-anchor");
                }
            }
        });


        var HotspotModule = _Brace.View.extend({

            initialize: function(opts) {
                this.stepCollection = new StepCollection(opts.steps);
                this.stepViews = this.stepCollection.map(__.bind(function (model) {
                    var view = new StepView({model: model});
                    this.listenTo(view, "dismissed", function () {
                        // Do all the dismiss stuff here
                        view.model.setDismissed(true);
                        if (view.model.getOnDismiss()) {
                            view.model.getOnDismiss()();
                        }
                        _jQuery(document).unbindArrive(view.model.resolveSelector());
                        this.renderNextStep()
                    }.bind(this));
                    return view;
                }, this));
                this.renderNextStep();
            },

            /**
             * Adds the hotspot to the element found by the selector for that step (if not found immediately, it sets
             * up a mutation observer to wait for that element to appear)
             */
            renderNextStep: function() {
                var nextStep = __.find(this.stepViews, function (step) {
                    return !step.model.getDismissed();
                });
                if (nextStep) {
                    var selector = nextStep.model.resolveSelector();
                    var $target = _jQuery(selector);
                    if ($target.length === 0) {
                        var that = this;
                        _jQuery(document).arrive(selector, function() {
                            that._renderStep(nextStep, _jQuery(this));
                        });
                    } else {
                        this._renderStep(nextStep, $target);
                    }
                }
            },

            _renderStep: function(step, $elem) {
                step.render($elem);
                if (step.model.getOnShow()) {
                    step.model.getOnShow()();
                }
                // Apply the dismissers
                var dismissingSelectors = step.model.resolveDismissingSelectors();
                if (dismissingSelectors) {
                    _jQuery(dismissingSelectors).one("click", __.bind(step.dismiss, step));
                }
            },

            dismissTour: function() {
                __.each(this.stepViews, function (step) {
                    step.model.setOnShow(function(){});
                    step.model.setOnDismiss(function(){});
                    step.dismiss();
                });
            }
        });

        // Cancel any running hotspot tours
        if (Chaperone.currentHotspotTour) {
            Chaperone.currentHotspotTour.dismissTour();
        }
        Chaperone.currentHotspotTour = new HotspotModule(opts);

        return Chaperone.currentHotspotTour;
    };

})();

