(function(AJS, $, _, syncing, exports) {
    var syncFromServer = syncing.syncFromServer;
    var syncToServer = syncing.syncToServer;

    /**
     * Event fired when elements should be re-checked to see if they're visible.
     */
    var visibilityCheck = $.Callbacks();
    var setupAjaxStopVisibilityCheck = _.once(function() {
        $(document).on("ajaxStop", function() {
            visibilityCheck.fire();
        });
    });

    /**
     * Event fired when StepViews should realign themselves (e.g., window.resize)
     */
    var alignmentCheck = $.Callbacks();
    function deferredAlignmentCheck() {
        _.defer(alignmentCheck.fire);
    };

    var setupScrollResizeAlignmentCheck = _.once(function() {
        $(window).on('scroll resize', deferredAlignmentCheck);
    });

    /**
     * Find an element and return a promise for when it is found. Promise resolves to a jQuery object.
     *
     * @param selector selector to find
     * @return {jQuery.Deferred} promise that is resolved with the $el when it's found.
     */
    function findElement(selector) {
        var deferred = $.Deferred();

        function resolveIfFound() {
            var $el = $(selector);
            if ($el.length) {
                deferred.resolve($el);
                visibilityCheck.remove(resolveIfFound);
            }
        }

        visibilityCheck.add(resolveIfFound);
        resolveIfFound();

        return deferred;
    }

    /**
     * Return a StepView (aka a Feature Discovery InlineDialog)
     * @param {string} key - key this step is under
     * @param {Object} step - step object passed in from JS for how to display this step
     * @param {{numSteps:number, stepIndex:number, hasNext:boolean}} paging - Paging info for the steps this view is part of.
     * @param {Object} options
     * @return {{showWhenTargetVisible: Function, on: Function}}
     * @constructor
     */
    function StepView(key, step, paging, options) {
        var onClose = $.Callbacks();
        var onNext = $.Callbacks();
        var onAction = $.Callbacks();

        function renderFeatureDiscovery($target) {
            var inlineDialog;
            // the dialog id is unique per step within a feature. i.e. multiple steps means multiple dialogs.
            var dialogId = 'feature-discovery-' + key + (step.id ? '_' + step.id : '');

            function remove(eventToFire) {
                return function() {
                    $target.off('.chaperone');
                    inlineDialog.open = false;
                    visibilityCheck.remove(removeIfTargetHidden);
                    alignmentCheck.remove(repositionDialog);
                    if (eventToFire) {
                        eventToFire.fire();
                    }
                };
            }

            function removeIfTargetHidden() {
                if (!$target.is(":visible")) {
                    remove()();
                }
            }

            var removeAsClosed = remove(onClose);
            var extraAttributes = {};

            if (step.width) {
                extraAttributes.style = 'width: ' + step.width + 'px';
            }


            if (!$target.attr('aria-controls')) {
                $target.attr('aria-controls', dialogId);
                $target.on('click.chaperone', remove(onAction));
            }

            $(document.body).append(aui.inlineDialog2.inlineDialog2({
                id: dialogId,
                content: chaperone.featureDiscovery.content({
                    step: step,
                    paging: paging
                }),
                extraClasses: 'feature-discovery-dialog ' + dialogId + (step.extraClasses ? ' ' + step.extraClasses : ''),
                persistent: !!step.persistent,
                // gravity was a 'nswe' point, the direction the arrow would point to.
                // e.g. 'w' would be equivalent to the alignment `right middle`
                // lets map gravity to one of the corresponding values or use alignment if provided.
                alignment: step.alignment || (step.gravity || 's').replace(/[nswe]/g, function(c) {
                    return {
                        n: 'bottom middle',
                        s: 'top middle',
                        e: 'left middle',
                        w: 'right middle'
                    }[c];
                }),
                extraAttributes: extraAttributes
            }));

            inlineDialog = document.getElementById(dialogId);

            var $inlineDialog = $(inlineDialog);
            var actionSelector = '.feature-discovery-action';
            var nextSelector = '.feature-discovery-next';
            var closeSelector = '.feature-discovery-close';

            if (step.action && step.action.click) {
                $inlineDialog.on('click', actionSelector, step.action.click);
            }

            $inlineDialog.on('click', actionSelector, remove(onAction));
            $inlineDialog.on('click', nextSelector, remove(onNext));
            $inlineDialog.on('click', closeSelector, removeAsClosed);

            function repositionDialog() {
                if (inlineDialog._auiAlignment && inlineDialog._auiAlignment._auiTether) {
                    // When https://ecosystem.atlassian.net/browse/AUI-4301 is resolved this can be changed to the new method.
                    inlineDialog._auiAlignment._auiTether.position();
                }
            }

            alignmentCheck.add(repositionDialog);
            visibilityCheck.add(removeIfTargetHidden);
            inlineDialog.open = true;
        }

        return {
            /**
             * Show this view as soon as the target for it is found.
             * @return {*}
             */
            showWhenTargetVisible: function() {
                return findElement(step.selector).then(renderFeatureDiscovery);
            },
            /**
             * Attach an event.
             * @param name {string} name of the event to attach to. next, close, or action
             * @param fn {Function} handler for the event.
             */
            on: function(name, fn) {
                switch (name) {
                    case 'close':
                        onClose.add(fn);
                        break;
                    case 'next':
                        onNext.add(fn);
                        break;
                    case 'action':
                        onAction.add(fn);
                        break;
                }
            }
        }
    }

    /**
     * @typedef {Object} Step
     *
     * @type {object}
     * @property {string} [id] - an id for this step. Used to record if this step was progressively dismissed. selector will be used as a fallback.
     * @property {string} selector - the selector to attach this step's feature discovery dialog to.
     * @property {string} [gravity=n] - the direction of gravity for the dialog to 'float' against. DEPRECATED: use alignment instead
     * @property {string} [alignment] - the alignment of the dialog - see AUI InlineDialog 2 alignment - https://docs.atlassian.com/aui/latest/docs/inline-dialog.html#alignment
     * @property {number} [width=300] - a number of pixels wide to make the dialog. DEPRECATED: prefer CSS to change width
     * @property {string} [title] - the title/heading text for the dialog.
     * @property {string} [content] - HTML content for the body of the dialog.
     * @property {Object} [action] - an object describing an action that can be taken from the dialog. This takes the parameters
     *                    to the aui.buttons.button template, plus one additional parameter `click`, which can be a function
     *                    to trigger when the action is clicked.
     * @property {{href:string, text:string}} [moreInfo] - a link descriptor pointing to extended information about the feature.
     * @property {boolean} [once=false] - When true, the dialog will be shown once and immediately dismissed. When false, it will show until Close/Next/Action is pressed.
     */

    /**
     * Check if all steps in a set are dismissed.
     * @param {Object} steps
     * @returns {boolean}
     */
    function allStepsDismissed(steps) {
        if (!steps) {
            return false;
        }
        return Object
                .keys(steps)
                .every(function(key) {
                    return steps[key].isDismissed;
                });
    }

    /**
     * Register a feature discovery dialog or chain of feature discovery dialogs.
     * @param {string} key the key to use for already-viewed, already-dismissed lookups
     * @param {Array<Step>} steps list of objects representing how a step should display
     * @param {Object} [options]
     * @param {string} [options.dismissalMode] 'progressive' to dismiss each step one at a time
     * @param {boolean} [options.modal=false] When true, dialogs will require interaction before closing.
     */
    exports.registerFeature = function(key, steps, options) {
        options = options || {};

        var gotServerData = syncFromServer(key).pipe(function(serverData) {
            serverData = serverData || {};
            // check if all steps are dismissed as a whole, or all individual steps have been dismissed.
            if (serverData.isAllDismissed || allStepsDismissed(serverData.stepInfoById)) {
                return $.Deferred().reject();
            }

            // Not all feature discovery has been dismissed so setup event handlers
            setupAjaxStopVisibilityCheck();
            setupScrollResizeAlignmentCheck();

            return serverData;
        });

        function dismiss(step, serverData, dismissAll) {
            var stepDismissal = serverData.stepInfoById[step.id || step.selector];
            switch (options.dismissalMode) {
                case 'progressive': // progressive allows partial viewing of the steps, and you pick up where oyu left off.
                    stepDismissal.isDismissed = true;
                    syncToServer(key, JSON.stringify(serverData));
                    break;
                default:
                    // by default, if they dismiss one, they dismiss them all
                    // We want to dismiss agressively on a Close, but wait until the last dismissed
                    if (dismissAll && !serverData.isAllDismissed) {
                        serverData.isAllDismissed = true;
                        syncToServer(key, JSON.stringify(serverData));
                    }
                    break;
            }
        }

        // show each step in order, once the previous steps have been actioned.
        _.reduce(steps, function(readyForStep, step) {
            return readyForStep.then(function(serverData) {
                var stepInfoById = serverData.stepInfoById = serverData.stepInfoById || {};
                var stepId = step.id || step.selector;
                var stepInfo = stepInfoById[stepId] = stepInfoById[stepId] || {};

                // this step has already been actioned, move to the next.
                if (stepInfo.isDismissed) {
                    return readyForStep;
                }

                var numSteps = steps.length;
                var stepIndex = _.indexOf(steps, step);
                var hasNext = stepIndex < numSteps - 1;
                var stepView = new StepView(key, step, {
                    numSteps: numSteps,
                    stepIndex: stepIndex,
                    hasNext: hasNext
                }, options);

                // Move to the next item on next/action, or stop completely on close.
                var deferred = $.Deferred();
                stepView.on('close', deferred.reject);
                stepView.on('next', _.bind(deferred.resolve, deferred, serverData));
                stepView.on('action', _.bind(deferred.resolve, deferred, serverData));

                // show the dialog when we can find the target for it.
                var isShown = stepView.showWhenTargetVisible();

                var dismissOne = _.bind(dismiss, null, step, serverData, !hasNext);
                var dismissAll = _.bind(dismiss, null, step, serverData, 'all');

                if (step.once) { // shown once, then gone forever
                    isShown.then(dismissOne);
                } else { // shown until they do something about it
                    stepView.on('close', dismissAll);
                    stepView.on('next', dismissOne);
                    stepView.on('action', dismissOne);
                }

                return deferred;
            });
        }, gotServerData);
    };

    exports.checkFeatureVisibility = function() {
        visibilityCheck.fire();
    };

    exports.checkFeatureAlignment = function() {
        alignmentCheck.fire();
    };
})(AJS, AJS.$, _, Chaperone._internal.syncing, Chaperone);
