define('streams/inline-activity-streams', [
    'atlassian/libs/uri-1.14.1',
    'jira/skate',
    'underscore',
    'jquery'
], function(
    URI,
    skate,
    _,
    jQuery
) {
    var ACTIVITY_STREAM_TAG = 'inline-activity-stream';
    var STREAMS_URL = AJS.contextPath() + '/plugins/servlet/streams';
    var XML2JS = new X2JS();
    var DEFAULT_INCREMENT = 20;

    function getConfigFromURL(url) {
        var parsedURI = URI.parse(url);
        return URI.parseQuery(parsedURI.query);
    }

    var ActivityStream = skate(ACTIVITY_STREAM_TAG, {
        type: skate.type.ELEMENT,
        created: function(element) {
            var url = element.getAttribute('data-url');
            element.updateConfig(getConfigFromURL(url));
            element._showMoreIncrement = parseInt(element.getAttribute('data-show-more-increment')) || DEFAULT_INCREMENT;

        },
        attached: function(element) {
            element.refresh();
        },
        events: {
            'click .g-atl-activity-show-more': function(element, eventObject, currentTarget) {
                jQuery(currentTarget).parents(ACTIVITY_STREAM_TAG).get(0).showMore();
            }
        },
        prototype: {
            update: function() {
                var url = this.getAttribute('data-url');
                var deferred = jQuery.Deferred();
                jQuery.ajax({
                    url: this.getAttribute('data-url'),
                    success: function(data, status, xhr) {
                        this.updateStreamsModel(processXML(data));
                        deferred.resolve(data);
                    }.bind(this),
                    error: function(xhr, status, error) {
                        deferred.reject(error);
                    }
                });
                return deferred.promise();
            },
            refresh: function() {
                this.renderLoading();
                this.update()
                    .then(function(){
                        this.updateView();
                        jQuery(this).trigger('activity.stream:refreshed');
                    }.bind(this));
            },
            updateConfig: function(newConfig) {
                this._configuration = newConfig;
                //maxResults can come through as a string so we should be defensive about this
                this._configuration.maxResults = parseInt(this._configuration.maxResults);
            },
            updateStreamsModel: function(newModel) {
                this._streamsModel = newModel;
            },
            updateView: function() {
                var areThereMoreEntries = this._streamsModel && (this._configuration.maxResults <= this._streamsModel.length);
                this.innerHTML = InlineActivityStreams.Templates.main({
                    entries: this._streamsModel || [],
                    moreEntries: areThereMoreEntries
                });
            },
            setURL: function(url) {
                this.setAttribute('data-url', url);
            },
            renderLoading: function() {
                this.innerHTML = InlineActivityStreams.Templates.loading();
                jQuery(this).spin({className: 'g-atl-activity-loading'});
            },
            hide: function() {
                this.setAttribute('aria-hidden', 'true');
            },
            show: function() {
                this.setAttribute('aria-hidden', 'false');
            },
            showMore: function() {
                jQuery(this).find('.g-atl-activity-show-more').addClass('loading').empty().spin();
                this._configuration.maxResults += this._showMoreIncrement;
                this.setURL(ActivityStream.convertConfigToUrl(this._configuration));
                this.update().then(function() {
                    this.updateView();
                }.bind(this));
            }
        }
    });

    ActivityStream.convertConfigToUrl = function(configuration) {
        var urlParts = URI.parse(STREAMS_URL);
        urlParts.query = (urlParts.query) ? urlParts.query + "&" : "";
        urlParts.query += URI.buildQuery(configuration);
        return URI.build(urlParts);
    };

    //TODO: this is required because the server returns XML, ideally we would change the server to return JSON
    function processXML(xml) {
        var result;
        var rawObject = XML2JS.xml2json(xml);

        //xml to json conversion doesn't return an array if there is only one element
        if(rawObject.feed.entry instanceof Array) {
            result = _.map(rawObject.feed.entry, getEntryObject);
        } else if(rawObject.feed.entry) {
            result = [getEntryObject(rawObject.feed.entry)];
        }
        return result;
    }

    function getActivityIcon(links) {
        var result = _.findWhere(links || [], {_rel: "http://streams.atlassian.com/syndication/icon"}) || {
                _href: "//:0",  // browser will not perform a request
                _title: ""
            };
        return {
            src: result._href,
            title: result._title
        };
    }

    function getEntryObject(item) {
        if (item) {
            return {
                activityAuthor: item.author,
                activityTitleHTML: item.title.toString(),
                activityContentHTML: item.content && item.content.toString(),
                activityTime: moment(item.updated).fromNow(),
                activityDatetime: item.updated,
                activityIcon: getActivityIcon(item.link)
            }
        } else {
            return null;
        }
    }

    return ActivityStream;
});

require('streams/inline-activity-streams');