/**
 * Stickytable headers for confluence
 */
define("sticky-table-headers/viewpage", [
    "jquery",
    "ajs",
    "sticky-table-headers/utils"
], function(
    $,
    AJS,
    Utils
) {
    "use strict";

    /**
     * setupTable - Enable sticky tables for a group of table elements
     *
     * @param {HTMLElement|JqueryHtmlElement} table Setup stickyheaders for tables with a defined parameter for valid tables
     * @param {Object} options StickyTableHeaders jquery plugin options
     */
    var setupTable = function(table, options) {
        // setup stalky table headers
        var $validTables = $(table).filter(function() {
            return Utils.isAValidTable(this);
        });

        $validTables.stickyTableHeaders(options).addClass('stickyTableHeaders');

        AJS.bind('sticky-table-headers.change.options', function (event, newOptions) {
            $validTables.stickyTableHeaders(newOptions);
        });

        Utils.bindHorizontalScrolling($validTables);
    };

    /**
     * initialize stickyheaders for all confluence tables
     *
     */
    var initialize = function() {
        var $table = $(".confluenceTable");
        var $win = $(window);

        // We need to abort ASAP if we're in the documentation space
        if (!Utils.isNotDocumentationTheme()) {
            return;
        }

        // the editor structure by default does not have a thead
        Utils.moveHeaderRowsToTHead($table);

        // We use mutation observer in order to trigger resizing
        if (window.MutationObserver) {

            // we need to listen to all outer container that could cause reflow
            var target = document.querySelector("#page");

            // we need to exclude stickyheaders own mutations
            var stickyHeadersMutationTarget = /sticky|tableFloating|confluenceTh/;

            // listen for changes in order to make sure tableHeaders are repositioned upon reflow
            var observer = new MutationObserver(
                $.debounce(function(mutations) {
                    var validtriggerFlag = false;
                    var className;

                    for(var i=0, len=mutations.length; i<len; i++) {
                        className=mutations[i].target.className;

                        if (!stickyHeadersMutationTarget.test(className)) {
                            validtriggerFlag = true;
                            break;
                        }
                    }

                    if (validtriggerFlag) {
                        $win.trigger('resize.stickyTableHeaders');
                    }
                }, 0)
            );

            // start observing
            observer.observe(target, {
                subtree: true,
                attributes: true,
                childList: true
            });
        }

        // enable all confluence tables
        setupTable($table, {
            cacheHeaderHeight: true
        });
    };

    return {
        initialize: initialize,
        setupTable: setupTable
    };
});
