/**
 * Stickytable headers for confluence
 */
define("sticky-table-headers/utils", [
    "jquery"
], function(
    $
) {
    "use strict";

    /**
     * isNotAlreadyEnabled - is table already enabled
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var isNotAlreadyEnabled = function($table) {
        return !$table.hasClass("stickyTableHeaders");
    };

    /**
     * isNotNestedTable - is table inside a nested table
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var isNotNestedTable = function($table) {
        return !$table.parents("table").length > 0;
    };

    /**
     * isTheadDirectVisibleChild - is thead present as the visible direct child of a table
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var isTheadDirectVisibleChild = function($table) {
        return $table.find("> thead:first").is(":visible");
    };

    /**
     * hasTheadGotOnlyOneChild - Do we have only a single row inside the thead
     *
     * This will fix problems when people copy tables from the viewpage inside the editor and
     * add extra rows after the header. We should not enable sticky headers on tables that have more than
     * one TR inside the Thead
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var hasTheadGotOnlyOneChild = function($table) {
        return $table.find("thead").children().length === 1;
    };

    /**
     * isNotInsidePageLayout - is table inside a page layouts
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var isNotInsidePageLayout = function($table) {
        return !$table.closest(".columnLayout").length > 0;
    };

    /**
     * isNotDocumentationTheme
     *
     * Documentation theme is incompatible with stickyheaders
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var isNotDocumentationTheme = function() {
        return !$("body").hasClass("theme-documentation");
    };

    /**
     * areAllConfluenceThInTheSameRow - confluenceTh on different rows should be disabled
     *
     * We don't support vertical headers
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var areAllConfluenceThInTheSameRow = function($table) {
        return $table.find(".confluenceTh").parent().length === 1;
    };

    /**
     * areAllAdjacentCellsConfluenceTh - All cells in the same row needs to be confluenceTh
     *
     * Stickyheaders should only be applied if all cells are headers to avoid scrolling without
     * background problems
     *
     * @param {JqueryHTMLElement} table jquery table element
     * @return {Boolean}
     */
    var areAllAdjacentCellsConfluenceTh = function($table) {
        return $table.find(".confluenceTh").siblings().hasClass("confluenceTh");
    };

    /**
     * isAValidTable - Check if an html table can be used for sticky headers
     *
     * @param {HTMLElement} table HTMLElement
     * @return {Boolean} is a valid table element for stickyheaders
     */
    var isAValidTable = function(table) {
        var $table = $(table);

        return isNotDocumentationTheme() &&
               isNotAlreadyEnabled($table) &&
               isNotInsidePageLayout($table) &&
               isNotNestedTable($table) &&
               isTheadDirectVisibleChild($table) &&
               hasTheadGotOnlyOneChild($table) &&
               areAllConfluenceThInTheSameRow($table) &&
               areAllAdjacentCellsConfluenceTh($table);
    };

    /**
     * CONFDEV-33100 - The sticky headers library re-renders the sticky header if the window scrolls but not if you scroll
     * inside the table-wrap div, which is required for horizontal scrolling
     *
     * @param {JqueryHTMLTableElement} $tables tables to bind the scroll event to
     * @param {JqueryHTMLWindowElement} $win window to trigger the scroll event
     */
    var bindHorizontalScrolling = function($tables, $win) {
        var $window = $win || $(window);

        $tables.closest('.table-wrap').scroll(function() {
            $window.trigger('scroll.stickyTableHeaders');
        });
    };

    /**
     * moveHeaderRowsToTHead - Moves first row confluence th to a thead
     *
     * @param {HTMLTableElement|JqueryHTMLTableElement} table Html table element
     */
    var moveHeaderRowsToTHead = function(table) {
        $(table).each(function() {
            var $thisTable = $(this);
            var hasMergedColumns = $thisTable.find(".confluenceTh[rowspan]").length > 0;
            var $th = $thisTable.find("> :first-child > :first-child .confluenceTh");
            var alredyWrappedOnAThead = $th.length && $th.closest('thead').length > 0;
            var isFirstRow = $th.length && $th.parent().index() === 0;

            // we don't have a thead, so lets create one, but only if the confluenceTh are the first tr
            if (isNotNestedTable($thisTable) && isNotInsidePageLayout($thisTable) && !alredyWrappedOnAThead && isFirstRow && !hasMergedColumns) {
                // move to a thead
                var $headerElement = $("<thead />").append($th.parent());

                $thisTable.prepend($headerElement);
            }
        });
    };

    return {
        isAValidTable: isAValidTable,
        bindHorizontalScrolling: bindHorizontalScrolling,
        moveHeaderRowsToTHead: moveHeaderRowsToTHead,
        isNotDocumentationTheme: isNotDocumentationTheme
    };
});
