/**
 * This card is used to view task detail and edit the notes.
 */
MW.TaskDetailAbstractView = Backbone.View.extend({

    tagName: 'div',
    className: 'mw-taskdetail card',

    events: {
        "click .mw-drilldown-top" : "gotoTasks",
        "click .mw-taskdetail-save" : "gotoTasks",
        "click .mw-header .nav button": "gotoTasks", // mobile

        "keypress .mw-taskdetail-notes-textarea" : "handleNotesKeypress",
        "keypress .mw-taskdetail-title" : "handleTitleKeypress",

        "click .mw-taskdetail-notes" : "editNotes",
        "mousedown .mw-taskdetail-notes" : "mouseNotes",
        "focus .mw-taskdetail-notes" : "focusNotes",

        "focusout .mw-taskdetail-notes-textarea" : "saveNotes",
        "focusout .mw-taskdetail-title" : "saveTitle",

        "click .mw-taskdetail-delete": "deleteTask",
        "click .mw-task-checkbox": "toggleCompleted"
    },

    initialize: function(options) {
        this.globalEvents = options.globalEvents;

        this.globalEvents.bind("deleteTask", this.removeTask, this);
        this.globalEvents.bind("cleanupNestedCard", this.removeElementAndBindings, this);

        MW.KeyboardShortcuts.changeContext("taskDetail");
    },

    toggleCompleted: function() {
        if (this.completedCheckbox.hasClass("completed")) {
            this.model.setState("TODO");
        } else {
            this.model.setState("DONE");
        }
        this.renderCompleteState();
    },

    showCard: function(slideTime) {
        MW.Router.navigateToTask(this.model.id);
        this.showCardAnimation(slideTime);
    },

    handleNotesKeypress: function(e){
        e.stopPropagation();
        this.renderTags();
    },

    handleTitleKeypress: function(e) {
        this.renderTags();

        if (e.keyCode == 13) {
            this.saveTitle();
            e.preventDefault();
        }
    },

    mouseNotes: function() {
        this.mouse = true;
    },

    focusNotes: function() {
        // Need to distinguish mouse events for clicking on links
        if (!this.mouse) {
            this.editNotes();
        }
    },

    editNotes: function() {
        this.mouse = false;
        this.notesTextareaDiv.hide();
        this.relatedContent.addClass("selected");
        var end = this.notesTextarea.val().length;
        var textarea = this.notesTextarea.show().focus()[0];
        if (textarea.setSelectionRange) { // Not supported in IE8
            textarea.setSelectionRange(end, end); // Place the cursor at the end of existing text.
        }
    },

    saveNotes: function() {
        var text = this.notesTextarea.val();
        this.updateModel(function() {
            this.model.setNotes(text);
            this.updateNotes(text);
            this.notesTextarea.hide();
            this.notesTextareaDiv.show();
            this.relatedContent.removeClass("selected");
        });
    },

    updateNotes: function(text) {
        // Use 'pre' to preserve whitespace in IE8
        var displayText = this.convertLinks(MW.$("<pre />").text(text).html());
        if (!displayText) { // If there is no text, add in a placeholder
            this.notesTextareaDiv.addClass("placeholder").text(AJS.I18n.getText("com.atlassian.mywork.add.notes"));
        } else {
            this.notesTextareaDiv.removeClass("placeholder").html(displayText);
        }
        this.notesTextarea.val(text);
    },

    convertLinks: function(text) {
        var exp = /(\b(https?|ftp|file):\/\/[\-A-Z0-9+&@#\/%?=~_|!:,.;]*[\-A-Z0-9+&@#\/%=~_|])/ig;
        var nbsp = String.fromCharCode(160);
        return text.replace(/ {2}/g, nbsp + nbsp).replace(/\n/g, '<br />').replace(exp,"<a href='$1' target='_blank'>$1</a>");
    },

    saveTitle: function() {
        this.updateModel(function(){
            this.model.setTitle(this.getTitleText());
        });
    },

    updateModel: function(f) {
        _.bind(f, this)();
    },

    deleteTask: function(e) {
        e.preventDefault();
        this.showDeleteConfirmation();
    },

    removeTask: function() {
        this.model.destroy();
        this.globalEvents.trigger("showTaskCardBack");
    },

    getTitleText: function() {
        return this.titleInput.val() || this.titleInput.text();
    },

    getNotesText: function() {
        // If there is notes text already stored with the task, use it.
        return this.model.get("notes") || "";
    },

    gotoTasks: function() {
        this.saveTitle();
        this.saveNotes();
        this.globalEvents.trigger("showTaskCardBack");
    },

    /**
     * Remove any bindings to global events, any bindings on this view object itself, and remove the DOM element.
     */
    removeElementAndBindings: function() {
        if (!this.$el.hasClass('showing')) {
            this.model.off(null, null, this);
            this.globalEvents.off(null, null, this);
            this.undelegateEvents();
            this.remove();
        }
    },

    render: function() {
        var $el = MW.$(this.el);

        var first = $el.children().length === 0;
        if (first)
        {
            this.renderTemplate($el);
        }

        this.titleInput = $el.find(".mw-taskdetail-title");
        this.notesTextarea = $el.find(".mw-taskdetail-notes-textarea");
        this.notesTextareaDiv = $el.find(".mw-taskdetail-notes");
        this.relatedContent = $el.find(".mw-related-content");

        var notes = this.getNotesText();
        this.updateNotes(notes);
        if (first) {
            this.notesTextarea.hide();
        }

        this.completedCheckbox = $el.find(".mw-task-checkbox");
        this.container = $el.find(".mw-taskdetail-container");

        this.renderTagPlaceholder();

        this.renderCompleteState();

        if (this.model.get('readonlyTitle')) {
            this.titleInput.attr("title", AJS.I18n.getText('com.atlassian.mywork.task.title.inlinetask'));
        }
        this.relatedContent.attr("title", AJS.I18n.getText('com.atlassian.mywork.task.related.content.title'));

        return this;
    },

    renderCompleteState: function () {
        var done = this.model.isCompleted();
        this.completedCheckbox.toggleClass("completed", done);
        this.container.toggleClass("completed", done);
    },

    // Tags are currently a dark feature
    renderTagPlaceholder: function() {
        if (!!AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nat.tags")) {
            MW.$(".mw-taskdetail-container").append(MyWork.Templates.taskDetailTags({
                task: this.model.attributes
            }));
            this.renderTags();
        }
    },

    renderTags: function() {
        if (!!AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nat.tags")) {
            var text = this.getTitleText(),
                bodyText = this.notesTextarea.val(),
                tagText = text + " " + bodyText,
                tagRegex = /(^|\s)#(\w+)/g,
                matches = tagText.match(tagRegex),
                tagHolder = MW.$(this.el).find(".mw-taskdetail-taglist");

            tagHolder.empty();

            if (!matches) {
                tagHolder.addClass("empty");
                tagHolder.text(AJS.I18n.getText('com.atlassian.mywork.tooltip.tag'));
            } else {
                tagHolder.removeClass("empty");
                _.each(matches, function(category) {
                    tagHolder.append(MyWork.Templates.categoryLabel({text: category.split("#")[1]}));
                });
            }
        }
    },

    // Deprecated - this flag should come from the Configuration for Confluence.
    isInlineTask: function() {
        return this.model.get("application") === "com.atlassian.confluence" && this.model.get("entity") === "inline-task";
    }

});
