var jiraIntegration = window.jiraIntegration || {};
jiraIntegration.JiraIssuesDialog = (function(
    $,
    _
) {


    // If the project avatar fails to load (probably because the current user isn't logged into JIRA)
    // We replace it with the default avatar from JIRA 6.1
    // When https://jira.atlassian.com/browse/STASHDEV-4736 is fixed, use a data url here instead so we can keep the markup correct.
    $(document).on('jira-img-error', 'img', function() {
        var $defaultDiv = $('<div class="jira-default-project-avatar"></div>').text(this.getAttribute('alt'));
        $(this).replaceWith($defaultDiv);
    });

    // Example usage from Stash
    /*
    var dialog = new JiraIssuesDialog({
        ajax : function(options) {
            // Stash's ajax transport handles common issues behind the scenes
            return ajax.ajax($.extend({
                statusCode : { // False means Stash should avoid handling errors/corner cases in status codes 200 and 500.
                    200 : false
                    500 : false
                }
            }, options));
        }
    });

    //when done with it
    dialog.destroy();
    // end product code
    */

    // This is here so that it can be shared between multiple instances of JiraIssuesDialog.
    var unauthedJiraInstancesPromise;

    /**
     * A dialog for showing and transitioning JIRA issues.
     *
     * Dependencies:
     *  - requires spin.js and the accompanying jQuery plugin
     *  - AUI, jQuery, Underscore
     *
     * @param options
     *  REQUIRED:
     *   - ajax : an ajax transport function that mimics the jQuery API, and reacts to status codes other than 200 and 500.
     *            Those status codes will be handled by the dialog itself and can be safely ignore.
     *            This must return a promise with signatures: done(data, textStatus, xhr) and fail(xhr)
     *  OPTIONAL:
     *   - id : the ID of the dialog,
     *   - issueKeys : the issue keys to show in the dialog. can be passed in later with dialog.setIssueKeys([])
     *   - entityKey : optional identifier for host entity object to use as context for ordering JIRA servers. see LinkableEntityResolver
     *   - maxIssues : the maximum number of issues to show
     *   - headerTitle : The title of the dialog. Can be specified later with dialog.setTitle()
     *   - get$AuthMessage : a function to return a custom "authentication required" message (jQuery object expected)
     *   - get$ConnectionLostMessage : a function to return a custom "JIRA unreachable" message (jQuery object expected)
     *   - get$NoIssuesExistMessage : a function to return a custom "those issue keys are bogus or oyu don't have permission" message (jQuery object expected)
     *
     * @constructor
     */
    function JiraIssuesDialog(options) {
        this.options = $.extend({}, JiraIssuesDialog.prototype.defaults, options);

        this.issueKeys = this.options.issueKeys || [];
        this.entityKey = this.options.entityKey || null;
        _.bindAll(this, 'show', 'hide', 'onOAuthSuccess', 'setIssueKeys', 'setEntityKey', 'setTitle', 'reset', 'destroy',
            '_renderJiraData');
        this._initDialog();
        this._truncated = false;
        this._jiraIssuesData = null;
        this._jiraIssuesDataPromise = null;
        this._events = {};
        this._currentXhr = null;
    }

    JiraIssuesDialog.prototype.defaults = {
        id: 'jira-issues-dialog',
        issueKeys: null,
        entityKey: null,
        maxIssues: 10,
        headerTitle: AJS.I18n.getText('dialog.title'),
        get$AuthMessage : function(jiraInstances) {
            var $authMessage = $(jiraIntegration.templates.dialog.authenticate({
                extraClasses : 'jira-oauth-dialog',
                multipleJiras : jiraInstances.length > 1
            }));
            var $listSpan = $authMessage.find('.jira-instance-list');
            jiraInstances.forEach(function(jiraInstance, index) {
                if(index !== 0) {
                    $listSpan.append(", ");
                }
                $listSpan.append(_generateOAuthLink(jiraInstance));
            });

            return $authMessage;
        },
        get$ConnectionLostMessage : function(exceptions) {
            if (exceptions.length === 1) {
                var exception = exceptions[0];

                return $(jiraIntegration.templates.dialog.error({
                    titleContent: jiraIntegration.templates.dialog.linkTitle({
                        text: AJS.I18n.getText('dialog.error.connection'),
                        link: exception.applicationUrl,
                        linkText: exception.applicationName
                    }),
                    text: AJS.I18n.getText('communication.problem') + ':\n' + exception.message,
                    extraClasses: 'communication-error'
                }));
            } else {
                var exceptionMessages = _.map(exceptions, function(exception) {
                    return jiraIntegration.templates.dialog.linkText({
                        link : exception.applicationUrl,
                        linkText: exception.applicationName,
                        text: exception.message,
                        extraClasses: 'multi'
                    });
                });
                return $(jiraIntegration.templates.dialog.error({
                    title: AJS.I18n.getText('dialog.error.connection.multi'),
                    content: exceptionMessages.join(''),
                    extraClasses: 'communication-error'
                }));
            }
        },
        get$UnknownErrorMessage : function() {
            return $(jiraIntegration.templates.dialog.error({
                title: AJS.I18n.getText('dialog.error.unknown.title'),
                text: AJS.I18n.getText('server.error')
            }));
        },
        get$NoIssuesExistMessage : function() {
            return $(jiraIntegration.templates.dialog.message({
                title: AJS.I18n.getText('dialog.issues.none'),
                text: AJS.I18n.getText('dialog.warning.mismatch.all'),
                iconClass: 'icon-jira',
                extraClasses: 'no-issues-exist'
            }))
        },
        ajax : function(options) {
            alert("JiraIssuesDialog requires the option 'ajax(options)' to be specified.\n" +
                  "This method should handle response status codes other than 200 and 500.");
        }
    };

    // 640 is an arbitrary "good" minHeight, but not if the window is shorter than that.
    // 250 is an arbitrary value that is currently (and hopefully in the future) larger than the default AUI "buffer"
    function minHeight (windowHeight) {
        return Math.max(0, Math.min(windowHeight - 250, 640));
    }

    JiraIssuesDialog.prototype.on = function(eventName, fn) {
        this._events[eventName] = this._events[eventName] || [];
        this._events[eventName].push(fn);
    };
    JiraIssuesDialog.prototype.off = function(eventName, fn) {
        if (this._events[eventName]) {
            this._events[eventName] = _.without(this._events[eventName], fn);
        }
    };
    JiraIssuesDialog.prototype._trigger = function(eventName /*, arguments */) {
        var self = this;
        var args = Array.prototype.slice.call(arguments, 1);
        if (this._events[eventName]) {
            _.map(this._events[eventName], function(fn) {
                fn.apply(self, args);
            });
        }
    };

    JiraIssuesDialog.prototype._initDialog = function() {
        var self = this;
        var windowHeight = $(window).height();
        this.dialog = new AJS.Dialog({
            width: 980,
            height: minHeight(windowHeight),
            id: this.options.id,
            closeOnOutsideClick: true,
            focusSelector : '.button-panel-cancel-link' // we don't want the transition buttons to focus by default
        });

        $('#' + this.options.id).addClass('jira-issues-dialog');
        this.setTitle();
        this.dialog.addCancel(AJS.I18n.getText('button.close'), this.hide)
            .addPanel('jira-issues-list');

        this._transitionClickHandler = function(e) {
            e.preventDefault();
            self._onTransitionClicked(this);
        };

        this.dialog.getCurrentPanel().body.on('click', '.jira-transition-cancel', function(e) {
            e.preventDefault();
            self._onTransitionCancelled(this);
        });

        $(document).on('hideLayer', function() {
            // Because AUI Dialog..
            if (self.multiIssueView) {
                self.multiIssueView.destroy();
            }
        });
        this._updateOAuthStatus();

        var $window = $(window);
        this.onWindowResize = _.debounce(function() {
            self._setMinHeight($window.height());
        }, 200);
        $window.on('resize', this.onWindowResize);
    };

    JiraIssuesDialog.prototype._setMinHeight = function(windowHeight) {
        var $body = this.dialog.getCurrentPanel().body;
        $body.css('min-height', minHeight(windowHeight));
        if ($body.is(':visible')) {
            this.dialog.updateHeight();
        }
    };

    function openInSameTab(e) {
        return (!e.which || e.which === 1) &&
              !(e.metaKey || e.ctrlKey || e.shiftKey || (e.altKey && !$.browser.msie));
    }

    JiraIssuesDialog.prototype.reset = function() {
        this._cancelPendingRender();
        this._truncated = false;
        this._jiraIssuesData = null;
    };

    /**
     * Sets the issue keys displayed by the dialog. If the issue keys are different to what is currently being displayed,
     * it will reset the dialog and load the new issues.
     *
     * @param issueKeys
     * @param initialIssueKey optional issue key to select when data has loaded, or immediately if the issue keys are the same.
     */
    JiraIssuesDialog.prototype.setIssueKeys = function(issueKeys, initialIssueKey) {
        issueKeys = _.uniq(issueKeys);
        this.initialIssueKey = initialIssueKey;
        if (!this.issueKeys || issueKeys.length !== this.issueKeys.length || _.difference(issueKeys, this.issueKeys).length) {
            this.issueKeys = issueKeys;
            this.reset();
        }
    };

    JiraIssuesDialog.prototype.setEntityKey = function(entityKey) {
        if (this.entityKey !== entityKey) {
            this.entityKey = entityKey;
            this.reset();
        }
    };

    JiraIssuesDialog.prototype.destroy = function() {
        this.reset();
        this.dialog.remove();
        this._events = null;
        this.dialog = null;
        this.issueKeys = null;
        this.entityKey = null;
        this.options = null;

        if (this.onWindowResize) {
            $(window).off('resize', this.onWindowResize);
            this.onWindowResize = null;
        }
    };

    JiraIssuesDialog.prototype.setTitle = function(str) {
        this.dialog.addHeader(str || this.options.headerTitle || JiraIssuesDialog.prototype.defaults.headerTitle);
    };

    JiraIssuesDialog.prototype.show = function() {
        this._bindOAuthCallbacks();
        $(document).on('click', '.jira-transition', this._transitionClickHandler);
        this.dialog.getCurrentPanel().body.removeClass('multi-issues');
        this.dialog.show();
        this._setMinHeight($(window).height());
        this._loadJiraData().done(this._renderJiraData);
        this._trigger('show');
    };

    JiraIssuesDialog.prototype.hide = function() {
        this._removeJiraDialog();
        this.dialog.hide();
        this._unbindOAuthCallbacks();
        $(document).off('click', '.jira-transition', this._transitionClickHandler);
        this._trigger('hide');
    };

    JiraIssuesDialog.prototype._removeJiraDialog = function() {
        $('#jira-instance-list-dialog').remove();
    };

    JiraIssuesDialog.prototype._getRestIssuesUrl = function() {
        return AJS.contextPath() + '/rest/jira-integration/latest/issues?issueKey=' +
               _.map(this.issueKeys, encodeURIComponent).join('&issueKey=') +
               (this.entityKey ? '&entityKey=' + encodeURIComponent(this.entityKey) : '') +
               '&fields=*all,-comment&minimum=' + this.options.maxIssues;
    };

    JiraIssuesDialog.prototype._getRestTransitionUrl = function(issueKey, applicationId) {
        return AJS.contextPath() + '/rest/jira-integration/latest/issues/' + encodeURIComponent(issueKey) +
               '/transitions?applicationId=' + encodeURIComponent(applicationId) +
               '&fields=*all,-comment';
    };

    JiraIssuesDialog.prototype._getOAuthStatusUrl = function() {
        return AJS.contextPath() + '/rest/jira-integration/latest/servers';
    };

    JiraIssuesDialog.prototype._loadJiraData = function () {
        if ((!this._jiraIssuesData || this._jiraIssuesDataIncomplete) && !this._jiraIssuesDataPromise) {
            var self = this;
            var $panel = this.dialog.getCurrentPanel().body;
            if (!$.fn.spin) {
                alert("This plugin requires spin.js and the jquery plugin to be available.");
            }
            $panel.empty().spin('large');
            this._jiraIssuesDataPromise = this.options.ajax({
                type : 'GET',
                url : this._getRestIssuesUrl(),
                dataType: 'json',
                contentType: 'application/json',
                jsonp: false
            }).pipe(function(data, textStatus, xhr) {
                if (data.errors && data.errors.length) {
                    self._handleAuthenticationRequired(data.errors);
                    return $.Deferred().reject();
                }

                if (data && data.length) {
                    // HACK JIRA 5.0 returns an empty list of transitions, so we have to grab them manually.
                    // We can't differentiate between 5.0 crappiness and an actual lack of transitions on all issues.
                    var needToGetTransitionsFor = _.filter(data, function(issue) {
                        return issue.canTransition && issue.transitions.length === 0;
                    });

                    if (needToGetTransitionsFor.length) {
                        _.forEach(needToGetTransitionsFor, function(issue) {
                            issue.transitions = self._getTransitions(issue.key, issue.applicationLinkId, data);

                            issue.transitions.done(function() {
                                sanitizeIssueUrls(issue);
                            });
                        });
                        _.forEach(_.difference(data, needToGetTransitionsFor), sanitizeIssueUrls);

                        // wait for the first one to finish
                        var waitingOn = needToGetTransitionsFor[0].transitions;
                        var deferred = $.Deferred();
                        waitingOn.done(function() {
                            deferred.resolve(data); // need to resolve with the full dataset still
                        });
                        waitingOn.fail(deferred.reject);
                        return deferred;
                    }

                    _.forEach(data, sanitizeIssueUrls);
                }

                return data;
            }, function(xhr, textStatus, error) {
                self._handle500Error(xhr);
            }).always(function() {
                $panel.spinStop();
            }).done(function(data) {
                if (data.length) {
                    self._truncated = data.length > self.options.maxIssues;
                    data = _.first(data, self.options.maxIssues);
                }

                self._jiraIssuesData = data;
                self._jiraIssuesDataIncomplete = data.length != self.issueKeys.length;
            });
            return self._jiraIssuesDataPromise.always(function() {
                self._jiraIssuesDataPromise = null;
            });
        } else if (this._jiraIssuesDataPromise) {
            return this._jiraIssuesDataPromise;
        } else {
            return $.Deferred().resolve(this._jiraIssuesData).promise();
        }
    };

    JiraIssuesDialog.prototype._getTransitions = function(issueKey, applicationId, issuesData) {
        var self = this;
        var issue;
        issuesData = issuesData || this._jiraIssuesData;
        if (issuesData) {
            issue = _.find(issuesData, function(i) { return i.key === issueKey });
            if (issue && issue.transitions && issue.transitions[0] && issue.transitions[0].fields) {
                return $.when(issue.transitions);
            }
        }
        return this.options.ajax({
            type : 'GET',
            url : this._getRestTransitionUrl(issueKey, applicationId),
            dataType: 'json',
            contentType: 'application/json',
            jsonp: false
        }).pipe(function(data) {
            if (data.errors && data.errors.length) {
                self._handleAuthenticationRequired(data.errors);
                return $.Deferred().reject();
            } else {
                if (issue) {
                    issue.transitions = data;
                }
                return data;
            }
        }, function(xhr, textStatus, error) {
            self._handle500Error(xhr);
        });
    };



    JiraIssuesDialog.prototype._transitionIssue = function(issue, transition, applicationId, fields) {
        var self = this;
        return this.options.ajax({
            type : 'POST',
            url : this._getRestTransitionUrl(issue.key, applicationId),
            dataType: 'json',
            contentType: 'application/json',
            jsonp: false,
            data : JSON.stringify({
                fields : fields,
                transition : {
                    id : transition.id
                }
            })
        }).pipe(function(data) {
            if (data.errors && data.errors.length) {
                self._handleAuthenticationRequired(data.errors);
                return $.Deferred().reject();
            }

            // HACK until future JIRAs are old JIRAs and old JIRAs are unsupported
            if (issue.fields.status.id === data.fields.status.id && transition.to && transition.to.id !== data.fields.status.id) {
                // the state is the same, and it's not supposed to be.
                // There is a bug in JIRA causing this to return as "succeeded"
                // https://jira.atlassian.com/browse/JRA-33909
                // We handle it as an error manually.
                return $.Deferred().reject(formErrorResponse(AJS.I18n.getText('transition.error.status.unchanged')));
            }

            self._trigger('transitioned', issue.key);

            // HACK JIRA 5.0 returns an empty list of transitions, so we have to grab them manually.
            // We can't differentiate between 5.0 crappiness and an actual lack of transitions on all issues.
            if (data.canTransition && !data.transitions.length) {
                var deferred = $.Deferred();
                self._getTransitions(data.key, data.applicationLinkId, [ data ]).done(function() {
                    sanitizeIssueUrls(data);
                    self._updateCachedData(data);
                    deferred.resolve(data);
                }).fail(deferred.reject);
                return deferred;
            }

            sanitizeIssueUrls(data);
            self._updateCachedData(data);
            return data;
        });
    };

    JiraIssuesDialog.prototype.isVisible = function() {
        return this.dialog.getCurrentPanel().body.is(':visible');
    };

    JiraIssuesDialog.prototype.onOAuthSuccess = function(e) {
        if (this.isVisible()) {
            this._removeJiraDialog();
            this._updateOAuthStatus(true);
            this._loadJiraData().done(this._renderJiraData);
            // stops AppLinks from refreshing the page because we update our dialog
            e.preventDefault();
        }
    };

    JiraIssuesDialog.prototype._onTransitionClicked = function(transitionEl) {
        var self = this;

        var $transition = $(transitionEl);
        var $form = $transition.closest('.jira-transition-form');
        var $issue = this.dialog.getCurrentPanel().body.find('.jira-issue-detailed');
        var $allTransitions = $issue.find('.jira-transition');

        var applicationId = $issue.attr('data-application-id');
        var issueKey = $issue.attr('data-issue-key');
        var transitionId = $transition.attr('data-transition-id');

        var issue = _.find(this._jiraIssuesData, function(i) { return i.key === issueKey });


        $allTransitions.attr('aria-disabled', 'true').prop('disabled', true);

        var transition;
        var fields;
        var renderDependsOn = this._getTransitions(issueKey, applicationId).pipe(function(data) {
            transition = _.find(data, function(t) { return t.id === transitionId });

            if (transition) {
                if ($form.length) {
                    fields = jiraIntegration.transitionForm.getJSON($form);

                } else if (jiraIntegration.transitionForm.isRequired(issue, transition)) {
                    return $.Deferred().reject('formRequired');
                }
            } else {
                // HACK: We don't have data for this transition anymore. This means the issue state has changed in JIRA out from under us.
                // We are going to purposefully attempt the transition in JIRA anyway so we get the standard JIRA error response.
                transition = { id : transitionId, name : $transition.text() };
            }

            return self._transitionIssue(issue, transition, applicationId, fields);
        });

        var $spinner;
        if ($form.length) {
            $spinner = $('<div class="jira-transition-spinner"></div>').prependTo($form.find('.buttons'));
        } else {
            $spinner = $('<div class="jira-transition-spinner"></div>').appendTo($issue.find('.jira-transitions'));
        }
        $spinner.spin();

        // Always update the list item since they could have navigated to another issue.
        var $issueItem = $issue.closest('.list-and-detail-view').find('> .list-view .issue-item[data-issuekey="'+issue.key+'"]');
        renderDependsOn.done(function(issue) {
            var status = issue.fields.status;
            $issueItem.find('.issue-item-status').html(jiraIntegration.templates.dialog.itemIssueStatus({
                issueStatus: status
            }));
        });

        this._registerPendingRender(renderDependsOn).always(function() {
            $allTransitions.attr('aria-disabled', 'false').prop('disabled', false);
            $spinner.spinStop().remove();
        }).done(function(issue) {
            $issue.replaceWith(self._renderDetailedIssue(issue));
        }).fail(function(xhrOrDataOrFormRequired) {
            if (!xhrOrDataOrFormRequired) {
                return;
            }

            if (xhrOrDataOrFormRequired === 'formRequired') {
                // need to fill out some fields first
                $issue.children('.jira-header').nextAll().remove();
                $issue.append(jiraIntegration.transitionForm.$render(issue, transition));
                $issue.addClass("jira-transitioning");
                return;
            }

            var data;
            if (xhrOrDataOrFormRequired.responseText) {
                try {
                    data = JSON.parse(xhrOrDataOrFormRequired.responseText);
                } catch (e) {
                    data = formErrorResponse(AJS.I18n.getText("server.error"));
                }
            } else {
                data = xhrOrDataOrFormRequired;
            }

            if (data && data.errors) {
                $issue.children('.jira-header').nextAll().remove();
                if (transition) {
                    $issue.append(jiraIntegration.transitionForm.$render(issue, transition, fields, data));
                    $issue.addClass("jira-transitioning");
                } else if (data.errors && data.errors.length) {
                    var errorItems = _.map(data.errors, function(error) {
                        return '<li>' + AJS.escapeHtml(error.message) + '</li>';
                    });

                    $issue.append(aui.message.error({ content : '<ul class="jira-errors">' + errorItems.join('') + '</ul>' }));
                } else {
                    $issue.replaceWith(self.options.get$UnknownErrorMessage());
                    self._setMinHeight(0);
                }
            }
        });
    };

    JiraIssuesDialog.prototype._renderDetailedIssue = function(issue) {
        var $detailedIssue = $(jiraIntegration.templates.dialog.detailedIssue({ issue: issue }));
        return $detailedIssue.attr('data-last-updated', new Date().getTime());
    };

    JiraIssuesDialog.prototype._onTransitionCancelled = function(cancelEl) {
        var $cancel = $(cancelEl);
        var $issue = $cancel.closest('.jira-issue-detailed');
        var issueKey = $issue.attr('data-issue-key');
        var issue = _.find(this._jiraIssuesData, function(i) { return i.key === issueKey });

        $issue.replaceWith(this._renderDetailedIssue(issue));
    };

    JiraIssuesDialog.prototype._updateCachedData = function(issue) {
        this._jiraIssuesData = _.map(this._jiraIssuesData, function(i) {
            return (i.key === issue.key) ? issue : i;
        });
    };

    JiraIssuesDialog.prototype._handleAuthenticationRequired = function(authErrors) {
        var self = this;

        this._removeViewWarning();

        unauthedJiraInstancesPromise.then(function(unauthedJiraInstances) {
            // if the number of errors differ to the number of authed JIRAs we got at the start something has probably
            // changed and we should recheck the list before showing the error message.
            if (authErrors.length !== unauthedJiraInstances.length) {
                self._updateOAuthStatus(true);
            }

            unauthedJiraInstancesPromise.then(function(unauthedJiraInstances) {
                self.dialog.getCurrentPanel().body.empty().append(
                    self.options.get$AuthMessage(unauthedJiraInstances)
                );
                self._setMinHeight(0);
            });
        });

    };

    JiraIssuesDialog.prototype._removeViewWarning = function() {
        this.dialog.popup.element.find('.view-warning').remove();
    };

    JiraIssuesDialog.prototype._handle500Error = function(xhr) {
        if (xhr.status === 500) {
            var jsonResponse;
            try {
                jsonResponse = JSON.parse(xhr.responseText);
            } catch(e) {}

            var $message;
            var commErrors = jsonResponse && jsonResponse.errors;
            if (!commErrors || !commErrors.length) {
                $message = this.options.get$UnknownErrorMessage();
            } else {
                $message = this.options.get$ConnectionLostMessage(commErrors);
            }

            this._removeViewWarning();
            this.dialog.getCurrentPanel().body.empty().append($message);
            this._setMinHeight(0);
        }
    };

    // I am waiting on dependsOnPromise before I can render.
    // Give me a promise that I can use to either render when it resolves, or abort my render if
    // it is rejected.
    JiraIssuesDialog.prototype._registerPendingRender = function(dependsOnPromise) {
        this._cancelPendingRender();
        var pendingRender = this._pendingRender = $.Deferred();
        dependsOnPromise.done(function() {
            pendingRender.resolveWith(this, arguments);
        }).fail(function() {
            pendingRender.rejectWith(this, arguments);
        });
        return this._pendingRender;
    };
    JiraIssuesDialog.prototype._cancelPendingRender = function() {
        if (this._pendingRender) {
            this._pendingRender.reject();
            this._pendingRender = null;
        }
    };

    JiraIssuesDialog.prototype._renderJiraData = function() {
        var self = this;
        var $view;
        var $panel = this.dialog.getCurrentPanel().body;

        this._cancelPendingRender();

        var warningText;
        var showOAuthInWarning;
        var maxMinHeight;
        switch (this._jiraIssuesData.length) {
            case 0 :
                $view = this.options.get$NoIssuesExistMessage();
                break;
            case 1 :
                showOAuthInWarning = this.issueKeys.length > 1;

                $view = this._renderDetailedIssue(this._jiraIssuesData[0]);
                // with single-issues, there is less chance of big height changes compared to multi-issue,
                // but we still we want a min-height set to avoid a short first issue causing a tiny dialog
                // which sucks when you open a long transition form.
                // cap the min-height at windowHeight in case of short windows.
                maxMinHeight = $(window).height();
                break;
            default :


                if (this._truncated) {
                    warningText = AJS.I18n.getText('dialog.warning.limit', this.options.maxIssues);
                } else if (this.issueKeys.length !== this._jiraIssuesData.length) {
                    showOAuthInWarning = true;
                }

                var selectedIssue = _.findWhere(this._jiraIssuesData, { key: this.initialIssueKey });
                $view = $(jiraIntegration.templates.dialog.multiIssueView({
                    issues: this._jiraIssuesData,
                    selectedIssue: selectedIssue
                }));

                this.multiIssueView = new ListAndDetailView($view, function ($selectedIssue, $listViewContainer, $detailViewContainer, e) {
                    if (!openInSameTab(e)) {
                        // Straight to JIRA on a middle click.
                        return;
                    }
                    e.preventDefault();

                    self._cancelPendingRender();

                    var issueKey = $selectedIssue.attr('data-issuekey');
                    var issueData = _.find(self._jiraIssuesData, function (issue) {
                        return issue.key === issueKey;
                    });

                    // HACK may have to wait for transitions to load
                    if (issueData.transitions && issueData.transitions.promise) {
                        $detailViewContainer.empty().spin('large');
                        self._registerPendingRender(issueData.transitions).always(function() {
                            $detailViewContainer.spinStop();
                        }).done(function() {
                            $detailViewContainer.html(self._renderDetailedIssue(issueData));
                        });
                    } else {
                        $detailViewContainer.html(self._renderDetailedIssue(issueData));
                    }
                });

                $panel.addClass('multi-issues');
                // with multi-issues, we want a min-height set to avoid a short first issue causing a tiny dialog
                // which sucks for viewing a sibling that is much longer.
                // cap the min-height at windowHeight in case of short windows.
                maxMinHeight = $(window).height();
                break;
        }

        this._removeViewWarning();
        if (warningText) {
            this.dialog.popup.element.find('.dialog-button-panel').prepend(jiraIntegration.templates.dialog.warningMessage({
                warningText : warningText
            }));
        } else if (showOAuthInWarning) {
            unauthedJiraInstancesPromise.then(this._renderOAuthStatus.bind(this));
        }

        $panel.html($view);
        $view.find('.issue-status').attr('data-last-updated', new Date().getTime());
        // don't enforce a minimum height by default. This changes for displaying issues, see above.
        this._setMinHeight(maxMinHeight || 0);
    };

    JiraIssuesDialog.prototype._bindOAuthCallbacks = function() {
        var $document = AJS.$(document);
        $document.bind('applinks.auth.approved', this.onOAuthSuccess);

        // create a hidden banner div.applinks-auth-confirmation-container so that the confirmation msg is not shown
        $('<div class="applinks-auth-confirmation-container hidden"></div>').appendTo(this.dialog.popup.element);
    };

    JiraIssuesDialog.prototype._unbindOAuthCallbacks = function() {
        var $document = AJS.$(document);
        $document.unbind('applinks.auth.approved', this.onOAuthSuccess);

        $('.applinks-auth-confirmation-container.hidden').remove();
    };

    /**
     *
     * @param {boolean} [force] If `true` the request will be made even if a result already exists.
     * @private
     */
    JiraIssuesDialog.prototype._updateOAuthStatus = function(force) {
        if (unauthedJiraInstancesPromise && !force) {
            return;
        }
        unauthedJiraInstancesPromise = this.options.ajax({
            type : 'GET',
            url : this._getOAuthStatusUrl()
        }).then(function(jiraInstances) {
            return jiraInstances.filter(function(jiraInstance) {return !jiraInstance.authenticated;});
        });
    };

    JiraIssuesDialog.prototype._renderOAuthStatus = function(unauthedJiraInstances) {
        this._removeJiraDialog();
        this._removeViewWarning();
        var $buttonPanel = this.dialog.popup.element.find('.dialog-button-panel');
        if (unauthedJiraInstances.length === 0) {
            $buttonPanel.prepend(jiraIntegration.templates.dialog.warningMessage({
                warningText : AJS.I18n.getText('dialog.warning.mismatch.partial')
            }));
            return;
        }

        var $footerMessage = $(jiraIntegration.templates.dialog.oauthFooterMessage({
            includeDialog: unauthedJiraInstances.length > 2,
            multipleJiras: unauthedJiraInstances.length > 1
        }));
        var $jiraInstancePicker = $footerMessage.find('.jira-instance-picker');
        unauthedJiraInstances.slice(0,2).forEach(function(jiraInstance, index) {
            if (index !== 0) {
                $jiraInstancePicker.append(", ");
            }
            $jiraInstancePicker.append(_generateOAuthLink(jiraInstance));
        });

        var $dialogContentsUL = $footerMessage.find('.jira-instance-list');
        unauthedJiraInstances.slice(2).forEach(function(jiraInstance) {
            $dialogContentsUL.append($("<li></li>").append(_generateOAuthLink(jiraInstance)));
        });

        $buttonPanel.prepend(jiraIntegration.templates.dialog.warningMessage({
            warningText : ''
        }));
        $buttonPanel.find('.view-warning').html($footerMessage);
    };

    function _generateOAuthLink(jiraInstance) {

        // We use ApplinksUtils to create the <a> so it attaches the correct event handlers onto it

        var $span = ApplinksUtils.createAuthRequestInline(null, {
            id: jiraInstance.id,
            authUri: AJS.contextPath() + '/plugins/servlet/applinks/oauth/login-dance/authorize?applicationLinkID='
                    + encodeURIComponent(jiraInstance.id),
            appUri: jiraInstance.displayUrl,
            appName: jiraInstance.name
        });

        var $appLink = $span.find('a.applink-authenticate')
            .text(jiraInstance.name)
            .attr('title', jiraInstance.name);

        return $appLink.wrap('<span class="instance"></span>');
    }

    function formErrorResponse(message) {
        return {
            errors : [{
                message : message
            }]
        };
    }

    // pulled from ListAndDetailView in Stash

    var keycodes = {
        j: 74,
        k: 75
    };

    function ListAndDetailView($listAndDetailView, selectHandler, options) {
        this.options = $.extend({}, ListAndDetailView.prototype.defaults, options);
        this.$listView = $listAndDetailView.find('.list-view');
        this.$detailView = $listAndDetailView.find('.detail-view');
        this.selectHandler = selectHandler;
        _.bindAll(this, 'itemClickHandler', 'shortcutHandler');
        this.init();
    }

    ListAndDetailView.prototype.defaults = {
        selectedClass: 'selectedItem'
    };

    ListAndDetailView.prototype.init = function() {
        var self = this;
        this.bindShortcuts();
        this.$listView.on('click', 'li', this.itemClickHandler);
    };

    ListAndDetailView.prototype.itemClickHandler = function(e) {
        var selectedClass = this.options.selectedClass;
        this.$listView.find('li.' + selectedClass).removeClass(selectedClass);
        var $listItem = $(e.currentTarget).addClass(selectedClass);
        this.selectHandler($listItem, this.$listView, this.$detailView, e);
    };

    ListAndDetailView.prototype.destroy = function() {
        this.unbindShortcuts();
    };

    ListAndDetailView.prototype.shortcutHandler = function(e) {
        if ($(e.target).is('input,textarea,select')) {
            return;
        }
        var nextPrev;
        if (e.which === keycodes.j) {
            nextPrev = 'next';
        } else if (e.which === keycodes.k) {
            nextPrev = 'prev';
        } else {
            return;
        }

        var $selected = this.$listView.find('li.' + this.options.selectedClass);
        var $target = $selected[nextPrev]('li');
        $target.click();
        this._scrollItemIntoView($target);
    };

    ListAndDetailView.prototype.bindShortcuts = function() {
        $(document).on('keydown', this.shortcutHandler);
    };

    ListAndDetailView.prototype.unbindShortcuts = function() {
        $(document).off('keydown', this.shortcutHandler);
    };

    // Scroll into view if necessary. Will only scroll if there is an <a>
    ListAndDetailView.prototype._scrollItemIntoView = function($listItem) {
        $listItem.find('a').focus().blur();
    };


    var displayUrlFromIssueUrl = /.*(?=\/browse\/[^\/]*-\d+)/;
    var rpcUrlFromIssueUrl = /.*(?=\/rest\/api\/\d+\/)/;

    /**
     * We requested these issues from JIRA through Stash's RPC url. We now need to transition them all over to the
     * user-facing display URL. Also, JIRA gives us some relative URLs. We need to to make those relative to JIRA instead
     * of relative to Stash.
     *
     * @param issue
     */
    function sanitizeIssueUrls(issue) {
        var displayBaseUrl = displayUrlFromIssueUrl.exec(issue.url)[0];
        var rpcBaseUrl = rpcUrlFromIssueUrl.exec(issue.self)[0];

        function setSanitized(val, key, obj) {
            if (typeof val === 'string') {
                obj[key] = sanitizeUrl(val, displayBaseUrl, rpcBaseUrl);
                return;
            }
            _.each(val, setSanitized);
        }
        function setHtmlSanitized(obj, key) {
            if (obj[key]) {
                obj[key] = sanitizeHtmlUrls(obj[key], displayBaseUrl, rpcBaseUrl);
            }
        }
        eachNested(issue.fields, isUrlProp, setSanitized);
        eachNested(issue.renderedFields, isUrlProp, setSanitized);
        eachNested(issue.transitions, isUrlProp, setSanitized);
        if (issue.renderedFields) {
            setHtmlSanitized(issue.renderedFields, 'description');
            _.each(issue.renderedFields.comment && issue.renderedFields.comment.comments || [], function(comment) {
                setHtmlSanitized(comment, 'body');
            });
        }
    }


    function isUrlProp(val, key, obj) {
        return /[uU]rls?$/.test(key);
    }

    function sanitizeHtmlUrls(html, displayBaseUrl, rpcBaseUrl) {
        try {
            var $dummy = $('<div />').html(html);
            _.each(['href', 'src'], function(attr) {
                _.each($dummy.find('[' + attr + ']'), function(el) {
                    el.setAttribute(attr, sanitizeUrl(el.getAttribute(attr), displayBaseUrl, rpcBaseUrl));
                });
            });
            html = $dummy.html();
        } catch(e) {
            // Was not HTML, oh well.
        }
        return html;
    }

    // matches URI schemes (e.g. http://, https://, mailto:, hipchat://) and //
    var isAbsoluteUrl = /^((\w+:)?\/\/|\w+:)/;
    function sanitizeUrl(url, displayBaseUrl, rpcBaseUrl) {
        if (url.substring(0, rpcBaseUrl.length) === rpcBaseUrl) {
            return urlJoin(displayBaseUrl, url.substring(rpcBaseUrl.length));
        }
        if (!isAbsoluteUrl.test(url)) {
            return urlJoin(displayBaseUrl, url);
        }
        return url;
    }


    function eachNested(obj, filter, fn) {
        if (obj && typeof obj === 'object') {
            _.each(obj, function(val, key) {
                if (filter(val, key, obj)) {
                    fn(val, key, obj);
                }
                eachNested(val, filter, fn);
            });
        }
    }

    function urlJoin(a, b) {
        var aSlash = a.charAt(a.length - 1) === '/';
        var bSlash = b.charAt(0) === '/';
        return aSlash && bSlash ? a + b.substring(1) :
               aSlash || bSlash ? a + b :
                                  a + '/' + b;
    }


    return JiraIssuesDialog;
}(
    AJS.$,
    _
));

// init the inline-dialog2 skate component
require(['aui/inline-dialog2']);
