
var jiraIntegration = window.jiraIntegration || {};
jiraIntegration.JiraCreateIssueForm = (function($, _) {
    JiraCreateIssueForm.defaults = {
        formClass: 'jira-interation-create-issue-form',
        ignoreFieldsWithDefaultValue: true, // Filter out fields that has default value ~ to make the form shorter
        excludedField: ['project', 'issuetype', 'reporter'], // don't need display Reporter, use user config in applink as default reporter.
        renderSummaryAndDescription: false,
        get$unsupportedFieldMessage: function(unSupportedFields, createIssueUrl) {
            return $(jiraIntegration.templates.jiraCreateIssueForm.unsupportedFieldsMessage({
                unsupportedFields: unSupportedFields,
                createIssueUrl: createIssueUrl
            }));
        },
        get$unexpectedError: function(message) {
            return $(aui.message.error({content: AJS.escapeHtml(message)}));
        },
        get$unauthorizeMessage: function(applicationName) {
            return $(jiraIntegration.templates.jiraCreateIssueForm.authorize({
                extraClasses : 'jira-oauth-dialog',
                applicationName : applicationName
            }));
        },
        get$unsupportedServerMessage: function(serverUrl) {
            return $(jiraIntegration.templates.jiraCreateIssueForm.unsupportedServerMessage({
                serverUrl: serverUrl
            }));
        }
    };

    function JiraCreateIssueForm(opts) {
        this.configuration = _.extend({}, JiraCreateIssueForm.defaults, opts);

        // if renderSummaryAndDescription is true, definitely excludedField should include summary and description
        if (this.configuration.renderSummaryAndDescription) {
            this.configuration.excludedField.push('summary', 'description');
        }

        this._init();
    }

    JiraCreateIssueForm.prototype._trigger = function(eventName, options) {
        var eventHandler = this.configuration[eventName];
        if (eventHandler) {
            eventHandler.apply(this, Array.prototype.slice.call(arguments, 1));
        }
    };

    JiraCreateIssueForm.prototype._bindEventListener = function() {
        var me = this;
        this.defaultFields.server.on('change', function(e) {
            me.resetForm();
            var serverId = this.value;
            me.currentServerId = serverId;
            if (serverId) {
                me.formIsLoading(me.defaultFields.server);
                jiraIntegration.JiraCreateIssueForm.Data.isIssueCreationSupported(serverId)
                .done(function(issueCreationSupported) {
                    if (issueCreationSupported)  {
                        me._loadProjectData(serverId);
                        me.formLoadingCompleted(me.defaultFields.server);
                    } else {
                        var displayUrl = jiraIntegration.JiraCreateIssueForm.Data.getServerById(serverId).displayUrl;
                        me._handleUnsupportedServer(displayUrl);
                    }
                })
                .fail(function(error) {
                    me._handleAjaxError(error);
                });
            } else {
                jiraIntegration.JiraCreateIssueForm.FieldHelper.resetSelectField(me.defaultFields.project);
                me.defaultFields.project.trigger('change');
            }
            me._trigger('onServerChanged', this.value);
        });
        this.defaultFields.project.on('change', function() {
            if (this.value) {
                $('option[value=""]', me.defaultFields.project).remove(); // We don't need to show "select a project" any more
                me._loadIssueType(me.defaultFields.server.val(), this.value);
            } else {
                jiraIntegration.JiraCreateIssueForm.FieldHelper.resetSelectField(me.defaultFields.issueType);
            }
            me.resetForm();
            me.defaultFields.issueType.trigger('change');
            me._trigger('onProjectChanged', this.value);
        });
        this.defaultFields.issueType.on('change', function() {
            me.resetForm();
            if (this.value) {
                me._loadRequireFields();
            } else {
                me.$containerRequireField && me.$containerRequireField.html('');
                $('.aui-message', this.formEl).remove();
            }
            me._trigger('onTypeChanged', this.value);
        });
    };

    JiraCreateIssueForm.prototype._init = function() {
        this.container = $(this.configuration.container);

        if (this.container.length) {
            this._renderForm();
            this._bindEventListener();
            this._loadServerData();
        } else {
            throw new Error('container property should be defined.');
        }
    };

    JiraCreateIssueForm.prototype._renderForm = function() {
        this.formEl = $(jiraIntegration.templates.jiraCreateIssueForm.form({
            formClass: this.configuration.formClass
        }));

        if (this.configuration.renderSummaryAndDescription) {
            this._renderSummaryAndDescription();
        }

        this.container.append(this.formEl);
        this.defaultFields = this._getDefaultFields();

        // Enrich default fields style to select2
        _.each(this.defaultFields, _.bind(function(field) {
            this.createSelect2WithIcon($(field));
        }, this));

        this.$containerRequireField = $('.create-issue-required-fields', this.container);
        this._trigger('onFormRendered');
    };

    JiraCreateIssueForm.prototype._renderSummaryAndDescription = function() {
        var defaulFieldsContainer = $('.create-issue-default-fields', this.formEl);

        defaulFieldsContainer.append(aui.form.textField({
            labelContent: AJS.I18n.getText('create.issue.form.field.summary.label'),
            isRequired: true,
            name: 'summary'
        }));
        defaulFieldsContainer.append(aui.form.textareaField({
            labelContent: AJS.I18n.getText('create.issue.form.field.description.label'),
            name: 'description'
        }));
    };

    JiraCreateIssueForm.prototype._getDefaultFields = function() {
        var form = $('.'+ this.configuration.formClass, this.configuration.container);
        return {
            server: $('.server-select', form),
            project: $('.project-select', form),
            issueType: $('.issuetype-select', form)
        };
    };

    JiraCreateIssueForm.prototype._loadServerData = function() {
        var me = this;
        this.formIsLoading(this.defaultFields.server);
        $.when(jiraIntegration.JiraCreateIssueForm.Data.loadServers())
            .then(function(data) {
                me.formLoadingCompleted(me.defaultFields.server);
                me._loadServerDataComplete(data);
            });
    };

    JiraCreateIssueForm.prototype._loadRequireFields = function() {
        var me = this;
        var context = {
            serverId: this.defaultFields.server.val(),
            projectKey: this.defaultFields.project.val(),
            issueType: this.defaultFields.issueType.val()
        };

        this.formIsLoading(this.defaultFields.issueType);
        var fieldsMeta = jiraIntegration.JiraCreateIssueForm.Data.getFieldMetadata(context);

        fieldsMeta.done(
            function(meta) {
                var fields = me._getValidRequiredFields(meta);

                if (me.configuration.renderSummaryAndDescription) {
                    var descriptionMeta = meta.fields.description;
                    var descriptionIsRequired = !!(descriptionMeta && descriptionMeta.required);
                    me._setDescriptionAsRequiredField(descriptionIsRequired);
                }

                me._renderRequireFields(fields, context);
                me.formLoadingCompleted(me.defaultFields.issueType);
            }
        ).fail(_.bind(me._handleAjaxError, this));
    };

    JiraCreateIssueForm.prototype._loadServerDataComplete = function(servers) {
        if (servers.length) {
            if (servers.length === 1) { // hide server fields if we have only one instance
                jiraIntegration.JiraCreateIssueForm.FieldHelper.hideField(this.defaultFields.server);
            }
            jiraIntegration.JiraCreateIssueForm.FieldHelper.fillSelectData(this.defaultFields.server, servers, true);
            this._loadProjectData(servers[0].id);
            this.currentServerId = servers[0].id;
            this._hasMultiServer = servers.length > 1;
        } else {
            // Show Jira Server Warning
            this._handlerUnexpectedError(AJS.I18n.getText("create.issue.form.field.server.error.empty"));
        }
    };

    JiraCreateIssueForm.prototype._loadProjectData = function(serverId) {
        var me = this;
        this.formIsLoading(this.defaultFields.project);
        var promise = jiraIntegration.JiraCreateIssueForm.Data.loadProjects(serverId);
        promise.then(
            function(projects) {
                if (projects.errors && projects.errors.length) {
                    var err = projects.errors[0];

                    if (err.authenticationUri) {
                        me._handleAuthorizeError(err);
                    } else {
                        me._handlerUnexpectedError(err.message);
                    }
                } else {
                    if (projects.length) {
                        jiraIntegration.JiraCreateIssueForm.FieldHelper.fillSelectData(me.defaultFields.project, projects);
                        jiraIntegration.JiraCreateIssueForm.FieldHelper.resetSelectField(me.defaultFields.issueType);
                    } else {
                        me._handlerUnexpectedError(AJS.I18n.getText("create.issue.form.field.project.error.empty"));
                    }
                }
                me.formLoadingCompleted(me.defaultFields.project);
            },
            _.bind(me._handleAjaxError, this)
        );
    };

    JiraCreateIssueForm.prototype._loadIssueType = function(serverId, projectId) {
        var issueTypes = jiraIntegration.JiraCreateIssueForm.Data.getIssueTypes(serverId, projectId);
        jiraIntegration.JiraCreateIssueForm.FieldHelper.fillSelectData(this.defaultFields.issueType, issueTypes, true);
    };

    JiraCreateIssueForm.prototype._renderRequireFields = function(fields, context) {
        this.$containerRequireField.html('');
        var unsupportedFields = _.reject(fields, jiraIntegration.fields.canRender);

        if (unsupportedFields.length) {
            this._handleUnsupportedFields(unsupportedFields);
        } else {
            this.$containerRequireField.html(_.map(fields, function(field) {
                return jiraIntegration.fields.renderField(null, field, null, null);
            }).join(''));
            jiraIntegration.fields.attachFieldBehaviors(this.$containerRequireField, {
                serverId: context.serverId,
                projectKey: context.projectKey
            }, null);
            this._trigger('onRequiredFieldsRendered', fields, unsupportedFields);
        }
    };

    JiraCreateIssueForm.prototype._handlerUnexpectedError = function(message) {
        var $message = this.configuration.get$unexpectedError(message);
        $('.aui-message', this.formEl).remove();
        this.formEl.prepend($message);
        this.formLoadingCompleted();
        this.formHasError();
    };

    JiraCreateIssueForm.prototype._handleUnsupportedFields = function(unsupportFields) {
        unsupportFields = _.map(unsupportFields, function(item) {
            return AJS.escapeHtml(item.name);
        });
        var $message = this.configuration.get$unsupportedFieldMessage(unsupportFields, this._getCreateJiraIssueUrl());
        $('.aui-message', this.formEl).remove();
        this.formEl.prepend($message);
        this._trigger('onError');
        this.formLoadingCompleted();
    };

    JiraCreateIssueForm.prototype._handleUnsupportedServer = function(serverUrl) {
        var $message = this.configuration.get$unsupportedServerMessage(serverUrl);
        $('.aui-message', this.formEl).remove();
        this.formEl.prepend($message);
        this.formLoadingCompleted();
        this.formHasError();
    };

    JiraCreateIssueForm.prototype._handleAjaxError = function(xhr) {
        var me = this;
        if (xhr.status >= 500 && xhr.status < 600) {
            me._handlerUnexpectedError(AJS.I18n.getText("create.issue.form.error.proxy") + ':' + xhr.status);
        }
        me.formHasError();
    };

    JiraCreateIssueForm.prototype._handleAuthorizeError = function(data) {
        var me = this;
        this.formHasError();
        var $message = this.configuration.get$unauthorizeMessage(data.applicationName);
        this.formEl.append($message);

        $('.oauth-init', $message).on('click', function(e) {
            AppLinks.authenticateRemoteCredentials(data.authenticationUri, function() {
                me.resetForm();
                me._loadProjectData(me.currentServerId);
            }, function() {
                me._handlerUnexpectedError(AJS.I18n.getText('create.issue.form.oauth.refused', data.applicationName));
            });
            e.preventDefault();
        });
    };

    JiraCreateIssueForm.prototype._getCreateJiraIssueUrl = function() {
        var projectId = this.defaultFields.project.find("option:selected").val();
        var issueTypeId = this.defaultFields.issueType.find("option:selected").val();
        var createIssueUrl = jiraIntegration.JiraCreateIssueForm.Data.getServerById(this.currentServerId).displayUrl;

        createIssueUrl = createIssueUrl + "/secure/CreateIssueDetails!Init.jspa?pid=" + projectId + "&issuetype=" + issueTypeId;

        var summaryValue = this._getFieldValue("summary");
        if (summaryValue.length) {
            createIssueUrl = createIssueUrl + "&summary=" + encodeURIComponent(summaryValue);
        }

        var descriptionValue = this._getFieldValue("description");
        if (descriptionValue.length) {
            createIssueUrl = createIssueUrl + "&description=" + encodeURIComponent(descriptionValue);
        }

        return createIssueUrl;
    };

    JiraCreateIssueForm.prototype._getValidRequiredFields = function (fieldMeta) {
        var me = this;
        return _.filter(fieldMeta.fields, function(field) {
            var typeId = field.schema ? (field.schema.system || field.schema.custom || field.schema.customId) : field;
            var isExcludedField = _.contains(me.configuration.excludedField, typeId);
            return (field.required && !isExcludedField && !field.hasDefaultValue);
        });
    };

    JiraCreateIssueForm.prototype._getFieldValue = function(fieldName) {
        var $field = $("[name='" + fieldName + "']", this.formEl);
        return ($field) ? $.trim($field.val()) : "";
    };

    JiraCreateIssueForm.prototype._setDescriptionAsRequiredField = function(isRequiredField) {
        var descriptionLabel = $('.field-group [name="description"]', this.formEl).prev('label');
        descriptionLabel.find('.aui-icon.aui-icon-required').remove(); // try remove first to avoid duplicated

        if (isRequiredField) {
            descriptionLabel.append(aui.icons.icon({icon: 'required'}));
        }
    };

    JiraCreateIssueForm.prototype.resetForm = function() {
        $('.aui-message', this.formEl).remove();
        $('.field-group', this.formEl).show();
        if (this.configuration.renderSummaryAndDescription) {
            this._setDescriptionAsRequiredField(false);
        }
        if (!this._hasMultiServer) {
            jiraIntegration.JiraCreateIssueForm.FieldHelper.hideField(this.defaultFields.server);
        }
        this.$containerRequireField.html('');
    };

    JiraCreateIssueForm.prototype.formHasError = function() {
        $('.field-group', this.formEl).hide();
        this.$containerRequireField.html('');
        jiraIntegration.JiraCreateIssueForm.FieldHelper.showField(this.defaultFields.server);
        jiraIntegration.JiraCreateIssueForm.FieldHelper.setFieldDisabled($('.insert-issue-button'), true);
        this._trigger('onError');
        this.formLoadingCompleted();
    };

    JiraCreateIssueForm.prototype.getCurrentServer = function() {
        return jiraIntegration.JiraCreateIssueForm.Data.getServerById(this.currentServerId);
    };

    JiraCreateIssueForm.prototype.formIsLoading = function($byField) {
        if ($byField) {
            jiraIntegration.JiraCreateIssueForm.FieldHelper.setIsLoading($byField, true);
        }
        var allFields = $(':input', $(this.formEl));
        jiraIntegration.JiraCreateIssueForm.FieldHelper.setFieldDisabled(allFields, true);
    };

    JiraCreateIssueForm.prototype.formLoadingCompleted = function($byField) {
        if ($byField) {
            jiraIntegration.JiraCreateIssueForm.FieldHelper.setIsLoading($byField, false);
        } else {
            $('.aui-icon.aui-icon-wait', this.formEl).remove();
        }
        var allFields = $(':input', $(this.formEl));
        jiraIntegration.JiraCreateIssueForm.FieldHelper.setFieldDisabled(allFields, false);
    };

    JiraCreateIssueForm.prototype.createSelect2WithIcon = function($element) {
        if ($element.is('.server-select')) {
            $element.auiSelect2({"minimumResultsForSearch": -1});
        } else {
            var select2Config = $element.is('.project-select') ? {
                "formatSelection": this.projectSelectFormat,
                "formatResult": this.projectSelectFormat,
                "extraAttributes": this.defaultFields
            } : {
                "formatSelection": this.issueTypeSelectFormat,
                "formatResult": this.issueTypeSelectFormat,
                "minimumResultsForSearch": -1,
                "extraAttributes": this.defaultFields
            };
            $element.auiSelect2(select2Config);
        }
        $element.auiSelect2("val", "");
    };

    JiraCreateIssueForm.prototype.projectSelectFormat = function(option) {
        var serverId = this.extraAttributes.server.find('option:selected').val();
        var imageUrl = jiraIntegration.JiraCreateIssueForm.Data.getProjectIconUrl(serverId, option.id);
        return jiraIntegration.templates.fields.select2WithIconOption({
                    "optionValue" : option.text,
                    "iconUrl" : imageUrl,
                    "isProject" : true
               });
    };

    JiraCreateIssueForm.prototype.issueTypeSelectFormat = function(option) {
        var serverId = this.extraAttributes.server.find('option:selected').val();
        var projectId = this.extraAttributes.project.find('option:selected').val();
        var imageUrl = jiraIntegration.JiraCreateIssueForm.Data.getIssueTypeIconUrl(serverId, projectId, option.id);
        return jiraIntegration.templates.fields.select2WithIconOption({
                    "optionValue" : option.text,
                    "iconUrl" : imageUrl,
                    "isProject" : false
               });
    };

    return JiraCreateIssueForm;
})(AJS.$, window._);
