var jiraIntegration = window.jiraIntegration || {};

jiraIntegration.fields = (function($, _) {
    var datePickerFormat = 'YYYY-MM-DD';
    var stringHandler = {
        template : jiraIntegration.templates.fields.stringField,
        getContext : getStringContext,
        getValue : getStringValue
    };
    var userHandler = {
        template : jiraIntegration.templates.fields.stringField,
        getContext : getUserContext,
        getValue : getUserValue,
        behavior : addUserAutoCompleteBehavior
    };
    var labelTagHandler = {
        template : jiraIntegration.templates.fields.arrayField,
        getContext : getArrayContext,
        getValue : getArrayValue,
        behavior : addLabelAutoCompleteBehavior
    };
    var textareaHandler = {
        template : jiraIntegration.templates.fields.textareaField,
        getContext : getStringContext,
        getValue : getStringValue
    };
    var numberHandler = {
        template : jiraIntegration.templates.fields.numberField,
        getContext : getStringContext,
        getValue : getNumberValue
    };
    var arrayHandler = {
        template : jiraIntegration.templates.fields.arrayField,
        getContext : getArrayContext,
        getValue : getArrayValue
    };
    var allowedValuesHandler = {
        template : jiraIntegration.templates.fields.allowedValuesField,
        getContext : getAllowedValuesContext,
        getValue : _.bind(getAllowedValuesBase, null, 'id'),
        behavior : addSelect2Behavior
    };
    var multipleChoicesHandler = {
        template : jiraIntegration.templates.fields.allowedValuesField,
        getContext : getAllowedValuesContext,
        getValue : _.bind(getAllowedValuesBase, null, 'id')
    };
    var timeTrackingHandler = {
        template : jiraIntegration.templates.fields.timeTrackingField,
        getContext : getTimeTrackingContext,
        getValue : getTimeTrackingValue
    };
    var dateHandler = {
        template : jiraIntegration.templates.fields.dateField,
        getContext : getStringContext,
        getValue : getDateStringValue,
        behavior : datePickerBehavior
    };
    var select2WithIconHandler = {
        template: jiraIntegration.templates.fields.select2WithIconField,
        getContext: getSelect2WithIconContext,
        getValue: _.bind(getAllowedValuesBase, null, 'id'),
        behavior: addSelect2WithIconBehavior
    };

    var restTypes = {
        "com.pyxis.greenhopper.jira:gh-epic-label":                       stringHandler,
        "string":                                                         stringHandler,
        "summary":                                                        stringHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:textfield":    stringHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:url":          stringHandler,
        "environment":                                                    textareaHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:textarea":     textareaHandler,
        "description":                                                    textareaHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:float":        numberHandler,
        "array":                                                          arrayHandler,
        "labels":                                                         labelTagHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:labels":       arrayHandler,
        "resolution":                                                     allowedValuesHandler,
        "fixVersions":                                                    allowedValuesHandler,
        "priority":                                                       select2WithIconHandler,
        "versions":                                                       allowedValuesHandler,
        "components":                                                     allowedValuesHandler,
        "security":                                                       allowedValuesHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:version":      allowedValuesHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:multiversion": allowedValuesHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:project":      allowedValuesHandler,
        "assignee":                                                       userHandler,
        "reporter":                                                       userHandler,
        "timetracking":                                                   timeTrackingHandler,
        "duedate":                                                        dateHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:datepicker":   dateHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:multiselect":  multipleChoicesHandler,
        "com.atlassian.jira.plugin.system.customfieldtypes:select":       multipleChoicesHandler
    };

    var defaultRenderOptions = {
         // Filter out fields that has default value ~ to make the form shorter
        ignoreFieldsWithDefaultValue: false
    };

    function getBaseContext(typeId, restField, errors) {
        var name = restField.schema.system || "customfield_" + restField.schema.customId;

        return {
            labelText : restField.name,
            name : name,
            isRequired :  restField.required,
            errorTexts : errors[name],
            jiraType : typeId
        }
    }

    function getStringContext(context, restField, issue, values) {
        var name = context.name;
        context.value = ($.isPlainObject(values[name]) ? values[name].name : values[name]) || (issue && issue.fields[name]) || '';
        return context;
    }

    function getStringValue($fieldInput) {
        return $fieldInput.val();
    }

    function getDateStringValue($fieldInput) {
        var dateString = $fieldInput.val();
        if (dateString === "") {
            return null;
        } else {
            return dateString;
        }
    }

    function getNumberValue($fieldInput) {
        var valString = $fieldInput.val();
        if (isNumeric(valString)) {
            return Number(valString);
        }
        return valString || null;
    }

    function datePickerBehavior($fieldInput) {
        var $input = $fieldInput.find('input');

        // AUI datepicker has many problems on IE (issue: AUI-2071), so if user using IE, we will render it as textfield
        // This issue was fixed by Issac Gerges and this fix will be effect from AUI 5.3.5
        if (!!navigator.userAgent.match(/Trident/) && AJS.version < '5.3.5') {
            var isPlaceholderSupported = 'placeholder' in document.createElement('input');
            $input.attr('placeholder', datePickerFormat);

            // Support placeholder on IE 8-9
            if (!isPlaceholderSupported) {
                $input.on('focus', function() {
                    if ($input.val() === $input.attr('placeholder')) {
                        $input.val('');
                    }
                }).on('blur', function() {
                    if ($input.val() === '') {
                        $input.val($input.attr("placeholder"));
                    }
                }).blur();
            }
        } else {
            $input.datePicker({
                'overrideBrowserDefault': true
            });
        }
    }

    function getArrayContext(context, restField, issue, values) {
        var name = context.name;
        context.value = (values[name] && values[name].join(',')) || (issue && issue.fields[name] && issue.fields[name].join(','));
        return context;
    }

    function getArrayValue($fieldInput) {
        return _.map($fieldInput.val().split(','), $.trim);
    }

    function getAllowedValuesContext(context, restField, issue, values) {
        var name = context.name;
        var userInputValue = values[name];
        var issueValue = issue && issue.fields[name];
        var selectedValue;

        if (userInputValue) {
            selectedValue = $.isArray(userInputValue) ? _.pluck(userInputValue, 'name') : [ userInputValue.name ];
        } else if (issueValue) {
            selectedValue = $.isArray(issueValue) ? _.pluck(issueValue, 'name') : [ issueValue.name ];
        } else {
            selectedValue = [];
        }
        context.options = _.map(restField.allowedValues, function(val) {
            return {
                value : val.id,
                text : val.name || val.value,
                selected : _.contains(selectedValue, val.name || val.id)
            };
        });
        context.isMultiple = _.contains(restField.operations, 'add');
        return context;
    }

    function getAllowedValuesBase(propertyName, $fieldInput) {
        var val = $fieldInput.val();
        var multiple = $fieldInput.attr('multiple');
        var getValue = function(val) {
            var valueObject = {};
            valueObject[propertyName] = val;
            return valueObject;
        };
        if (multiple) {
            return  $.isArray(val) ?
                _.map(val, getValue) :
                [getValue(val)];
        }
        return getValue(val);
    }

    function addSelect2Behavior($inputField) {
        // Select2-fy multi selects
        $inputField.find('select[multiple]').auiSelect2();
    }

    function getUserContext(context, restField, issue, values) {
        var name = context.name;
        context.value = (values[name] && values[name].name) || (issue && issue.fields[name] && issue.fields[name].name) || '';
        return context;
    }

    function getUserValue($fieldInput) {
        return {
            name : $fieldInput.val()
        };
    }

    /** 
     * Add auto complete behavior to text field using auiSelect2
     */
    function addAutoCompleteBehavior($inputField, context, restType, issue, autoCompleteConfig) {
        var $input = $inputField.find('input');
        var fieldName = $inputField.attr('name');
        $input.removeClass('text'); // to avoid layout problem

        var defaultConfig = {
            minimumInputLength: 1,
            id: fieldName,
            name: fieldName,
            query: function(query) {
                function onsuccess(datas) {
                    query.callback({results: datas});
                }

                getAutoCompleteData(context, restType, query.term, issue).done(onsuccess);
            }
        };

        $input.auiSelect2($.extend(defaultConfig, autoCompleteConfig));

        $inputField.find('div.aui-select2-container').addClass('jira-select2-drop-box');
    }

    /**
     * Add user auto complete behavior to text field using auiSelect2
     */
    function addUserAutoCompleteBehavior($inputField, context, restType, issue) {
        var userAutoCompleteConfig = {
            formatInputTooShort: function() {
                return AJS.I18n.getText("field.user.search.prompt");
            },
            formatResult: function(result) {
                return jiraIntegration.templates.fields.userOptionSelect({
                    name: result.id,
                    displayName: result.text
                });
            }
        };

        addAutoCompleteBehavior($inputField, context, restType, issue, userAutoCompleteConfig);
    }

    function addLabelAutoCompleteBehavior($inputField, context, restType, issue) {
        var $input = $inputField.find('input');

        jiraIntegration.fields._labelPicker.build($input,
            function(term) {
                return getAutoCompleteData(context, restType, term, issue);
            }
        );
    }

    /** 
     * Generic method to get auto complete data from REST resource
     */
    function getAutoCompleteData(context, restType, term, issue) {
        var postData = $.extend({restType: restType, issueKey: (issue && issue.key) || '', term: term}, context);
        return $.ajax({
            type : 'POST',
            timeout: 0,
            contentType : 'application/json',
            dataType : 'json',
            url : AJS.contextPath() + '/rest/jira-integration/latest/fields/autocomplete',
            data : JSON.stringify(postData)
        });
    }

    function getTimeTrackingContext(context, restField, issue, values) {
        var name = context.name;
        context.value = (values[name] && values[name].remainingEstimate) || (issue && issue.fields[name] && issue.fields[name].remainingEstimate) || '';
        return context;
    }

    function getTimeTrackingValue($fieldInput) {
        return {
            remainingEstimate : $fieldInput.val()
        };
    }

    /**
     * Add select with icon behavior to select field using auiSelect2
     */
    function getSelect2WithIconContext(context, restField, issue, values) {
        var name = context.name;
        var selectedValue = (values[name] && values[name].name) || (issue && issue.fields && issue.fields[name] && issue.fields[name].name) || '';

        context.options = _.map(restField.allowedValues, function (val) {
            return {
                value : val.id,
                text : val.name,
                selected : selectedValue === val.name,
                iconUrl : val.iconUrl //add iconUrl attribute for option element to display icon
            };
        });
        return context;
    }

    function select2WithIconFieldFormat(state) {
        var iconUrl;
        if (state.id) {
            iconUrl = $(state.element).attr("data-icon-url");
        }

        return jiraIntegration.templates.fields.select2WithIconOption({
            "optionValue" : state.text,
            "iconUrl" : iconUrl
        });
    }

    function addSelect2WithIconBehavior($field, context, restType, issue) {
        if (!$.fn.auiSelect2) {
            AJS.log('AUI version 5.2 or greater is required as this plugin needs the .auiSelect2() jQuery plugin.');
            return;
        }

        var $select = $field.find('select');
        $select.addClass('jira-select2-drop-box');
        $select.auiSelect2({
            hasAvatar : true,
            minimumResultsForSearch : -1,
            formatSelection : select2WithIconFieldFormat,
            formatResult : select2WithIconFieldFormat
        });
    }

    /**
     * @param type {string} The JIRA string to expect listed as 'schema.system' or 'schema.customId' in the REST response
     * @param handler {{
     *         template : function(Object) : string,
     *         getContext : function(baseContext, restField, issue, values) :Object,
     *         getValue : function($renderedInput) : Object
     *         canRender : ?function(restField) : boolean
     *     }}
     *     getContext produces an object that the template can read
     *     template produces a string of HTML that contains an <input>, <textarea> or <select> whose name matches baseContext.name
     *     getValue takes in a jQuery object that represents the <input>, <textarea> or <select> form the template.
     *     canRender takes in a restField and returns true if it can be rendered, or false if it can't. This is optional and defaults to always returning true.
     */
    function addFieldHandler(type, handler) {
        if (_.has(restTypes, type) && console && console.warn) {
            console.warn('Redefining handler for type ' + type + ".");
        }
        restTypes[type] = handler;
    }

    function getFieldTypeFromRestField(restField) {
        return restField.schema ? (restField.schema.system || restField.schema.custom || restField.schema.customId) : restField;
    }

    function getFieldHandler(restField) {
        return restTypes[getFieldTypeFromRestField(restField)];
    }

    function getFieldHandlerFromInputField($inputField) {
        var typeId = getRestTypeFromInputField($inputField);
        return typeId && getFieldHandler(typeId);
    }

    function getRestTypeFromInputField($inputField) {
        return $inputField.closest('.jira-field').attr('data-jira-type');
    }

    function getCreateRequiredFieldsMeta(context, options) {
        return $.ajax({
            type : 'GET',
            timeout: 0,
            url : AJS.contextPath() + '/rest/jira-integration/1.0/servers/' + context.serverId
                + '/projects/' + context.projectKey + '/issue-types/' + context.issueType + '/fields-meta'
        }).pipe(function(data) {
            var fieldsMeta = [];
            _.each(data.fields, function(field) {
                var typeId = getFieldTypeFromRestField(field);
                if (field.required && !_.contains(options.excludedFields, typeId)) { // using key instead to avoid internationalize problem
                    fieldsMeta.push(field);
                }
            });
            return fieldsMeta;
        });
    }

    function isNumeric(stringNumber) {
        return /\d/.test(stringNumber) && /^-?\d*\.?\d*$/.test(stringNumber);
    }

    return {
        addFieldHandler : addFieldHandler,
        getFieldHandler : getFieldHandler,
        canRender : function(restField) {
            var restTypeId = getFieldTypeFromRestField(restField);
            var restType = restTypes[restTypeId];

            if (!restType) {
                return false;
            }

            return restField.operations && restField.operations.length && (!restType.canRender || restType.canRender(restField));
        },
        renderField : function(issue, restField, values, errors) {
            var restTypeId = restField.schema.system || restField.schema.custom || restField.schema.customId;
            var restType = restTypes[restTypeId];

            var baseContext = getBaseContext(restTypeId, restField, errors || {});

            var unrenderable = !restType || (restType.canRender && !restType.canRender(restField));
            var noPermission = !restField.operations || !restField.operations.length; // Hopefully this doesn't happen.
            if (unrenderable || noPermission) {
                baseContext.reasonContent = unrenderable ? AJS.I18n.getText('fields.unrenderable', '<a href="' + (issue ? issue.url : '#') + '">', '</a>') :
                                            noPermission ? AJS.I18n.getText('fields.no.permission', '<a href="' + (issue ? issue.url : '#') + '">', '</a>') :
                                            null;
                if (!baseContext.reasonContent) {
                    throw new Error('Invalid unrenderable reason.');
                }
                return jiraIntegration.templates.fields.unrenderableTypeField(baseContext);
            }

            return restType.template(restType.getContext(baseContext, restField, issue, values || {}));
        },
        getJSON : function($fieldInput) {
            var handler = getFieldHandlerFromInputField($fieldInput);
            return handler && handler.getValue && handler.getValue($fieldInput);
        },
        attachFieldBehaviors : function($container, context, issue) {
            $container.find('.jira-field').each(function(index, fieldInput) {
                var $fieldInput = $(fieldInput);
                var typeId = getRestTypeFromInputField($fieldInput);
                var handler = typeId && getFieldHandler(typeId);
                var behavior = handler && handler.behavior;
                if (behavior) {
                    behavior($fieldInput, context, typeId, issue);
                }
            });
        },
        renderCreateRequiredFields: function($container, afterElement, context, options, errorCallback) {
            options = _.extend({}, defaultRenderOptions, options);

            function renderFields(fields) {
                if (options.ignoreFieldsWithDefaultValue) {
                    fields = _.filter(fields, function(field) {
                        return !field.hasDefaultValue;
                    });
                }

                var unsupportedFields = _.filter(fields, function(field) {
                    return !jiraIntegration.fields.canRender(field);
                });
                if (unsupportedFields.length) {
                    if (errorCallback) {
                        errorCallback(unsupportedFields);
                    }
                    return;
                }
                $container.html(_.map(fields, function(field) {
                    return jiraIntegration.fields.renderField(null, field, null, null);
                }).join(''));

                jiraIntegration.fields.attachFieldBehaviors($container, {
                    serverId: context.serverId,
                    projectKey: context.projectKey
                }, null);
            }

            // if requiredFields is provided use it to render, otherwise make request to get requiredFields
            if (options.requiredFields) {
                renderFields(options.requiredFields);
            }
            else {
                getCreateRequiredFieldsMeta(context, options).done(renderFields);
            }
        }
    };
}(AJS.$, window._));