/**
 * Settings that each Autocomplete will be initialized on, depending on the trigger character used to activate the
 * autocomplete.
 */
define('confluence-templates/autocomplete-settings-variables', [
    'confluence/legacy',
    'ajs'
], function(
    Confluence,
    AJS
) {
    "use strict";

    return function () {

        var variableManager = Confluence.VariableManager.defaultVariableManager;

        function makeVariableDropdownItem(name, additionalAttrs) {
            var item = {
                callback: function(autoCompleteControl) {
                    autoCompleteControl.replaceWithSelectedSearchText();
                    createVariable(name, additionalAttrs);
                },
                name : AJS.escapeHtml(name),
                href : '#'
            };

            return item;
        }

        function createVariable(name, additionalAttrs) {
            !variableManager.contains(name) && variableManager.add(name);
            Confluence.VariablePlaceHolder.insertPlaceholder(name, additionalAttrs);
        }

        function ignoreCaseComparator(a, b) {
            if(!a && !b) { return 0; }
            if(!a) { return -1; }
            if(!b) { return 1; }

            return a.toLocaleLowerCase().localeCompare(b.toLocaleLowerCase());
        }

        // Link settings.
        return {

            ch : "$",
            cache: false,
            endChars : [],
            dropDownDelay: 0, // No delay needed because there is no AJAX request involved
            preventStartNodes: "",

            dropDownClassName: "autocomplete-variables",
            selectFirstItem: true,

            getHeaderText : function (autoCompleteControl, value) {
                if (value)
                {
                    return AJS.I18n.getText("editor.autocomplete.variables.header.text");
                }
                else
                {
                    return AJS.I18n.getText("editor.autocomplete.variables.header.text.hint");
                }
            },

            getAdditionalLinks : function (autoCompleteControl, value) {
                var unescapedValue = Confluence.unescapeEntities(value);
                if(variableManager.get(unescapedValue) || value.length === 0) {
                    // Variable exists or has no value yet.
                    return [];
                }
                return [
                    {
                        className: "dropdown-create-variable",
                        href: '#',
                        callback: function(autoCompleteControl) {
                            var searchText = autoCompleteControl.plainText();
                            autoCompleteControl.replaceWithSelectedSearchText();
                            createVariable(searchText);
                        },
                        name : AJS.I18n.getText("editor.autocomplete.variables.create", value)
                    }
                ];
            },

            getDataAndRunCallback : function(autoCompleteControl, val, callback) {
                var k;
                var i;
                var l;
                var unescapedVal = Confluence.unescapeEntities(val);
                var variables = variableManager.find(unescapedVal);
                var sortedVariables = [];
                var dropdownItems = [];

                for(k in variables) {
                    if(variables.hasOwnProperty(k)) {
                        sortedVariables.push(k);
                    }
                }
                sortedVariables.sort(ignoreCaseComparator);
                for(i = 0, l = sortedVariables.length; i < l; i++) {
                    k = sortedVariables[i];
                    dropdownItems.push(makeVariableDropdownItem(k, variables[k]));
                }
                callback([dropdownItems], val);
            },

            update : function(autoCompleteControl, link) {
            }
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-templates/autocomplete-settings-variables', function(AutocompleteSettingsVariables) {
    var AJS = require('ajs');
    var Confluence = require('confluence/legacy');
    var Meta = require('confluence/meta');

    AJS.bind("init.rte", function() {
        var tinymce = require('tinymce');

        if(Meta.get("content-type") !== "template") { return; }

        // Confluence 4.1.x uses tinymce.confluence, 4.2 onwards use Confluence.Editor
        if (!Confluence.Editor.Autocompleter) {
            Confluence.Editor.Autocompleter = tinymce.confluence.Autocompleter;
        }

        Confluence.Editor.Autocompleter.Settings["$"] = AutocompleteSettingsVariables();
    });
});
