define('confluence-quick-reload/utils/quick-reload-timer', [
    'jquery',
    'underscore'
],
/**
 * Creates a timer that responds to the user being active
 * and then slowly backs off when they leave the tab.
 * It fluctuates between {options.min} ms. when active to up to {options.max} ms. when inactive,
 * and will start to back-off after {options.inactivity} ms. of inactivity.
 * Consumers will need to call start to kick-off the initial timer.
 *
 * @exports confluence-quick-reload/utils/quick-reload-timer
 */
function (
    $,
    _
) {
    "use strict";

    // additional back-off decay multiplier. Only if visibility API is supported
    var INACTIVE_TAB_MULTIPLIER = 3;

    function SmartTimer(callback, overrideOptions) {
        this.options = $.extend({
            min: 1000 * 30,
            max: 1000 * 60 * 60,
            inactivity: 1000 * 60 * 2
        }, overrideOptions);

        // bind all methods to ensure 'this' always refers to SmartTimer object
        _.bindAll(this, 'start', 'stop', '_now', '_timeSinceLastSeen', '_setActive', '_setActivityTrigger', '_clamp',
                '_intervalMultiplier');

        this.callback = callback;

        this._timer = null;

        this._setActivityTrigger(this._setActive);
    }

    SmartTimer.prototype.start = function(forceCheck) {
        this.lastSeenTimestamp = this._now();
        this.stop(); // don't want to ever start two timers by accident
        if (forceCheck) {
            this.callback();
        }
        var runCallback = function() {
            this.callback();
            resetTimeout.call(this);
        };
        var resetTimeout = function() {
            this.stop();
            this._timer = setTimeout(_.bind(runCallback, this), this.options.min * this._intervalMultiplier());
        };
        resetTimeout.call(this);
    };

    SmartTimer.prototype.stop = function() {
        if (this._timer !== null) {
            clearTimeout(this._timer);
            this._timer = null;
        }
    };

    /**
     * Forces a timer update if the user has not focused/viewed the page for an extended period of time
     * @private
     */
    SmartTimer.prototype._setActive = function() {
        if (this._timer === null) {
            return; // timer is stopped
        }
        var wasInactive = this._timeSinceLastSeen() > this.options.inactivity;
        this.lastSeenTimestamp = this._now();
        if (wasInactive) {
            this.start(true); // true forces check immediately
        }
    };

    /**
     * The function passed as parameter will be executed when new page activity gets detected
     * @param {function} setActive
     */
    SmartTimer.prototype._setActivityTrigger = function(setActive) {
        var handleVisibilityChange = function () {
            if (!document.hidden) {
                setActive();
            }
        };
        document.addEventListener("visibilitychange", handleVisibilityChange, false);
        $(window).focus(setActive);
        $('body').click(setActive);
        $(window).scroll(setActive);
    };

    SmartTimer.prototype._now = function() {
        return new Date().getTime();
    };

    //Calculates time elapsed from last user activity on this page. Minimum possible value is always > 0.
    SmartTimer.prototype._timeSinceLastSeen = function() {
        return (this._now() + 1) - this.lastSeenTimestamp;
    };

    // Explicitly clamping values to avoid values less than 0 (zero)
    // and greater than 2147483647 (32 bit int which is upper cap for setInterval).
    // Both ranges of invalid values will result in zillions callbacks per second.
    SmartTimer.prototype._clamp = function(min, max, val) {
        var clamped = Math.max(Math.min(max, val), min);
        return isNaN(clamped) ? min : clamped;
    };

    //Calculates multiplier for interval. Intention is to have final interval anywhere between options.min and options.max.
    //It will double the interval time while user does not scroll or focus/click on the window. If the window is not visible,
    //the interval will go up by a factor of 3. The interval will not exceed options.max
    SmartTimer.prototype._intervalMultiplier = function() {
        var multiplier = (document.hidden ? INACTIVE_TAB_MULTIPLIER : 1) *  // if tab is inactive, push back-off decay faster
            Math.ceil(this._timeSinceLastSeen() / this.options.inactivity);
        return this._clamp(1, this.options.max / this.options.min, multiplier);
    };

    return SmartTimer;
});
