/* eslint confluence/must-use-amd:0 */
/**
 * If you need to debug tests, place a 'debugger' statement at the beginning on the test and run the test with
 * Chrome developer tools open. Don't forget to remove the debugger statement when finished
 */
(function($) {
    "use strict";

    var ADMIN_USER = {
        username: 'admin',
        fullName: 'A. D. Ministrator'
    };
    var TEST1_USER = {
        username: 'test1',
        fullName: 'Test User1'
    };
    var TEST2_USER = {
        username: 'test2',
        fullName: 'Test User2'
    };

    var pageId = 1234;
    var POLL_PERIOD = 30000; // 30 seconds

    AJS.test.require("com.atlassian.confluence.plugins.quickreload:quick-reload-resources", function () {

        module("Confluence quick reload plugin tests", {
            setup: function () {
                this.server = sinon.fakeServer.create();
                this.clock = sinon.useFakeTimers();

                AJS.Meta.set('page-id', pageId);

                // Event sent to inline comments plugin to highlight text on the page
                AJS.bind('qr:add-new-highlight', function (e, commentAndUser, deferred) {
                    deferred.resolve(commentAndUser);
                });
                this.quickReloadManager = require('confluence-quick-reload/main/quick-reload-manager');

                this.commentHandler = require('confluence-quick-reload/handlers/quick-reload-comments');
                this.inlineCommentHandler = require('confluence-quick-reload/handlers/quick-reload-inline-comments');
                this.pageEditHandler = require('confluence-quick-reload/handlers/quick-reload-page');
                this.quickReloadCount = require('confluence-quick-reload/utils/quick-reload-count');

                this.quickReloadManager.addHandler(this.commentHandler);
                this.quickReloadManager.addHandler(this.inlineCommentHandler);
                this.quickReloadManager.addHandler(this.pageEditHandler);

                this.quickReloadManager.enable();
            },
            teardown: function () {
                // remove all aui flags
                $('#aui-flag-container .aui-flag').each(function () {
                    this.close();
                });
                $('#aui-flag-container').remove();

                this.quickReloadManager.disable();
                this.quickReloadManager.removeHandler(this.commentHandler);
                this.quickReloadManager.removeHandler(this.inlineCommentHandler);
                this.quickReloadManager.removeHandler(this.pageEditHandler);
                this.quickReloadManager = null;
                this.commentHandler.results = [];
                this.inlineCommentHandler.results = [];
                this.pageEditHandler.results = [];


                this.quickReloadCount.setCount(0); // a singleton, so have to reset it
                AJS.unbind('qr:add-new-highlight');
                this.server.restore();
                this.clock.restore();
            }
        });

        test("Page comment single author", function () {
            this.server.respondWith(JSON.stringify(returnPageComment(ADMIN_USER)));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('#aui-flag-container').length === 1);
            ok($('.aui-flag').length === 1, "Page comment flag should show");
            ok($('.aui-flag').hasClass('qr-flag'), "Flag has quick reload custom CSS class");
            ok($('.aui-flag .title').text() === "New comment");
            ok($('.aui-flag .qr-notice-text').text() === '"default comment1"', "Comment text displayed");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by A. D. Ministrator");
            ok($('.aui-flag .qr-notice-show').text() === "Show");
        });

        test("Page comment single author, multiple comments", function () {
            this.server.respondWith(JSON.stringify(returnPageComment([ADMIN_USER, ADMIN_USER])));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 1, "One page comment flag should show");
            ok($('.aui-flag .title').text() === "2 New comments");
            ok($('.aui-flag .aui-avatar').length === 1, "Only one avatar if same author commented");
            ok($('.aui-flag .qr-notice-text').text() === '"default comment2"', "Comment text displayed if single author");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by A. D. Ministrator");
            ok($('.aui-flag .qr-notice-show').text() === "Show");
        });

        test("Page comment single author, then multiple authors", function () {
            this.server.respondWith(JSON.stringify(returnPageComment(ADMIN_USER)));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            this.server.respondWith(JSON.stringify(returnPageComment([TEST1_USER, TEST2_USER])));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 1, "Subsequent page comment notifications should not create a new flag");
            ok($('.aui-flag .title').text() === "3 New comments");
            ok($('.aui-flag .aui-avatar').length === 3, "All user avatars should show");
            ok($('.aui-flag .qr-notice-text').text() === '', "No comment text if multiple comments");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User2 and 2 others");
            ok($('.aui-flag .qr-notice-show').text() === "Show all");
            ok(document.title.indexOf('(3)') !== -1, "Page title updated with number of unread notifications");
        });

        test("Page comment notification with many, many authors", function () {
            var users = [];
            for (var i = 1; i < 13; i++) {
                // create comment users Test User1 to Test User12
                users.push({username: 'test' + i, fullName: 'Test User' + i});
            }
            this.server.respondWith(JSON.stringify(returnPageComment(users)));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 1, "Subsequent page comment notifications should not create a new flag");
            ok($('.aui-flag .title').text() === "12 New comments");
            ok($('.aui-flag .aui-avatar').length === 8, "Maximum of 8 avatars should show");
            ok($('.aui-flag .aui-avatar img').first().attr("title") === "Test User12", "Last user to comment is shown first");
            ok($('.aui-flag .qr-notice-text').text() === '', "No comment text if multiple comments");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User12 and 11 others");
            ok($('.aui-flag .qr-notice-show').text() === "Show all");
        });

        test("Page comment notification user1, then user2 then user1 again", function () {
            this.server.respondWith(JSON.stringify(returnPageComment([TEST1_USER, TEST2_USER])));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            this.server.respondWith(JSON.stringify(returnPageComment(TEST1_USER)));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag .aui-avatar img').first().attr("title") === "Test User1", "Last user to comment is shown first");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User1 and 1 other");
            ok(document.title.indexOf('(3)') !== -1, "Page title updated with number of unread notifications");
        });

        test("Inline comment notifications", function () {
            var response1 = returnPageComment(TEST1_USER);
            response1.comments[0].comment.isInlineComment = true;
            response1.comments[0].comment.parentId = 0;
            var response2 = returnPageComment(TEST1_USER);
            response2.comments[0].comment.isInlineComment = true;
            response2.comments[0].comment.parentId = 0;
            var response3 = returnPageComment([TEST2_USER, ADMIN_USER]);
            response3.comments[0].comment.isInlineComment = true;
            response3.comments[0].comment.parentId = 0;
            response3.comments[1].comment.isInlineComment = true;
            response3.comments[1].comment.parentId = 0;

            this.server.respondWith(JSON.stringify(response1));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 1);
            ok($('.aui-flag .title').text() === "New inline comment");
            ok($('.aui-flag .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag .qr-notice-text').text() === '"<p>default comment1</p>"');
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User1");
            ok($('.aui-flag .qr-notice-show').text() === "Show");
            ok(document.title.indexOf('(1)') !== -1, "Page title updated with number of unread notifications");

            this.server.respondWith(JSON.stringify(response2));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 2, "Each inline comment has own flag");
            ok($('.aui-flag .title').last().text() === "New inline comments");
            ok($('.aui-flag .qr-notice-summary-text').last().text() === "by Test User1");
            ok($('.aui-flag .qr-notice-show').last().text() === "Show");
            ok(document.title.indexOf('(2)') !== -1, "Page title updated with number of unread notifications");

            // Add two inline comments at the same time
            this.server.respondWith(JSON.stringify(response3));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 4, "Each inline comment has own flag");
            ok($('.aui-flag .title').last().text() === "New inline comments");
            ok($('.aui-flag .qr-notice-summary-text').last().text() === "by A. D. Ministrator");
            ok($('.aui-flag .qr-notice-text').last().text() === '"<p>default comment2</p>"');
            ok($('.aui-flag .qr-notice-show').last().text() === "Show");
            ok(document.title.indexOf('(4)') !== -1, "Page title updated with number of unread notifications");
        });

        test("Inline comment - if top level comment created, show it, and ignore any replies. Then show another reply for existing comment", function () {
            var topLevelCommentId = 1234;
            var replyId = 5678;
            var comment = returnPageComment([TEST1_USER, ADMIN_USER]);
            // Top level comment created by TEST1_USER
            comment.comments[0].comment.isInlineComment = true;
            comment.comments[0].comment.parentId = 0;
            comment.comments[0].comment.id = topLevelCommentId;

            // Reply comment created by ADMIN_USER
            comment.comments[1].comment.isInlineComment = true;
            comment.comments[1].comment.parentId = topLevelCommentId;
            comment.comments[1].comment.id = replyId;

            this.server.respondWith(JSON.stringify(comment));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag[aria-hidden="false"]').length === 1);
            ok($('.aui-flag[aria-hidden="false"] .title').text() === "New inline comment");
            ok($('.aui-flag[aria-hidden="false"] .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-text').text() === '"<p>default comment1</p>"');
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-summary-text').text() === "by Test User1");
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-show').text() === "Show");
            ok(document.title.indexOf('(1)') !== -1, "Page title updated with number of unread notifications");

            $('.aui-flag[aria-hidden="false"]:last .aui-icon.icon-close').click();

            // Reply comment created by ADMIN_USER
            var reply = returnPageComment([TEST1_USER]);
            reply.comments[0].comment.isInlineComment = true;
            reply.comments[0].comment.parentId = topLevelCommentId;
            reply.comments[0].comment.id = replyId + 1;

            this.server.respondWith(JSON.stringify(reply));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag[aria-hidden="false"]').length === 1);
            ok($('.aui-flag[aria-hidden="false"] .title').text() === "New inline comment");
            ok($('.aui-flag[aria-hidden="false"] .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-text').text() === '"<p>default comment1</p>"');
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-summary-text').text() === "by Test User1");
            ok($('.aui-flag[aria-hidden="false"] .qr-notice-show').text() === "Show");
            ok(document.title.indexOf('(1)') !== -1, "Page title updated with number of unread notifications");
        });

        test("Inline comment - if no top level comment created notification, show notification for last reply only", function () {
            var topLevelCommentId = 1234;
            var reply1Id = 5678;
            var reply2Id = 9012;
            var comment = returnPageComment([TEST1_USER, ADMIN_USER, TEST2_USER]); // IC, IC Reply, Page comment
            // Top level inline comment created by TEST1_USER
            comment.comments[0].comment.isInlineComment = true;
            comment.comments[0].comment.parentId = topLevelCommentId;
            comment.comments[0].comment.id = reply1Id;

            // Reply inline comment created by ADMIN_USER
            comment.comments[1].comment.isInlineComment = true;
            comment.comments[1].comment.parentId = topLevelCommentId;
            comment.comments[1].comment.id = reply2Id;

            this.server.respondWith(JSON.stringify(comment));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag[aria-hidden="false"]').length === 2);
            ok($('.aui-flag[aria-hidden="false"]:last .title').text() === "New inline comment");
            ok($('.aui-flag[aria-hidden="false"]:last .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-text').text() === '"<p>default comment1</p>"');
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-summary-text').text() === "by Test User1");
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-show').text() === "Show");
            ok(document.title.indexOf('(2)') !== -1, "Page title updated with number of unread notifications");

            // close first flag, which is the reply to inline comment
            $('.aui-flag[aria-hidden="false"]:last .qr-notice-show').click();

            ok($('.aui-flag[aria-hidden="false"]').length === 1);
            ok($('.aui-flag[aria-hidden="false"]:last .title').text() === "New comment");
            ok($('.aui-flag[aria-hidden="false"]:last .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-text').text() === '"default comment3"');
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-summary-text').text() === "by Test User2");
            ok($('.aui-flag[aria-hidden="false"]:last .qr-notice-show').text() === "Show");
            ok(document.title.indexOf('(1)') !== -1, "Page title updated with number of unread notifications");

            $('.aui-flag[aria-hidden="false"]:last .aui-icon.icon-close').click();

            // should close flag when clicking view comment
            ok($('.aui-flag[aria-hidden="false"]').length === 0);
            // should have no paranthese'd number
            ok(document.title.indexOf('(') === -1, "Page title updated with number of unread notifications");
        });

        test("Page edit notification", function () {
            var edit1 = returnPageEdit(TEST1_USER);

            this.server.respondWith(JSON.stringify(edit1));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            ok($('.aui-flag').length === 1);
            ok($('.aui-flag .title').text() === "New page edits");
            ok($('.aui-flag .aui-avatar').length === 1, "User avatar should show");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User1");
            ok($('.aui-flag .qr-notice-show').text() === "Reload page");
            ok(document.title.indexOf('(1)') !== -1, "Page title updated with number of unread notifications");

            var edit2 = returnPageEdit(TEST2_USER);

            this.server.respondWith(JSON.stringify(edit2));
            this.clock.tick(POLL_PERIOD); // Wait 30 seconds
            this.server.respond();

            // Second edit should be displayed in same notification
            ok($('.aui-flag').length === 1);
            ok($('.aui-flag .title').text() === "2 new edits");
            ok($('.aui-flag .aui-avatar').length === 2, "User avatar should show");
            ok($('.aui-flag .qr-notice-summary-text').text() === "by Test User2 and 1 other");
            ok($('.aui-flag .qr-notice-show').text() === "Reload page");
            ok(document.title.indexOf('(2)') !== -1, "Page title updated with number of unread notifications");
        });

        test("Server error disables quick reload polling", function () {
            ok(this.quickReloadManager.isEnabled(), "Quick reload polling is enabled");
            this.server.respondWith([404, {}, ""]);
            this.clock.tick(POLL_PERIOD);
            this.server.respond();
            ok(this.quickReloadManager.isEnabled() === false, "Polling disabled after 404");
            this.quickReloadManager.enable();
            ok(this.quickReloadManager.isEnabled(), "Quick reload polling is enabled");
            this.server.respondWith([500, {}, ""]);
            this.clock.tick(POLL_PERIOD);
            this.server.respond();
            ok(this.quickReloadManager.isEnabled() === false, "Polling disabled after 500");
            this.quickReloadManager.enable();
            ok(this.quickReloadManager.isEnabled(), "Quick reload polling is enabled");
            this.server.respondWith([503, {}, ""]);
            this.clock.tick(POLL_PERIOD);
            this.server.respond();
            ok(this.quickReloadManager.isEnabled() === false, "Polling disabled after 503");
            this.quickReloadManager.enable();
            ok(this.quickReloadManager.isEnabled(), "Quick reload polling is enabled");
            this.server.respondWith([504, {}, ""]);
            this.clock.tick(POLL_PERIOD);
            this.server.respond();
            ok(this.quickReloadManager.isEnabled() === false, "Polling disabled after 504");
            this.quickReloadManager.enable();
        });

        module("Smart timer tests", {
            setup: function () {
                this.clock = sinon.useFakeTimers();
                this.SmartTimer = require('confluence-quick-reload/utils/quick-reload-timer');
            },
            teardown: function () {
                this.clock.restore();
                this.SmartTimer = null;
            }
        });

        test("Default polling time is 30 seconds", function () {
            var callback = sinon.spy();
            var timer = new this.SmartTimer(callback);
            timer.start();
            this.clock.tick(29000);
            ok(!callback.calledOnce, "Callback should not be called before 30 seconds");
            this.clock.tick(1000);
            ok(callback.calledOnce, "Callback should be called after 30 seconds");
            this.clock.tick(29000);
            ok(!callback.calledTwice, "Callback should not be called before 30 seconds");
            this.clock.tick(1000);
            ok(callback.calledTwice, "Callback should be called after 30 seconds");
        });

        test("Can configure polling time", function () {
            var callback = sinon.spy();
            var timer = new this.SmartTimer(callback, {min: 60000});
            timer.start();
            this.clock.tick(59000);
            ok(!callback.calledOnce, "Callback should not be called before 60 seconds");
            this.clock.tick(2000);
            ok(callback.calledOnce, "Callback should be called after 60 seconds");
        });

        test("Can stop and start the timer", function () {
            var callback = sinon.spy();
            var timer = new this.SmartTimer(callback, {min: 60000});
            timer.start();
            this.clock.tick(59000);
            ok(!callback.calledOnce, "Callback should not be called before 60 seconds");
            this.clock.tick(2000);
            ok(callback.calledOnce, "Callback should be called after 60 seconds");
            // stop the timer and elapse 2 minutes
            timer.stop();
            this.clock.tick(120000);
            ok(!callback.calledTwice, "Callback should not be called when timer is stopped");
            // restart the timer
            timer.start();
            this.clock.tick(59000);
            ok(!callback.calledTwice, "Callback should not be called before 60 seconds");
            this.clock.tick(2000);
            ok(callback.calledTwice, "Callback should be called after 60 seconds");
            // stop the timer and then force an update
            timer.stop();
            timer.start(true);
            ok(callback.calledThrice);
        });

        test("Timer backs off after inactivity period", function () {
            var callback = sinon.spy();
            var timer = new this.SmartTimer(callback, {min: 10000, inactivity: 10000, max: 60000});
            timer.start();
            // due to backoff algorithm, callback will be called after 10s, 20s, 40s
            this.clock.tick(9000);
            ok(callback.callCount !== 1);
            this.clock.tick(1000);
            ok(callback.callCount === 1);
            this.clock.tick(19000);
            ok(callback.callCount !== 2);
            this.clock.tick(1000);
            ok(callback.callCount === 2);
            this.clock.tick(39000);
            ok(callback.callCount !== 3);
            this.clock.tick(1000);
            ok(callback.callCount === 3);
            // next one would be at after 60s because the max is 60s
            this.clock.tick(59000);
            ok(callback.callCount !== 4);
            this.clock.tick(1000);
            ok(callback.callCount === 4);
            // perform a user action to reset the interval back to min of 10000
            $('body').click();
            ok(callback.callCount === 5);
            this.clock.tick(9000);
            ok(callback.callCount !== 6);
            this.clock.tick(1000);
            ok(callback.callCount === 6);
        });

        var _commentId = 0;

        /**
         * Create mocked page comments, one comment per user with the body default comment1, default comment2, etc
         * @param users a single or list of users to create mocked comments for
         * @returns {object} page comment mocked object
         */
        function returnPageComment(users) {
            users = Array.isArray(users) ? users : [users];

            var comments = [];
            var counter = 0;

            for (var i = 0; i < users.length; i++) {
                comments.push({
                    "commenter": {
                        "userName": users[i].username,
                        "displayName": users[i].fullName,
                        "profilePicture": {
                            "path": "/confluence/images/icons/profilepics/default.png"
                        }
                    },
                    "comment": {
                        "id": (++_commentId),
                        "html": "<p>default comment" + (++counter) + "</p>",
                        "asyncRenderSafe": true,
                        "ownerId": parseInt(Math.random() * 10000),
                        "parentId": (++_commentId + 99999),
                        "isInlineComment": false
                    }
                });
            }

            return {
                "comments": comments,
                "page": "",
                "time": (new Date()).getTime() // get the current time
            };
        }

        /**
         * Create mocked page edit. Each REST response only contains the most recent user to edit the page
         * @param user
         * @returns {object} REST response with a page edit by user
         */
        function returnPageEdit(user) {
            return {
                "comments": [],
                "page": {
                    "editor": {
                        "userName": user.username,
                        "displayName": user.fullName,
                        "profilePicture": {
                            "path": "/confluence/images/icons/profilepics/default.png"
                        }
                    }
                },
                "time": (new Date()).getTime() // get the current time
            };
        }

    });
})($);