define('confluence-quick-reload/main/quick-reload-manager', [
    'underscore',
    'jquery',
    'ajs',
    'confluence-quick-reload/utils/quick-reload-timer'
],
/**
 * Quick reload library for informing users of new comments, edits and inline comments when editing a page
 *
 * @tainted AJS.Meta
 * @exports confluence-quick-reload/main/quick-reload-manager
 */
function(
    _,
    $,
    AJS,
    SmartTimer
) {
    "use strict";

    function QuickReloadManager() {
        this._isEnabled = false;
        this.handlers = [];
        this.lastFetchTime = $('meta[name="confluence-request-time"]').attr("content") || (new Date()).getTime();
        this._timer = null;

        _.bindAll(this, 'addHandler', 'removeHandler', 'update', 'enable', 'disable', '_onUpdateSuccess', '_onUpdateError');
    }

    /**
     * Adds a handler that processes quick reload data and renders it inside the application. Each handler must be unique
     * @param {Object} handler a quick reload handler object
     */
    QuickReloadManager.prototype.addHandler = function(handler) {
        var alreadyRegistered = false;
        for (var i = 0; i < this.handlers.length; i++) {
            if (this.handlers[i] === handler) {
                alreadyRegistered = true;
            }
        }
        if (alreadyRegistered !== true) {
            this.handlers.push(handler);
        }
    };

    /**
     * Removes a handler that processes quick reload data
     * @param {Object} handler a quick reload handler object
     */
    QuickReloadManager.prototype.removeHandler = function(handler) {
        if (handler) {
            for (var i = 0; i < this.handlers.length; i++) {
                if (this.handlers[i] === handler) {
                    this.handlers.splice(i, 1);
                    break;
                }
            }
        }
    };

    /**
     * Calls the QuickLoad service, finds any new updates since the last request and shows the notice dialog.
     */
    QuickReloadManager.prototype.update = function() {
        // This is assuming we never come back to the view page without refreshing
        // Ideally this we respond to an event instead of manually checking
        if (_isFullScreenEdit()) {
            this.disable();
            return;
        }

        $.ajax({
            type: "GET",
            url: AJS.contextPath() + "/rest/quickreload/latest/" + AJS.Meta.get("page-id") + "?since=" + encodeURIComponent(this.lastFetchTime),
            dataType: "json"
        }).done(this._onUpdateSuccess).fail(this._onUpdateError);
    };

    /**
     * Enables quick reload polling for new page edits, comments and inline comments
     */
    QuickReloadManager.prototype.enable = function() {
        if (this._timer === null) {
            this._timer = new SmartTimer(this.update);
        }
        this._timer.start();
        this._isEnabled = true;
    };

    /**
     * Disabled quick reload polling for new page edits, comments and inline comments
     */
    QuickReloadManager.prototype.disable = function() {
        if (this._timer !== null) {
            this._timer.stop();
        }
        this._isEnabled = false;
    };

    /**
     * Returns a value informing the user if quick polling is enabled
     * @returns {Boolean} whether quick reload polling is enabled
     */
    QuickReloadManager.prototype.isEnabled = function() {
        return this._isEnabled;
    };

    /**
     * Callback on successful REST call to quick reload resource. It passes the REST data to each quick reload
     * handler for processing and rendering
     * @param data
     * @param textStatus
     * @param jqXHR
     * @private
     */
    QuickReloadManager.prototype._onUpdateSuccess = function(data, textStatus, jqXHR) {
        if (jqXHR.status === 204) {
            return;
        }
        this.lastFetchTime = data.time;
        _.map(this.handlers, function (handler) {
            var newResults = data[handler.property];
            if (!Array.isArray(newResults)) {
                newResults = [newResults];
            }
            newResults = handler.filterNewResults(handler.results, newResults);

            if (newResults.length > 0) {
                handler.results = handler.results.concat(newResults);
                handler.render(newResults);
            }
        }, this);
    };

    /**
     * Error handler for quick reload REST resource. Disables quick reload polling if an error occurs
     * @param request
     * @private
     */
    QuickReloadManager.prototype._onUpdateError = function(request) {

        // Quick reload plugin will get disabled if received one of the following status codes:
        var blackListedStatusCodes = {
            404 : "not found - the plugin has been probably been removed or disabled from Confluence",
            500 : "generic server error",
            503 : "service unavailable",
            504 : "gateway timeout"
        };

        request = request || {};
        var blackListedError = blackListedStatusCodes[request.status];
        if (blackListedError){
            this.disable();
            AJS.log('Quick comment reload plugin has been disabled in this page due to a server error: "' + blackListedError + '". Please refresh the page to get it back.');
        }
    };

    // Is there a better way to check this, but allow comment editor?
    function _isFullScreenEdit() {
        return $('body').hasClass('contenteditor');
    }

    // return a singleton
    return (new QuickReloadManager());
});
