define('confluence-quick-reload/handlers/quick-reload-inline-comments', [
    'underscore',
    'jquery',
    'ajs',
    'confluence/flag',
    'confluence-quick-reload/utils/quick-reload-count'
],
/**
 * Processes and displays inline comment notifications coming from quick reload plugin
 *
 * @tainted QuickReload.Templates
 * @tainted AJS.Meta
 * @exports confluence-quick-reload/handlers/quick-reload-inline-comments
 */
function(
    _,
    $,
    AJS,
    Flag,
    QuickReloadCount
) {
    "use strict";

    return {
        results: [],

        property: 'comments',

        /**
         * Returns all new inline comments (top level or replies) not created by the current user. If a comment is
         * created and it has replies, only show the notification for the top level comment. If there is no new comment
         * but multiple replies for a single comment, only show the last reply
         * @param currentResults currently displayed results on the page, updated when notifications are closed
         * @param newResults latest quick reload data received from REST
         * @returns an array of all inline comments not created by the user
         */
        filterNewResults: function(currentResults, newResults) {
            newResults = _.clone(newResults);

            // remove non-inline comments and comments created by current user
            newResults = _.filter(newResults, function(commentAndUser) {
                return commentAndUser.comment.isInlineComment &&
                       commentAndUser.commenter.userName !== AJS.Meta.get("remote-user");
            });

            // we need to filter out any notifications with the same parentId as an existing notification because we
            // only care about the first comment or reply in a given thread
            if (newResults.length > 0) {
                var displayedCommentIds = _.map(currentResults, _getTopLevelCommentId); // already been displayed

                // in the newest payload, we only want to most recent reply in a thread. if no replies exist, show the
                // parent comment instead
                var newTopLevelComments = _.filter(newResults, _isTopLevelComment);
                var newReplyComments = _.filter(newResults, function(commentAndUser) { return !_isTopLevelComment(commentAndUser); });
                var newTopLevelCommentIds = _.map(newTopLevelComments, _getTopLevelCommentId);
                var newReplyParentCommentIds = _.map(newReplyComments, _getTopLevelCommentId);

                var replyWithoutTopLevelCommentParentIds = _.difference(newReplyParentCommentIds, newTopLevelCommentIds);

                var uniqueReplyThreadIds = [];
                var firstReplyOnCommentThread = _.filter(newReplyComments, function(commentAndUser) {
                    if (replyWithoutTopLevelCommentParentIds.indexOf(commentAndUser.comment.parentId) !== -1) {
                        // only a unique reply if no other reply on thread has been encountered before
                        if (uniqueReplyThreadIds.indexOf(commentAndUser.comment.parentId) === -1) {
                            uniqueReplyThreadIds.push(commentAndUser.comment.parentId);
                            return true;
                        }
                    }
                    return false;
                });

                var uniqueResults = newTopLevelComments.concat(firstReplyOnCommentThread);

                newResults = _.filter(uniqueResults, function(commentAndUser) {
                    return displayedCommentIds.indexOf(_getTopLevelCommentId(commentAndUser)) === -1;
                });
            }

            return newResults;
        },

        /**
         * Function to call when desiring to render new inline comment flags on the screen
         * @param newResults the results retrieved from the latest polling attempt to quick reload
         */
        render: function(newResults) {
            QuickReloadCount.setCount(QuickReloadCount.getCount() + newResults.length);

            _.each(newResults, function (commentAndUser) {
                var commentId = _getTopLevelCommentId(commentAndUser);

                var showCallback = function () {
                    AJS.trigger("qr:show-new-thread", commentId);
                };

                var reloadCallback = function() {
                    var canonicalUrl = AJS.Meta.Links.canonical();
                    var separator = canonicalUrl.indexOf('?') === -1 ? "?" : "&";
                    window.location = canonicalUrl + separator + "focusedCommentId=" + commentId + "#comment-" + commentId;
                };

                var closeCallback = function() {
                    // The comment corresponding to the closed notification needs to be removed from results
                    this.results = _.filter(this.results, function(commentAndUser) {
                        return _getTopLevelCommentId(commentAndUser) !== commentId;
                    });
                    QuickReloadCount.setCount(QuickReloadCount.getCount() - 1);
                };
                closeCallback = _.bind(closeCallback, this);

                var commentCount = this.results.length;

                if (_isTopLevelComment(commentAndUser)) {
                    var deferred = new $.Deferred();
                    deferred.fail(function(commentAndUser) {
                        _renderFlag(commentAndUser, true, reloadCallback, closeCallback, commentCount);
                    });
                    deferred.done(function(commentAndUser) {
                        _renderFlag(commentAndUser, false, showCallback, closeCallback, commentCount);
                    });
                    AJS.trigger("qr:add-new-highlight", [commentAndUser, deferred]);
                } else { // this is a new reply whose the threaded notification is not shown yet
                    _renderFlag(commentAndUser, false, showCallback, closeCallback, commentCount);
                }
            }, this);
        }

    };

    /**
     * Renders a visual flag for the inline comment notification
     * @param commentAndUser
     * @param reloadRequired
     * @param showCallback
     * @param closeCallback
     * @param commentCount the total inline comment count not yet show on the page
     * @private
     */
    function _renderFlag(commentAndUser, reloadRequired, showCallback, closeCallback, commentCount) {
        var flagDefaults = {
            close: 'manual',
            type: 'info',
            extraClasses: 'qr-flag',
            fifo: true,
            stack: 'quickreload'
        };

        var commenter = [commentAndUser.commenter];
        var body = QuickReload.Templates.inlineComment({
            user: commenter,
            noticeText: _getCommentText(commentAndUser),
            reloadRequired: reloadRequired
        });

        var title = (commentCount > 1) ? AJS.I18n.getText('quick.reload.newinlinecomments')
                                       : AJS.I18n.getText('quick.reload.newinlinecomment');

        var flag = new Flag($.extend({}, {
            title: title,
            body: body
        }, flagDefaults));

        $(flag).find('.qr-notice-show').click(showCallback);
        // only close the flag on clicking show link if no reload required
        if (!reloadRequired) {
            $(flag).find('.qr-notice-show').click(flag.close);
        }

        $(flag).on('aui-flag-close', closeCallback);
    }

    /**
     * Returns the inline comment text
     * @param commentAndUser object containing comment and commenter properties
     * @returns String
     * @private
     */
    function _getCommentText(commentAndUser) {
        return commentAndUser && commentAndUser.comment && $("<div>").text(commentAndUser.comment.html).text();
    }

    /**
     * Determines if an inline comment is a top level comment. If it is not a top level comment, it is a inline comment reply
     * @param comment
     * @returns {boolean}
     * @private
     */
    function _isTopLevelComment(commentAndUser) {
        return commentAndUser.comment.parentId === 0;
    }

    /**
     * Given an inline comment, get the top level comment id. If the comment is a top level comment, the comment id is
     * returned. If it is an inline comment reply, its parent id is returned
     * @param commentAndUser
     * @returns Number top level comment id
     * @private
     */
    function _getTopLevelCommentId(commentAndUser) {
        return _isTopLevelComment(commentAndUser) ? commentAndUser.comment.id : commentAndUser.comment.parentId;
    }
});
