define('confluence-paste/clipboard-access', [
    'jquery',
    'ajs',
    'window',
    'tinymce'
], function(
    $,
    AJS,
    window,
    tinymce
) {
    "use strict";

    var ClipboardAccess = function() {

        // firefox and IE11 use a mutation observer to detect changes to the DOM and grab the img node that is inserted into the content editable
        function registerMutationObserver(editor){
            var observer = new MutationObserver(function (mutations) {
                for (var i = 0; i < mutations.length; i++) {
                    var mutation = mutations[i];

                    if (mutation.addedNodes.length) {
                        for (var x = 0; x < mutation.addedNodes.length; x++) {
                            var node = mutation.addedNodes[x];

                            if (node.localName === "img") {
                                window.setTimeout(pasteHandler.pasteFromDataURL, 0);
                                return;
                            }
                        }
                    }
                }
            });
            observer.observe(editor.getDoc(), { childList: true, subtree: true });
        }

        // chrome uses a paste handler and grabs the image data out of the paste event
        function registerOnPasteListener(ed){
            ed.onPaste.add(function (editor, e) {
                if (pasteHandler.onPaste(e)) {
                    return true;
                } else {
                    tinymce.dom.Event.cancel(e);
                    return false;
                }
            });
        }

        var pasteHandler = {
            initializeEditor: function() {
                if (AJS && AJS.Rte && AJS.Rte.BootstrapManager) {
                    AJS.Rte.BootstrapManager.addOnInitCallback(function () {

                        var ed = AJS.Rte.getEditor();
                        if (ed) {
                            //register observer to listen for img nodes inserted into the DOM, firefox and IE still don't
                            // expose enough information to grab the image from the clipboard
                            if (window.MutationObserver && !$.browser.webkit) {
                                registerMutationObserver(ed);
                            }
                            else {
                                registerOnPasteListener(ed);
                            }
                        }
                        else {
                            AJS.log("Could not getEditor from adapter");
                        }
                    });
                }
                else {
                    AJS.log("Could not initialize pasteHandler on Confluence.Editor");
                }
            },

            /**
             * searches the wysiwyg text area for embedded image data URLs and attaches them to the document
             * via the upload, then removes the data URI
             */
            pasteFromDataURL: function() {
                //firefox 4.0 inserts images as data urls
                //do not count the images have data url built by macro placeholder in editor with editor-inline-macro class
                var embeddedImgs = $('img[src^="data:image"]:not(".editor-inline-macro")', tinymce.activeEditor.getDoc());
                $.each(embeddedImgs, function() {
                    var imgNode = this;
                    var data = imgNode.src.substring(22, imgNode.src.length);

                    // calculate the unencoded image size, accounting for potential padding.
                    var size = (data.length * 3) / 4;
                    //count padded bytes, last two characters might be padding with the '=' char
                    var paddedBytes = 0;
                    for (var len = data.length, i = len - 1; data[i] == "="; i--) {
                        paddedBytes++;
                    }

                    AJS.Confluence.Uploader.upload([
                        pasteHandler.createUploadFile(data, size - paddedBytes, "base64")
                    ]);
                });

                //remove all embedded images
                embeddedImgs.remove();
            },

            /**
             * onPaste handler, Either uses webkits clipboardData object on the paste event, or interprets
             * the data uri that has been embedded in the page by firefox.
             */
            onPaste: function(event) {

                if (event && event.clipboardData) {
                    // webkit exposes a clipboardData object on the paste event
                    return pasteHandler.pasteFromEvent(event);
                }
                return true;
            },

            showClipboardAccessError: function() {
                new AJS.Dialog({
                    width: 400,
                    closeOnOutsideClick: true
                })
                        .addPanel("alert", $("<div />", { text: AJS.I18n.getText("clipboardaccess.browser.not.supported") }))
                        .addButton(AJS.I18n.getText("clipboardaccess.ok"), function(dialog) { dialog.remove(); })
                        .show()
                        .updateHeight();
            },

            /**
             * chrome supports clipboard data on the paste event, can read the image straight from it.
             * @param event
             */
            pasteFromEvent : function(event) {
                if (event && event.clipboardData) {

                    if (event.clipboardData.items) {
                        var dataTransferItem = event.clipboardData.items[0];
                        if (!dataTransferItem || dataTransferItem.kind != "file") {
                            return true;
                        }

                        var dataFile = dataTransferItem.getAsFile();
                        var uploadFile = pasteHandler.createUploadFile(dataFile, dataFile.size);
                        AJS.Confluence.Uploader.upload([uploadFile]);
                        return false;
                    } else {
                        // Safari (and possibly other browsers) will
                        // allow the image to be pasted, but the data
                        // URI is not placed as the image's `src`
                        // attribute. It replaces it with a custom
                        // protocol, which can't be accessed via JavaScript.
                        window.setTimeout(function(me) {
                            var webkitFakeUrlImages = $("img[src^='webkit-fake-url://']", tinymce.activeEditor.getDoc());
                            if (webkitFakeUrlImages.length) {
                                webkitFakeUrlImages.remove();
                                me.showClipboardAccessError();
                            }
                        }, 0, this);
                    }
                }
                return true;
            },

            /**
             * creates an upload file with an optionally specified encoding
             * @param data - the data to be uploaded, either as binary or base64 encoded
             * @param sz - the size of the decoded image
             * @param encoding - optional encoding, only base64 is currently supported.
             */
            createUploadFile: function(data, sz, encoding) {
                var curDate = new Date();
                var month = curDate.getMonth() + 1;
                return {
                    fileName: "image" + curDate.getFullYear() + "-" + month + "-" + curDate.getDate() + "_" +
                    curDate.getHours() + "-" + curDate.getMinutes() + "-" + curDate.getSeconds() + ".png",
                    fileSize: sz,
                    slice : function(offset, length) {
                        return data.slice(offset, offset + length);
                    },
                    getAsBinary : function() {
                        return data;
                    },
                    getData : function() {
                        return data;
                    },
                    encoding : encoding
                };
            }
        };

        pasteHandler.initializeEditor($);
        return pasteHandler;
    };

    return ClipboardAccess;
});

require('confluence/module-exporter').safeRequire('confluence-paste/clipboard-access', function(ClipboardAccess) {
    var AJS = require('ajs');

    AJS.toInit(ClipboardAccess);
});