define('confluence-paste/autoconvert/tinymce-autoconvert', [
    'tinymce',
    'ajs',
    'jquery',
    'document',
    'confluence/api/constants'
], function(
    tinymce,
    AJS,
    $,
    document,
    CONSTANTS
) {
    "use strict";

    /***********
     * Constructor
     */
    var Autoconvert = function (ed){
        ed.onUndo.add(this.cleanLinks);
        this.editor = ed;
        this.weightedHandlers = [];
        this.handlers = [];
    };

    /***********
     * Instance Methods
     */
    Autoconvert.prototype = {
        addHandler: function (handler, weight){
            this.weightedHandlers.push({handler: handler, weight:weight || 0});
            this.weightedHandlers.sort(function(a,b) { return a.weight - b.weight; });
            this.handlers = [];
            var len = this.weightedHandlers.length;
            for (var i = 0; i < len; i++) {
                this.handlers.push(this.weightedHandlers[i].handler);
            }
        },
        cleanLinks: function(ed, level){
            $('a.atlassian-autoconvert', ed.getDoc()).removeClass('atlassian-autoconvert');
            return true;
        },
        callHandlers: function (n){
            var index = 0;
            var self = this;
            var originalNode = $(n);
            var node = $(n).clone();
            var rawHref = node.attr('href');
            var uri = parseUri(hackAtSymbols(rawHref));

            /**
             * Runs handlers one at a time, this method acts as a continuation and is passed to each handler.
             */
            function run(newNode){
                if (newNode && (newNode.length !== 1 || !newNode.attr('href'))) {
                    return handlersFinished(newNode);
                }

                if (newNode) {
                    node = newNode;
                    index = 0;
                    uri = parseUri(node.attr('href'));
                    callHandler(self.handlers[index], uri, node);
                } else {
                    index++;
                    if (index < self.handlers.length) {
                        callHandler(self.handlers[index], uri, node);
                    } else {
                        return handlersFinished(node);
                    }
                }
            }

            /**
             * This method ensures that a single handler can't break the event loop by never calling the continuation or by
             * calling it multiple times.
             */
            function callHandler(callback, uri, node){
                var modifiableRun = run;
                var timer;
                var continuation;

                continuation = function(){
                    var localIndex = index;
                    clearTimeout(timer);

                    //actually call the next iteration of run
                    modifiableRun.apply(self, arguments);

                    //ensure a single handler can't call continuation multiple times
                    modifiableRun = function(){
                        AJS.log('Autoconvert: Callback #' + localIndex + ' called the continuation multiple times.');
                    };
                };
                timer = setTimeout(function(){
                    AJS.log("Autoconvert: Callback #" + index + " failed to call continuation in time. If there is no subsequent log, the handler is improperly implemented and should be uninstalled.");
                    modifiableRun = function(){
                        AJS.log("Autoconvert: Callback #" + index + " did eventualy return. Probably a slow async call.");
                    };
                    run();
                }, tinymce.plugins.Autoconvert.callbackTimeout);

                //Call the callback passing it a modified version of the continuation that allows us to prevent continuing after a timeout.
                callback(uri, node, continuation);
            }

            function handlersFinished(newNode){
                self.editor.undoManager.beforeChange();
                var selection = self.editor.selection;
                var bm = self.editor.selection.getBookmark();


                //replace node that's actually in the editor with the newly modified node
                originalNode.replaceWith(newNode);

                selection.moveToBookmark(bm);
                self.editor.undoManager.add();


            }

            if (this.handlers.length !== 0) {
                callHandler(this.handlers[0], uri, node);
            }
        }
    };

    /***********
     * Static properties
     */
    Autoconvert.callbackTimeout = 5000;

    /***********
     * Static methods
     */
    Autoconvert.convertMacroToDom = function(macro, success, error){
        var conversionData = {
            macro : macro,
            contentId :  AJS.Meta.get("content-id")
        };
        $.ajax( {
            type : "POST",
            contentType : "application/json; charset=utf-8",
            url : CONSTANTS.CONTEXT_PATH + "/rest/tinymce/1/macro/placeholder",
            data : $.toJSON(conversionData),
            dataType : "text", // if this is switched back to json be sure to use "text json". See CONFDEV-4799 for details.
            success : success,
            error : error,
            timeout: 45000
        });
    };

    Autoconvert.convertMacroToDomViaWikiText = function(macro, success, error){
        var wikiTxt = ['{', macro.name];
        var firstParam = true;
        if (macro.defaultParameterValue){
            wikiTxt.push(':', macro.defaultParameterValue);
            firstParam = false;
        }
        for (var key in macro.params){
            if (firstParam){
                wikiTxt.push(':');
                firstParam = false;
            }
            else{
                wikiTxt.push('|');
            }
            wikiTxt.push(key,'=',macro.params[key]);
        }
        wikiTxt.push('}');

        Autoconvert.getHtmlFromWikiMarkup(
            wikiTxt.join(''),
            function(data){
                success($(data)[0]);
            },
            function(){
                error();
            }
        );

    };

    Autoconvert.getHtmlFromWikiMarkup = function(text, success, error){
        var conversionData = {
            wiki : text,
            entityId :  AJS.Meta.get("content-id"),
            spaceKey : AJS.Meta.get("space-key")
        };
        $.ajax( {
            type : "POST",
            contentType : "application/json; charset=utf-8",
            url : CONSTANTS.CONTEXT_PATH + "/rest/tinymce/1/wikixhtmlconverter",
            data : $.toJSON(conversionData),
            dataType : "text", // if this is switched back to json be sure to use "text json". See CONFDEV-4799 for details.
            success : success,
            error : error,
            timeout: 45000
        });
    };

    /***********
     * Private methods
     */

    // parseUri 1.2.2
    // (c) Steven Levithan <stevenlevithan.com>
    // MIT License
    function parseUri (str) {
        var o   = parseUri.options;
        var m   = o.parser[o.strictMode ? "strict" : "loose"].exec(str);
        var uri = {};
        var i   = 14;

        while (i--) { uri[o.key[i]] = m[i] || ""; }

        uri[o.q.name] = {};
        uri[o.key[12]].replace(o.q.parser, function ($0, $1, $2) {
            if ($1) { uri[o.q.name][$1] = $2; }
        });

        return uri;
    }

    parseUri.options = {
        strictMode: false,
        key: ["source","protocol","authority","userInfo","user","password","host","port","relative","path","directory","file","query","anchor"],
        q:   {
            name:   "queryKey",
            parser: /(?:^|&)([^&=]*)=?([^&]*)/g
        },
        parser: {
            strict: /^(?:([^:\/?#]+):)?(?:\/\/((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?))?((((?:[^?#\/]*\/)*)([^?#]*))(?:\?([^#]*))?(?:#(.*))?)/,
            loose:  /^(?:(?![^:@]+:[^:@\/]*@)([^:\/?#.]+):)?(?:\/\/)?((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/
        }
    };


    /**
     * parseUri has a bug whereby it deems everything up to the first @ to be part of the "authority"
     * section of the Uri. Even in strict mode.  We should fix this properly, but for now, we're just
     * replacing the first @ with an encoded one.
     */
    var hackAtSymbols = function(rawHref) {
        return rawHref.replace(/^(.*?\/\/[^/]*?\/.*?)(@)/, "$1%40");
    };

    // ======== TinyMCE plugin creation ===========================================================
    // Since this process relies on tinymce.plugins.AutoConvert already existing these two separate
    // concerns need to be included in the same source file. Otherwise you cannot create a new
    // tinymce.plugins.Autoconvert until TinyMCE is initialised, yet you cannot initialise this
    // TinyMCE plugin until tinymce.plugins.Autoconvert.autoConvert exists.

    var autoConvertPlugin = {
        init: function (ed) {
            var autoConvert = new Autoconvert(ed);

            // Need to refactored out for not to have this assignment here
            tinymce.plugins.Autoconvert.autoConvert = autoConvert;

            function postPasteHandler(e, pl, o) {
                var anchor = $(o.node).children('a');
                if (anchor.length === 1 && anchor[0].tagName === "A" && anchor.text() === anchor.attr('href')) {
                    anchor.attr('class', 'atlassian-autoconvert');
                    setTimeout(function () {
                        var anchor = $('a.atlassian-autoconvert', $(ed.selection.getNode().ownerDocument)).removeClass('atlassian-autoconvert');
                        autoConvert.callHandlers(anchor);
                    }, 0);
                }
            }

            ed.onInit.add(function () {
                $(document).bind('postPaste', postPasteHandler);

                ed.addCommand("addAutoconverter", function (ui, options) {
                    autoConvert.addHandler(options);
                });
            });

            ed.onRemove.add(function () {
                $(document).unbind('postPaste', postPasteHandler);
            });
        },

        getInfo: function() {
            return {
                longname: 'Atlassian Autoconvert',
                author: 'Atlassian',
                authorurl: 'http://www.atlassian.com',
                infourl: 'http://www.atlassian.com',
                version: '1.0'
            };
        }
    };

    return {
        autoConverter: Autoconvert,
        plugin: autoConvertPlugin
    };
});

require('confluence/module-exporter').safeRequire('confluence-paste/autoconvert/tinymce-autoconvert', function(AutoconvertPlugin) {
    var tinymce = require('tinymce');

    tinymce.plugins.Autoconvert = AutoconvertPlugin.autoConverter;

    tinymce.create('tinymce.plugins.AutoconvertPlugin', AutoconvertPlugin.plugin);
    tinymce.PluginManager.add('autoconvert', tinymce.plugins.AutoconvertPlugin);

});
