define('confluence/ob/modules/tutorial/editor/editor-sequence', [
    'jquery',
    'ajs',
    'confluence/ob/common/promise',
    'confluence/ob/modules/tutorial/editor/blueprints',
    'confluence/ob/modules/tutorial/editor/tour',
    'confluence/ob/modules/tutorial/editor/editor'
], function ($, AJS, Promise, Blueprint, EditorTour, Editor) {
    'use strict';

    var IntroEditorSequence = function(options) {};

    IntroEditorSequence.prototype.init = function(container, analytics) {
        var loadingIndicator = Confluence.PageLoadingIndicator($("body"));

        if (!Editor.isCreatingNewPage(window)) {
            return Promise(function() {}); // do nothing
        }

        return Promise(function(resolve, reject) {
            function showWelcomeDialog() {
                loadingIndicator.show();

                // Remove the CTA for the save button, which will be restored after all of the steps or skip the editor tutorial
                $('#rte-button-publish').removeClass('aui-button-primary');

                var $content = $(Confluence.Templates.OB.editorWelcome()).appendTo('body');

                $content.find('#onboarding-get-started').click(function () {
                    analytics.pushEvent(IntroEditorSequence.EVENTS.LET_GET_STARTED);
                    hideWelcomeDialog($content, onGetStartedClicked);
                });
                $content.find('#onboarding-skip-editor-tutorial').click(function () {
                    $('#rte-button-publish').addClass('aui-button-primary');
                    analytics.pushEvent(IntroEditorSequence.EVENTS.I_KNOW_THIS_ALREADY);
                    hideWelcomeDialog($content);
                    reject();
                });
            }

            function hideWelcomeDialog($content, callback) {
                $content.fadeOut('fast', $.isFunction(callback) ? callback : $.noop);
                loadingIndicator.hide();
            }

            /**
             * We don't want to go back one step when user click on 'Close' link, so
             * we will hide the current one and create a fake Close link to redirect user to Dashboard
             */
            function fakeCloseButton() {
                var $newHideButton = $(Confluence.Templates.OB.editorCloseButton());
                $newHideButton.on('click', function(e) {
                    e.preventDefault(); // if not, it will call docreatepage.action
                    analytics.pushEvent(IntroEditorSequence.EVENTS.EDITOR_CLOSE_BUTTON_CLICK);
                    reject();
                    window.location.href = [window.location.protocol, '//', window.location.host, AJS.contextPath() + '/dashboard.action'].join('');
                });
                $newHideButton.insertAfter($('#rte-button-cancel').hide());
            }

            function onGetStartedClicked() {
                EditorTour.initiate(analytics, function() {
                    resolve();
                });

                Blueprint.createOptionsPanel().then(
                        function($panel) {
                            Blueprint.showOptionPanel(function() {
                                $panel.find('.onboarding-editor-template-option:first').trigger('click');
                            });

                            $panel.on('click', '.footer-panel .next', function() {
                                $(this).prop('disabled', true);
                                analytics.pushEvent(IntroEditorSequence.EVENTS.BLUEPRINTS_NEXT_BUTTON_CLICK);
                                Blueprint.hideOptionPanel(function() {
                                    EditorTour.next();
                                });
                            });

                            $('#rte-button-publish').on('click', function() {
                                var contentTitle = $('#content-title').val();
                                if (contentTitle) {
                                    sessionStorage.setItem('selected-blueprint', Blueprint.getSelectedTemplateId());
                                }

                                analytics.pushEvent(IntroEditorSequence.EVENTS.EDITOR_SAVE_BUTTON_CLICK);
                                resolve();
                            });
                        },
                        function() {
                            EditorTour.setStep(['pageTitle', 'dragDrop', 'save']);
                            EditorTour.next();
                        }
                );

                $('#rte-button-preview').click(function() {
                    analytics.pushEvent(IntroEditorSequence.EVENTS.EDITOR_PREVIEW_BUTTON_CLICK);
                    EditorTour.pause();
                    if (Blueprint.isOptionPanelVisible()) {
                        Blueprint.hideOptionPanel();

                        var onClick = function() {
                            EditorTour.resume();
                            Blueprint.showOptionPanel();
                            $(this).off('click', onClick);
                        };
                        $('#rte-button-edit').on('click', onClick);
                    }
                });

                $('#rte-button-edit').click(function() {
                    analytics.pushEvent(IntroEditorSequence.EVENTS.EDITOR_EDIT_BUTTON_CLICK);
                    EditorTour.resume();
                });
            }

            showWelcomeDialog();
            fakeCloseButton();
            Blueprint.initiate(analytics);
        });
    };

    IntroEditorSequence.EVENTS = {
        I_KNOW_THIS_ALREADY: "i.know.this.already",
        LET_GET_STARTED: "let.get.started",
        BLUEPRINTS_NEXT_BUTTON_CLICK: "blueprints.next.button.click",
        EDITOR_SAVE_BUTTON_CLICK: "editor.save.button.click",
        EDITOR_CLOSE_BUTTON_CLICK: "editor.close.button.click",
        EDITOR_PREVIEW_BUTTON_CLICK: "editor.preview.button.click",
        EDITOR_EDIT_BUTTON_CLICK: "editor.edit.button.click"
    };

    return IntroEditorSequence;
});