(function() {
    
    /**
     * Create a new BinarySearchTree which uses the supplied comparator
     * 
     * @param comparator a function which will accept two arguments and compare them 
     * returning a negative integer, zero, or a positive integer as the first argument 
     * is less than, equal to, or greater than the second. 
     */
    function BinarySearchTree(comparator) {
        if (typeof comparator != "function") {
            throw new Error("The BinarySearchTree requires a comparator function.");
        }
        
        this._root = null;
        this._comparator = comparator;
    }

    BinarySearchTree.prototype = {

        constructor: BinarySearchTree,
    
        /**
         * Add the supplied value to the tree. If the value already exists then
         * no changes will be made.
         */
        add: function(value) {
            var node = {
                value: value,
                left: null,
                right: null
            };
            var current = this._root;

            // special case - tree is empty
            if (this._root === null) {
                this._root = node;
                return;
            } 
            
            while(true) {
                var compare = this._comparator(value, current.value);
                
                if (compare < 0) {
                    if (current.left === null) {
                        current.left = node;
                        break;
                    } else {
                        current = current.left;
                    }
                } else if (compare > 0) {
                    if (current.right === null){
                        current.right = node;
                        break;
                    } else {
                        current = current.right;
                    }       
                } else {
                    // if the new value is equal to the current one, just ignore the addition
                    break;
                }
            } 
        },
    
        contains: function(value) {
            var current = this._root

            while (current) {
                var compare = this._comparator(value, current.value);
                if (compare < 0) {
                    current = current.left;
                } else if (compare > 0) {
                    current = current.right;
                } else {
                    return true;
                }                
            }

            return false;            
        },

        /**
         * @returns the number of items in the tree
         */
        size: function() {
            var length = 0;

            this.traverse(function(node) {
                length++;
            });

            return length;
        },
        
        /**
         * @returns an array containing the values in the tree
         */
        toArray: function() {
            var result = new Array();

            this.traverse(function(node) {
                result.push(node.value);
            });

            return result;
        },
        
        /*
         * In order traversal performing the supplied operation at each node.
         * @param operation a function taking a single argument that will be the node visited.
         */
        traverse: function(operation) {
            function inOrder(node){
                if (node) {
                    // traverse to the left
                    if (node.left !== null){
                        inOrder(node.left);
                    }            

                    //call the process method on this node
                    operation.call(this, node);

                    // traverse to the right
                    if (node.right !== null) {
                        inOrder(node.right);
                    }
                }
            }

            inOrder(this._root);
        }
    };
    
    ConfluenceMobile.Utils.BinarySearchTree = BinarySearchTree;
    
})();