(function ($) {

    // Constructor so we can have the optional one fly out from the right too.
    var FlyOut = function (options) {

        this.el = $(options.el);

        this.onUpdate = options.onUpdate || function () { };

        this.onAnimationFinished = options.onAnimationFinished || function () { };

        this.visible = this.changingVisibilityState = this.expanded = false;

        this.body = $("body");

        this.classes = {
            OPEN : "fly-out-open",
            FULL_WIDTH: "fly-out-full-width",
            HIDING: "fly-out-hiding",
            SHOWING: "fly-out-showing",
            SHOWING_FINISHED: "fly-out-finished-showing",
            HIDING_FINISHED: "fly-out-finished-hiding"
        }

        var that = this;
        this.body.bind($.fx.transitionEnd, function (event) {

            if (that.visible) {
                that.body
                    .removeClass(that.classes.SHOWING)
                    .addClass(that.classes.SHOWING_FINISHED);

            } else {
                that.body
                    .removeClass([that.classes.HIDING, that.classes.SHOWING].join(" "))
                    .addClass(that.classes.HIDING_FINISHED);
            }

            that.changingVisibilityState = false;

            that.onAnimationFinished.call(that);

        });

        this._update();

    }

    FlyOut.prototype = {
        _update: function () {
            // CSS Transforms handle all the animation.
            this.body
                .toggleClass(this.classes.OPEN , this.visible)
                .toggleClass(this.classes.SHOWING, this.changingVisibilityState && this.visible)
                .toggleClass(this.classes.HIDING, this.changingVisibilityState && !this.visible)
                .toggleClass(this.classes.FULL_WIDTH, this.expanded);

            if (this.changingVisibilityState) {
                this.body.removeClass(this.classes.SHOWING_FINISHED + " " + this.classes.HIDING_FINISHED);
            }

            this.onUpdate.call(this);

        },

        toggle: function () {
            this.visible = !this.visible;
            this.changingVisibilityState = true;
            this._update();
        },

        show: function () {
            if (this.visible) {
                return;
            }

            this.changingVisibilityState = true;

            this.visible = true;
            this._update();
        },

        hide: function () {
            if (!this.visible) {
                return;
            }

            this.changingVisibilityState = true;
            this.restoreWidth();

            this.visible = false;
            this._update();
        },

        expandWidth: function () {
            if (this.expanded) {
                return;
            }

            this.changingVisibilityState = false;

            this.expanded = true;
            this._update();
        },

        restoreWidth: function () {
            if (!this.expanded) {
                return;
            }

            this.changingVisibilityState = false;

            this.expanded = false;
            this._update();
        },

        isVisible: function () {
            return !!this.visible;
        },

        isChangingVisibilityState: function () {
            return !!this.changingVisibilityState;
        }
    }

    $(function () {

        var $innerContainer = $("#inner-container"),
            $navButton = $innerContainer.children(".header").find(".nav button"),
            $flyOut = $(".fly-out"),
            $searchInput = $flyOut.find("input[type='search']"),
            $logInLink = $flyOut.find(".log-in-link");

        var flyout = ConfluenceMobile.flyout = new FlyOut({
            el: $flyOut,
            onUpdate: function() {
                $navButton.toggleClass("activated", this.isVisible());

                // CONFDEV-8626 - avoid focus on the search box
                if (this.isChangingVisibilityState()) {
                    $searchInput[0].disabled = true;

                    // The fly out is not animated in Android.
                    if ($.os.android && this.isVisible()) {
                        setTimeout(function () {
                            $searchInput[0].disabled = false;
                        }, 300);
                    }
                }
            },
            onAnimationFinished: function() {
                $searchInput[0].disabled = !this.isVisible() && this.isChangingVisibilityState();
                ConfluenceMobile.hideAddressBar();
            }
        });

        ConfluenceMobile.routerEventAggregator.bind("change", function () {
            ConfluenceMobile.hideAddressBar();            
            ConfluenceMobile.flyout.hide();
        });        

        /**
         * Log in should preserve your current 'route' so that you are 
         * returned here after the log in operation.
         */
        $logInLink.on("click", function(event) {
            var fragment = "#" + Backbone.history.getFragment();
            $logInLink.attr("href", $logInLink.attr("href") + encodeURIComponent(fragment));
        });
        
        $navButton.click(function (event) {
            // Stop event from propagating to "#inner-container".
            event.stopPropagation();
            flyout.toggle();

            ConfluenceMobile.Analytics.trackEvent("flyout", "toggle");
        });

        $innerContainer.click(function (event) {
            if (!flyout.isVisible()) {
                return;
            }

            flyout.hide(); // clicking anywhere in the inner container should close the flyout

            event.preventDefault();
        });

        $flyOut.find("a.content-link").each(function() {
            // When items on the left flyout is clicked, hide the flyout
            $(this).click(function() {
                flyout.hide();
            });
        });

    });

})(Zepto);