/**
 * Settings that each Autocomplete will be initialized on, depending on the trigger character used to activate the
 * autocomplete.
 */
AJS.Rte.BootstrapManager.addOnInitCallback(function () {
    var THROTTLE_MS = 1000;
    var latestQuery;
    var lastRunQueryTimestamp;
    var lastRunQueryLength;
    var resultCount;
    var delayCallingGetRestData;
    var lastXhr;
    var dataRunCount = 0;
    var dataReturnCount = 0;
    var queriesAttemptedWithoutResponse = 0;

    // Confluence 4.1.x uses tinymce.confluence, 4.2 onwards use Confluence.Editor
    if (!Confluence.Editor.Autocompleter) {
        Confluence.Editor.Autocompleter = tinymce.confluence.Autocompleter;
    }

    var getUrl = function (val) {
            if (val) {
                return AJS.contextPath() + "/rest/prototype/1/search.json";
            } else if (AJS.Meta.get('remote-user')) {
                return AJS.contextPath() + "/rest/mentions/1/network.json";
            }

            return null;
        },

        getParams = function (autoCompleteControl, val) {
            var params = {
                "max-results": autoCompleteControl.maxResults || 10
            };
            if (val) {
                params.query = Confluence.unescapeEntities(val);
                params.search = "user";
            }
            return params;
        },

        /**
         * Check if the matching result is in the same order of user input.
         * For example:
         * Result "firstName lastName" is in the same order of user input "first last",
         * Result "lastName firstName" is not.
         *
         * @param query User input
         * @param item Item to display in the mention
         * @returns {boolean}
         */
        isMatchedInInputOrder = function (query, item) {
            var querySegments = query.split(/\s/);
            var searchable = item.title;

            var lastPosition = -1;
            for (var i = 0; i < querySegments.length; i++) {
                var pos = searchable.indexOf(querySegments[i]);
                if (pos < lastPosition) {
                    return false;
                }

                lastPosition = pos;
            }

            return true;
        },

        loadData = function (json, query, callback, field, getRestSpecificAdditionLinks) {
            // response received, reset and increment counters
            queriesAttemptedWithoutResponse = 0;
            dataReturnCount++;
            AJS.debug("Queries returned: " + dataReturnCount);

            if (latestQuery !== query) {
                // Discard outdated results as user have typed more or removed some characters.
                // This happens when a previous request return slower than the newer one.
                AJS.debug("Discarding outdated result");
                return;
            }

            var hasErrors = json.statusMessage;
            var matrix;
            if (hasErrors) {
                matrix = [[{html: json.statusMessage, className: "error"}]];
            } else {
                var items = [];
                matrix = [items];
                if (json.result) {
                    AJS.$.each(json.result, function (index, entry) {
                        var item = AJS.$.extend({}, entry);
                        item.key = item.username;
                        item.linkAlias = item.title; // keep track of the unmodified full name for use as the link alias
                        item.title = item.title + " (" + item.username + ")";
                        items.push(item);
                    });

                    // Track for request throttling
                    resultCount = items.length;
                    lastRunQueryLength = query.length;

                    // We can search by "firstName lastName" or "lastName firstName"
                    // However server results are sort of random.
                    // Here we want to ensure results that matching "firstName lastName" are placed at top
                    items.sort(function(a, b) {
                        var aMatched = isMatchedInInputOrder(query, a);
                        var bMatched = isMatchedInInputOrder(query, b);

                        if (aMatched === bMatched) {
                            // If both items are not in order. Use string compare
                            return (a < b ? a : b)
                        }

                        return aMatched ? -1 : 1;
                    });
                }
                matrix = AJS.REST.convertFromRest(matrix);
            }
            // do conversion
            function restSpecificAdditionLinksCallback(value, additionalLinks) {
                if (getRestSpecificAdditionLinks && typeof getRestSpecificAdditionLinks === "function") {
                    getRestSpecificAdditionLinks(matrix, value, additionalLinks)
                }
            }

            // Specify the tokens for highlight. We want to highlight "[us]er [ab]c" if the input is "us ab"
            callback(matrix, query, restSpecificAdditionLinksCallback, query.split(/\s/));

            // clear lastXhr
            lastXhr = null;
        },

    // HACK: copied from Confluence, because we can't override loadData()
        getRestData = function (autoCompleteControl, getUrl, getParams, val, callback, suggestionField, getRestSpecificAdditionLinks) {
            var url = getUrl(val);
            var cacheManager = autoCompleteControl.settings.cacheManager;
            var cachedData = cacheManager.get(val);

            if (url) {
                // If we're still running an old search, abort it
                if (lastXhr) {
                    AJS.debug("Aborting older search");
                    lastXhr.abort();
                }

                if (cachedData) {
                    //Cached response
                    loadData(cachedData, val, callback, suggestionField, getRestSpecificAdditionLinks);
                } else {
                    lastXhr = $.ajax({
                        type: "GET",
                        url: url,
                        data: getParams(autoCompleteControl, val),
                        dataType: "json",
                        global: false,
                        timeout: 10000
                    });

                    // Always update the cache (eventual consistency)
                    lastXhr.done(function (json) {
                        cacheManager.put(val, json);
                    });

                    //Async response
                    lastXhr.done(function (json) {
                        loadData(json, val, callback, suggestionField, getRestSpecificAdditionLinks);
                    });
                    lastXhr.fail(function (xml, status) { // ajax error handler
                        if (status === "timeout") {
                            loadData({statusMessage: "Timeout", query: val}, val, callback, suggestionField);
                        }
                    });
                }
            } else {
                // If no url, default items may be displayed - run the callback with no data.
                callback([], val);
            }
        };

    // Link settings.
    Confluence.Editor.Autocompleter.Settings["@"] = {
        ch: "@",
        endChars: [],
        dropDownClassName: "autocomplete-mentions",
        selectFirstItem: true,

        getHeaderText: function (autoCompleteControl, value) {
            if (value) {
                return AJS.I18n.getText("editor.autocomplete.mentions.header.text");
            }
            else {
                return AJS.I18n.getText("editor.autocomplete.mentions.header.text.hint");
            }
        },

        getAdditionalLinks: function (autoCompleteControl, value, callback) {
            var additionalLinks = [];
            callback(value, additionalLinks);
            return additionalLinks;
        },

        getDataAndRunCallback: function (autoCompleteControl, val, callback) {
            // When mention was placed inside inline task. Firefox will append a space after @ making server fail to return results.
            val = $.trim(val);

            // Don't process if the string is the same as the last run query
            // or if there were no results on the last search and the new query is the same but longer
            if (val.length > 0 && (val === latestQuery || (resultCount === 0 && (val.indexOf(latestQuery) === 0)))) {
                AJS.debug("Not running search. String contains last query: " + (val.indexOf(latestQuery) === 0));
                return;
            }

            // Don't process if we've attempted 3 queries without getting a response
            if (queriesAttemptedWithoutResponse >= 3) {
                AJS.debug("Not running search. 3 queries have been attempted without receiving a response.");
                return;
            }

            var _getRestData = function(){
                // Keep track of latest input. Compare it with the query in result callback.
                // So we can ignore the result if we are getting an outdated result due to server lag.
                // Also use to discard current search if it contains a string that previously returned no results
                latestQuery = val;
                lastRunQueryTimestamp = Date.now();
                dataRunCount++;
                AJS.debug("Queries run: " + dataRunCount);

                getRestData(autoCompleteControl, getUrl, getParams, val, callback, "content",
                    function (matrix, value, additionalLinks) {
                        if (value !== '' && matrix.length < 2) {
                            if (additionalLinks.length === 0) {
                                additionalLinks.push({
                                    html: '<span class="no-result">' + AJS.I18n.getText("editor.autocomplete.mentions.nouserfound") + '</span>'
                                });
                            }
                        } else {
                            additionalLinks.length = 0;
                        }
                    }
                );
            };

            var now = Date.now();
            var timeSinceLastRun = (now - lastRunQueryTimestamp);

            // If this is the first search, or more than 3 extra chars have been typed, run search immediately
            // otherwise wait for one second for more recent search requests
            // This will stop mentions spamming Lucene on every character - CONFSRV-1022
            if (!lastRunQueryTimestamp || timeSinceLastRun > THROTTLE_MS || val.length > lastRunQueryLength + 3) {
                queriesAttemptedWithoutResponse++;
                AJS.debug("Running search immediately: '" + val +
                    "'. Queries attempted without response (including this one): " + queriesAttemptedWithoutResponse);
                clearTimeout(delayCallingGetRestData);
                _getRestData();
            }
            else {
                queriesAttemptedWithoutResponse++;
                AJS.debug("Delaying search for " + (THROTTLE_MS - timeSinceLastRun) + "ms: '" + val +
                "'. Queries attempted without response (including this one): " + queriesAttemptedWithoutResponse);
                clearTimeout(delayCallingGetRestData);
                delayCallingGetRestData = setTimeout(_getRestData, THROTTLE_MS - timeSinceLastRun);
            }
        },

        update: function (autoCompleteControl, link) {

            if (link.restObj) {
                // link alias is different to menu item title in this autocomplete because we show the username
                var restObj = AJS.$.extend(link.restObj, {title: link.restObj.linkAlias});
                link = Confluence.Link.fromREST(restObj);
            }
            link.insert();
        }
    }
});
