define('confluence-link-browser/link-object', [
    'jquery',
    'ajs',
    'confluence/legacy'
], function ($,
             AJS,
             Confluence) {
    "use strict";

    var OPEN_IN_NEW_WINDOW_DARK_FEATURE = 'link.openInNewWindow';

    /**
     * The Link object contains all of Confluence's business logic about a front-end Link.
     *
     * It should not interact with the RTE directly but rather use the LinkAdapter singleton.
     */
    var Link = {

        // Until this code gets refactored to use proper prototyping just do a duck-type check
        // instead of a constructor check.
        isLink: function (obj) {
            return obj && !!obj.fillNode;
        }
    };

    var newLink = function (props) {

        // If the object passed in is already a Link just return it.
        if (Link.isLink(props)) {
            return props;
        }

        var link = {

            /**
             * Inserts this link into the RTE
             */
            insert: function () {
                return Confluence.Editor.LinkAdapter.setLink(link);
            },

            /**
             * Fills the passed DOM node with the contents of this Link object.
             * @param $link the jQuery-wrapped node to fill with the link data
             */
            fillNode: function ($link) {
                var attrs = this.attrs;
                attrs.href = attrs.href || '#';
                $link.attr(attrs);

                if (this.classes && this.classes.length) {
                    $link.addClass(this.classes.join(" "));
                }

                $link.html(this.body.html);
                return $link;
            },

            /**
             * Returns a representation of this link as data only - no functions.
             */
            getData: function () {
                var props = {};
                var key;

                for (key in this) {
                    if (this.hasOwnProperty(key) && !$.isFunction(this[key])) {
                        props[key] = this[key];
                    }
                }
                return props;
            },

            /**
             * If the body for this link is a lone image, return it.
             */
            getLinkedImage: function () {
                if (this.body && this.body.jquery) {
                    return this.body.length === 1 && this.body.is('img') && this.body;
                }
                return null;
            },

            getResourceId: function () {
                return this.attrs["data-linked-resource-id"] || "";
            },

            getResourceVersion: function () {
                return this.attrs["data-linked-resource-version"] || "";
            },

            /**
             * Returns true if this link points to Confluence content like a Page, Attachment, Space, et al.
             */
            isToConfluenceEntity: function () {
                return this.attrs["data-linked-resource-id"];
            },

            isToAttachmentOnSamePage: function (contentId) {
                return this.attrs["data-linked-resource-type"] == 'attachment' && this.attrs["data-linked-resource-container-id"] == contentId;
            },

            /**
             * A standard link will be stored as an a tag whereas a custom 'atlassian-content' link will be stored as an ac:link
             * tag. This method lets you differentiate between them in the browser.
             *
             * When used in conjunction with isToConfluenceEntity this can identify a relative link (link to the current page).
             *
             * @return true if the link is an atlassian-content (custom XHTML namespace) link (as opposed to an HTML link)
             */
            isCustomAtlassianContentLink: function () {
                if (this.classes && this.classes.length) {
                    return $.inArray("confluence-link", this.classes) != -1;
                } else {
                    return false;
                }
            },

            /**
             * @return true if the link has the correct anchor attribute
             */
            hasAnchor: function () {
                return this.attrs["data-anchor"];
            },

            getResourceType: function () {
                return this.attrs["data-linked-resource-type"];
            },

            getDefaultAlias: function () {
                return this.attrs["data-linked-resource-default-alias"];
            },

            getHref: function () {
                return this.attrs.href;
            },

            getAnchor: function () {
                return this.attrs["data-anchor"];
            },

            getHtml: function () {
                return this.body.html;
            },

            getShortcut: function () {
                return this.attrs["data-linked-resource-shortcut"];
            },

            isHrefValid: function () {
                return this.attrs.href && this.attrs.href != 'http://';
            },

            isImage: function () {
                return this.body.isImage;
            },

            isNewLink: function () {
                return $.isEmptyObject(this.attrs);    // a link with no attributes has no destination ==> new
            },

            isShortcutLink: function () {
                return this.getResourceType() === "shortcut";
            },

            isExternalLink: function () {
                return !this.isCustomAtlassianContentLink();
            },
            showsBreadcrumbs: function () {
                return true;
            },
            getTarget: function () {
                return this.attrs.target;
            },
            setTarget: function (target) {
                if (target) {
                    this.attrs.target = target;
                } else {
                    this.removeTarget();
                }
            },
            removeTarget: function () {
                if( this.attrs && this.attrs.target ){
                    delete this.attrs.target;
                }
            }
        };

        // if the attrs in the property contain class values extract them
        if (props && props.attrs) {
            var attrsOnly = {};
            var classAttrValue = null;

            $.each(props.attrs, function (key, value) {
                if (key == "class") {
                    classAttrValue = value;
                } else {
                    attrsOnly[key] = value;
                }
            });

            props.attrs = attrsOnly;
            if (classAttrValue) {
                var classAttrArray = classAttrValue.split(" ");

                if (props.classes && props.classes.length) {
                    props.classes = props.classes.concat(classAttrArray);
                } else {
                    props.classes = classAttrArray;
                }
            }
        }

        $.extend(link, props);

        return link;
    };

    /**
     * Returns the image node if the node is an image or if it is a linked image.
     * If no image node is found, null is returned.
     * @param node the node to inspect
     */
    function getImageNode(node) {
        if ($.nodeName(node, 'img')) {
            return node;
        }

        //we are only interested in nodes that have just an image node.
        var hasOneChild = node.hasChildNodes() && node.childNodes.length === 1;
        if (hasOneChild && $.nodeName(node, "a") && $.nodeName(node.firstChild, "img")) {
            return node.firstChild;
        }

        return null;
    }

    /**
     * Returns the name of given image node. If the node is null, an empty string is returned.
     * @param imageNode - the image node to inspect.
     */
    function getImageName(imageNode) {
        if (!imageNode) {
            return "";
        }

        return ($(imageNode).attr('data-linked-resource-default-alias') || $(imageNode).attr('src'));
    }

    /**
     * Returns a map of the attributes present in a link node.
     * @param node
     */
    function getLinkNodeAttributes(node) {
        var attrs = {};
        if (node.nodeName === "A") {
            $(node.attributes).each(function () {
                attrs[this.name] = this.value;
            });
        }

        return attrs;
    }

    /**
     * Returns true if the range between two sibling nodes (have the same parent)
     * contains only text nodes.
     * @param startNode - the node to start from
     * @param endNode - the node to end with
     */
    function areSiblingsTextNodes(startNode, endNode) {
        var currentNode = startNode;
        while (currentNode && currentNode != endNode) {
            if (currentNode.nodeType != 3) {
                return false;
            }

            currentNode = currentNode.nextSibling;
        }

        return endNode.nodeType === 3;
    }

    /**
     * This attempts to resolves the given range to account for the situation
     * where one cursor is set just inside of an element, while the other
     * is set inside the parent element.
     * If the range can be resolved such that the start and end container
     * share the same parent, the two values are returned.
     * Otherwise, null is returned.
     * @param rng
     */
    function resolveRange(rng) {
        var tinymce = require('tinymce');

        var dom = tinymce.activeEditor.dom;
        var start = dom.isBlock(rng.startContainer) ? rng.startContainer.childNodes[rng.startOffset] : rng.startContainer;
        var end = dom.isBlock(rng.endContainer) ? rng.endContainer.childNodes[rng.endOffset - 1] : rng.endContainer;

        if (start && end && (start.parentNode == end.parentNode)) {
            return {
                start: start,
                end: end
            };
        } else {
            return null;
        }
    }

    function isSelectionContainsEditableLinkText(rng) {
        // Collapsed ranges are always editable - the link text field will be empty
        if (rng.collapsed) {
            return true;
        }

        var start = rng.startContainer;
        var end = rng.endContainer;

        // If the selection is a single text node that text can be edited.
        // Anything else - including selected images - cannot be edited.
        if (start == end) {
            if (start.nodeType === 3) {
                return true;
            }

            return areSiblingsTextNodes(start.childNodes[rng.startOffset], start.childNodes[rng.endOffset - 1]);
        }

        var resolvedContainers = resolveRange(rng);
        return !!resolvedContainers && areSiblingsTextNodes(resolvedContainers.start, resolvedContainers.end);
    }

    Link.fromData = function (data) {
        return newLink(data);
    };

    /**
     * Given a link node and an alias, creates a link object with correct body and attributes.
     * @param linkNode The link node DOM object. The node is not expected to exist in the Editor.
     * @param alias The link alias.
     */
    Link.fromNode = function (linkNode, alias) {
        var link = newLink({
            attrs: {},
            body: {
                html: alias,
                text: alias
            }
        });

        return newLink({
            attrs: getLinkNodeAttributes(linkNode),
            body: {
                html: alias,
                text: alias
            }
        });
    };


    /**
     * Creates a link object from a provided link in the Editor.
     *
     * @param linkEl - an existing link in the editor
     * @param linkText - formatted link text
     */
    Link.fromSelectedAnchor = function (linkEl, linkText) {
        var $linkEl = $(linkEl);
        var imageNode = getImageNode(linkEl);
        var isEditable = !imageNode && areSiblingsTextNodes(linkEl.firstChild, linkEl.lastChild);

        return newLink({
            attrs: getLinkNodeAttributes(linkEl),
            body: {
                isEditable: isEditable,
                isImage: !!imageNode,
                html: $linkEl.html(),
                imgName: getImageName(imageNode),
                text: linkText
            }
        });
    };

    /**
     * Called when creating a Link from an editor selection.
     *
     * If the selection is inside an existing link use #fromSelectedAnchor
     *
     * @param rng - a w3c range object of the selection
     * @param selectedNode - the common ancestor node of the start and end containers
     * @param selectedHtml - the raw html of the selection
     * @param selectedText - formatted selection text
     */
    Link.fromSelection = function (rng, selectedNode, selectedHtml, selectedText) {
        var imageNode = getImageNode(selectedNode);
        var isEditable = !imageNode && isSelectionContainsEditableLinkText(rng);

        return newLink({
            attrs: {},
            body: {
                isEditable: isEditable,
                isImage: !!imageNode,
                html: selectedHtml,
                imgName: getImageName(imageNode),
                text: selectedText
            }
        });
    };

    /**
     * Creates a Link object from a REST object
     * @param restObj
     */
    Link.fromREST = function (restObj) {
        var link = newLink({
            attrs: {
                "data-base-url": AJS.Confluence.getBaseUrl(),
                "data-linked-resource-id": restObj.id,
                "data-linked-resource-type": restObj.type,
                "data-linked-resource-content-type": restObj.contentType,
                href: AJS.REST.findLink(restObj.link),
                "data-linked-resource-default-alias": restObj.title
            },
            body: {
                html: AJS.escapeHtml(restObj.title),
                text: AJS.escapeHtml(restObj.title)
            },
            classes: ["confluence-link"]
        });

        // CONDEV-5286: The problem here is that the REST response will have a type of 'user'
        // whereas the conversion to storage format requires 'userinfo'.
        if (restObj.type === "user") {
            link.attrs["data-linked-resource-type"] = "userinfo";
        }

        return link;
    };

    /**
     * Creates a link that points to a new page.
     */
    Link.createLinkToNewPage = function (pageTitle, spaceKey) {
        return newLink({
            attrs: {
                "data-space-key": spaceKey,
                "data-content-title": pageTitle,
                href: AJS.contextPath() + '/pages/createpage.action?spaceKey=' + spaceKey + '&title=' + pageTitle
            },
            body: {
                html: AJS.escapeEntities(pageTitle),
                text: pageTitle
            },
            classes: ["createlink", "confluence-link"]
        });
    };

    /**
     * Creates a link to an external webpage or email address.
     */
    Link.makeExternalLink = function (href) {
        return newLink({
            attrs: {
                href: href
            },
            body: {
                html: href,
                text: href
            }
        });
    };

    Link.isExternalLink = function (destination) { //Same check as ConfluenceLinkResolver.java
        return destination && (destination.match(/^(\/\/|mailto:|file:|http:|https:)/) || destination.indexOf("\\") === 0);
    };

    return Link;
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-link-browser/link-object', 'Confluence.Link');