define('confluence-link-browser/link-browser', [
    'ajs',
    'confluence/legacy',
    'jquery'
], function (AJS,
             Confluence,
             $) {
    var OPEN_IN_NEW_WINDOW_DARK_FEATURE = 'link.openInNewWindow';
    var popup;
    var locationPresenter;
    var submitButton;
    var currentTab;
    var dialogId = "insert-link-dialog";

    // The key of the search panel, uses internally and exposed.
    var SEARCH_PANEL = 'search';
    var ATTACHMENTS_PANEL = 'attachments';
    var WEBLINK_PANEL = 'weblink';
    var ADVANCED_PANEL = 'advanced';

    /**
     * Called when the user pressed the Insert/Edit button or presses Enter.
     */
    function submit() {
        if (submitButton.prop("disabled")) {
            return;
        }

        submitButton.prop('disabled', true);
        AJS.debug("link-browser.js: submit");

        // Some tabs may need to alter the location if the submit has been called before blur on some element.
        currentTab.preSubmit && currentTab.preSubmit();

        var link = locationPresenter.getLink();

        if (!AJS.DarkFeatures.isEnabled(OPEN_IN_NEW_WINDOW_DARK_FEATURE)) {
            link.removeTarget();
        }
        close();

        link.insert();

        // CONFDEV-10562 (AUI's remove.dialog event fires before the link has been inserted)
        AJS.trigger('closed.link-browser');
    }

    /**
     * Closes the dialog and resets state to before it was opened - called by both submit and cancel.
     */
    function close() {
        popup.hide().remove();
        AJS.Rte.BookmarkManager.restoreBookmark();
    }

    /**
     * Called when the user presses the Cancel link or presses Escape
     */
    function cancel() {
        close();
        AJS.trigger('closed.link-browser');
    }

    /**
     * Make the Link Browser dialog element, setting button text based on whether a link is being created or edited.
     */
    function makePopup(isNew) {
        var dialog;
        var dialogTitle;
        var submitText;

        dialog = new AJS.ConfluenceDialog({
            width: 840,
            height: 590,
            id: dialogId,
            onCancel: cancel,
            onSubmit: submit
        });

        dialogTitle = isNew ? AJS.I18n.getText("link.browser.insert.title") : AJS.I18n.getText("link.browser.edit.title");
        submitText = isNew ? AJS.I18n.getText("link.browser.insert.button") : AJS.I18n.getText("link.browser.save.button");

        dialog.addHeader(dialogTitle);

        dialog.addButton(submitText, submit);

        dialog.addCancel(AJS.I18n.getText("cancel.name"), cancel);
        dialog.addHelpText(AJS.I18n.getText('link.browser.autocomplete.hint'));

        // See CONFDEV-14467 - this can be removed when CONFDEV-12434 is implemented
        var $dialogTip = $('#' + dialogId + ' .dialog-tip');
        $dialogTip.attr('title', $dialogTip.text());

        // CONFDEV-12853: Add help link via prepend() instead of append() to prevent FF display issue
        $("#" + dialogId + " .dialog-components .dialog-title").prepend(Confluence.Templates.LinkBrowser.helpLink());

        submitButton = dialog.get("button:0")[0].item;
        submitButton.attr("id", 'link-browser-insert');
        submitButton.prop("disabled", true);

        return dialog;
    }

    // Creates the panel element for the tab specified by the web-item,
    // and initializes the tab-controller for it.
    function makeTab(i, tabPresenter, controller) {
        var key = tabPresenter.key;
        var tabTitle = tabPresenter.label;
        var className = key + "-panel";
        var panel = Confluence.Templates.LinkBrowser[key + 'Panel']({atlToken: AJS.Meta.get('atl-token')});

        popup.addPanel(tabTitle, panel, className, className + '-id');

        // The Panel object instance.
        var panelObj = popup.get("panel:" + i);

        // The controller.tabs map will have been pre-populated by a 'dialog-created.link-browser' event binding.
        var tab = controller.tabs[key];
        tab.panelObj = panelObj;
        tab.key = key;
        tab.createPanel({
            baseElement: $(panelObj[0].body)
        });

        // Called automatically by the AJS.Dialog code when a panel is changed
        panelObj[0].onblur = tab.onDeselect;

        panelObj[0].onselect = function () {
            // Most tabs don't show breadcrumbs so it's opt-in by declaring a property.
            var hasBreadcrumbs = !!tab.hasBreadcrumbs;
            AJS.debug("Link Browser: on tab select, breadcrumbs enabled: " + hasBreadcrumbs);
            tab.onSelect();
            locationPresenter.refresh(hasBreadcrumbs);
            currentTab = tab;
        };

        return tab;
    }

    /**
     * Browser tabs are pluggable based on web-items with section
     * "system.editor.link.browser.tabs", sorted by item weight.
     * {@see link-browser-web-items.vm}
     */
    function getTabPresenters() {
        var supportedLinkTypes = $("#link-browser-tab-items").find("div").map(function () {
            var item = $(this);
            return {
                key: item.text(),
                weight: item.attr("data-weight"),
                label: this.title
            };
        }).sort(function (a, b) {
            return a.weight - b.weight;
        });
        var isAttachmentSupported = AJS.Meta.get("content-type") !== "template";
        return $.grep(supportedLinkTypes, function (item, index) {
            return isAttachmentSupported || item.key !== "attachments";
        });
    }

    /**
     * Sets up the Link Browser tabs and the tab it will be opened to, based on any existing link being edited or a
     * present panel key.
     *
     * @param controller the LB controller
     * @param presetPanelKey a panel key like 'search', 'attachments' or 'web-link'
     * @param linkObj a existing link object being edited
     */
    function setupTabs(controller, linkObj, presetPanelKey) {

        var tabPresenters = getTabPresenters();

        // Loops through each web-item tab, setting up the associated panel.
        var firstTab;
        var currentTab = null;
        for (var i = 0, ii = tabPresenters.length; i < ii; i++) {
            var tab = makeTab(i, tabPresenters[i], controller);
            if (i === 0) {
                firstTab = tab;
            }
            if (!linkObj.isNewLink() && $.isFunction(tab.handlesLink) && tab.handlesLink(linkObj)) {
                currentTab = tab;
                tab.openedLink = linkObj;
            }
            else if (presetPanelKey == tab.key) {
                // User wants to open the Link Browser to a particular panel
                currentTab = tab;
            }
        }
        if (currentTab) {
            // A particular tab is selected either by user choice or editing an existing link:
            // don't let the dialog perform its default action of opening the last tab selected.
            controller.popup.overrideLastTab();
        } else {
            currentTab = firstTab;  // Select the first tab by default
        }
        return currentTab;
    }

    /**
     * Returns the Main controller object, separate to the DOM. Should be QUnit-testable.
     */
    function makeController() {
        return {
            OPEN_IN_NEW_WINDOW_DARK_FEATURE: OPEN_IN_NEW_WINDOW_DARK_FEATURE,

            // Contains a map of LinkTabPanel objects.
            tabs: {},

            // Called by Tabs when the link they represent is changed
            setLink: function (link, hasBreadcrumbs) {
                locationPresenter.setLink(link, hasBreadcrumbs);
            },

            getLink: function () {
                return locationPresenter.getLink();
            },

            getSelectedDataTableItem: function () {
                return $('.data-table:visible tr.selected');
            },

            // Called by Tabs when the link they represent is valid or not.
            linkValid: function (valid) {
                submitButton.prop("disabled", !valid);
            },

            // If the link text can't be focused because it isn't visible, focus the submit button instead.
            focusLinkText: function () {
                if (!locationPresenter.focusLinkText()) {
                    AJS.debug('LinkBrowser: focusing submit button');
                    submitButton.focus();
                }
            },

            getLinkText: function () {
                return locationPresenter.getLinkText();
            },

            isLinkTextVisible: function () {
                return locationPresenter.isLinkTextVisible();
            },

            isNewWindowCheckboxVisible: function () {
                return locationPresenter.isNewWindowCheckboxVisible();
            },

            hasBreadcrumbs: function (crumbTexts) {
                return locationPresenter.hasBreadcrumbs(crumbTexts);
            },

            getLocationPresenter: function () {
                return locationPresenter;
            },

            // Select the 'Search' tab and perform a search on the passed text
            doSearch: function (searchText) {
                this.tabs[SEARCH_PANEL].doSearch(searchText);
            },

            getSearchTextField: function () {
                return this.tabs[SEARCH_PANEL].getSearchTextField();
            },

            // Moves the location panel to be inside the new parent. Used for styling.
            moveLocationPanel: function (newParent) {
                locationPresenter.moveLocationPanel(newParent);
            },

            // Moves the location panel back to its normal position
            restoreLocationPanel: function () {
                locationPresenter.restoreLocationPanel();
            },

            /**
             * Selects the Link Browser tab with the given index.
             */
            gotoPanel: function (index) {
                this.popup.gotoPanel(index);
            },

            /**
             * Returns the currently-selected tab.
             */
            getCurrentPanel: function () {
                return this.popup.getCurrentPanel();
            },

            /**
             * If the web-link panel is visible, sets the URL field to be the passed value.
             */
            setWebLinkURL: function (url) {
                var webLinkTab = this.tabs[WEBLINK_PANEL];
                if (currentTab != webLinkTab) {
                    AJS.debug('Cannot set URL ' + url + ' on hidden Web Link panel');
                    return;
                }
                webLinkTab.setURL(url);
            },

            getWebLinkUrl: function () {
                var webLinkTab = this.tabs[WEBLINK_PANEL];
                if (currentTab != webLinkTab) {
                    AJS.debug('Cannot get URL on hidden Web Link panel');
                    return null;
                }
                return webLinkTab.getURL();
            },

            getTitle: function () {
                return this.popup.getTitle();
            },

            getSubmitButtonText: function () {
                return submitButton.text();
            },

            isSubmitButtonEnabled: function () {
                return submitButton.is(':enabled');
            },

            isVisible: function () {
                return this.popup.isVisible();
            },

            showOpenInNewWindowCheckbox: function (show) {
                return locationPresenter.showOpenInNewWindowCheckbox(show);
            },

            /**
             * If the advanced panel is visible, sets the link field.
             */
            getAdvancedTextField: function () {
                var advancedTab = this.tabs[ADVANCED_PANEL];
                if (currentTab != advancedTab) {
                    AJS.debug('Cannot get link text on hidden advanced panel');
                    return null;
                }
                return advancedTab.getLink();
            },

            setAdvancedTextField: function (text) {
                var advancedTab = this.tabs[ADVANCED_PANEL];
                if (currentTab != advancedTab) {
                    AJS.debug('Cannot set link text on hidden advanced panel');
                    return;
                }
                advancedTab.setLink(text);
            },

            submit: submit,
            cancel: cancel
        };
    }

    /**
     * Main method to open the Link Browser, called from open.
     */
    function openLinkDialog(options) {

        var controller = makeController();

        var linkObj = options.linkInfo;
        popup = makePopup(linkObj.isNewLink());
        controller.popup = popup;

        // Link browser tabs aren't registered until this point - new tabs
        // need to add a binding to this event.
        AJS.trigger("dialog-created.link-browser", [controller]);

        // Create the location panel that is shared between all Dialog panels.
        locationPresenter = Confluence.Editor.LinkBrowser.LinkInfoPresenter(controller);
        locationPresenter.setLinkBody(linkObj.body);

        currentTab = setupTabs(controller, linkObj, options.panelKey);

        popup.popup.element.find('.dialog-page-body:first').append(locationPresenter.getContainer());

        currentTab.panelObj.select();
        currentTab.openedLink = null;   // this ref shouldn't be used after the first tab is selected, so kill it

        popup.show();
        //controller the right thing to send?
        AJS.trigger("dialog-shown.link-browser", popup);
        return controller;
    }

    return {
        // These ids let RTE menu and autocomplete items open the Link Browser to a particular panel.
        SEARCH_PANEL: SEARCH_PANEL,
        ATTACHMENTS_PANEL: ATTACHMENTS_PANEL,
        WEBLINK_PANEL: WEBLINK_PANEL,
        ADVANCED_PANEL: ADVANCED_PANEL,

        /**
         * If the options include an "opener" function, that is used to launch the dialog; otherwise a LinkInfo object is
         * created for the current Editor and the default launcher called.
         *
         * @param options : panelKey - (optional) a panel of the Link Browser to open to
         *                  opener - (optional) a function used to launch the Link Browser
         */
        open: function (options) {
            // Init for AUI fancy file input
            $(function () {
                $('.ffi input[type="file"]#file_0').fancyFileInput();
            });

            // Prevent it from opening if another popup dialog is open
            if ($('.aui-dialog:visible').length) {
                return null;
            }

            options = options || {};

            // Store the current selection and scroll position, and get the selected text.
            AJS.Rte.BookmarkManager.storeBookmark();

            options.linkInfo = options.linkInfo || Confluence.Editor.LinkAdapter.getLink();

            if (options.opener) {
                // Any supplied opener function must include required state in its scope.
                return options.opener(options.linkInfo.alias, options.linkInfo);
            }

            return openLinkDialog(options);
        },

        cancel: cancel
    };
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-link-browser/link-browser', 'Confluence.Editor.LinkBrowser');

