define('confluence-link-browser/link-browser-tab-advanced', [
    'jquery',
    'ajs',
    'confluence/legacy',
    'confluence/api/constants'
], function ($,
             AJS,
             Confluence,
             CONSTANTS) {
    "use strict";

    // The Advanced tab registers itself when the Link Browser is created.
    return {
        dialogCreatedLinkBrowserAdvanced: function (e, linkBrowser) {
            var key = 'advanced';                   // This panel's key.
            var linkFieldName = 'advanced-link';    // The ID of the link input element.
            var errorFieldName = 'advanced-error';  // The ID of the error element.
            var $linkField;                         // The jQueryfied link input element.
            var $errorField;                        // The jQueryfied error field.
            var thisPanel;                          // A reference to this panel.
            var keyTimeout;                         // Used to tell when the user has stopped typing.

            /**
             * Triggered on a successful AJAX request.
             */
            var successfulConversion = function (data, textStatus) {
                var $data = $(data);

                if (!$data.length) {
                    return; // Do nothing if no data was returned.
                }

                // Use the first available link.
                var $link = $data.find("a:first");

                if (!$link.length) { // The markup wasn't link markup.
                    $errorField.text(AJS.I18n.getText("link.browser.advanced.error.not.link"));
                } else if ($link.hasClass("unresolved")) {
                    // If the link used an unspecified shortcut or unrecognised space, it will return with an unresolved class
                    var shortcutKey = $link.attr("shortcut-key");
                    var spaceKey = $link.attr("data-space-key");
                    if (!!shortcutKey) {
                        $errorField.text(AJS.I18n.getText("link.browser.advanced.error.unknown.shortcut", shortcutKey));
                    } else if (!!spaceKey) {
                        $errorField.text(AJS.I18n.getText("link.browser.advanced.error.unknown.space", spaceKey));
                    }
                } else { // The link is valid.
                    var linkObj = Confluence.Link.fromNode($link[0], $linkField.val());
                    linkBrowser.setLink(linkObj);
                }
            };


            /**
             * Triggered on an unsuccessful AJAX request.
             */
            var erroredConversion = function (XMLHttpRequest, textStatus, errorThrown) {
                AJS.debug("Error during conversion: textStatus = " + textStatus + ", errorThrown = " + errorThrown);
                $errorField.text(AJS.I18n.getText("link.browser.advanced.error.internal.error"));
            };


            /**
             * Uses the wikixhtmlconverter REST API to convert the provided wikimarkup into XHTML.
             */
            var convert = function (markup, successfulConversion, erroredConversion) {
                var conversionData = {
                    wiki: markup,
                    entityId: AJS.Meta.get("content-id"),
                    spaceKey: AJS.Meta.get("space-key")
                };

                $.ajax({
                    type: "POST",
                    contentType: "application/json; charset=utf-8",
                    url: CONSTANTS.CONTEXT_PATH + "/rest/tinymce/1/wikixhtmlconverter",
                    data: $.toJSON(conversionData),
                    dataType: "text",
                    success: successfulConversion,
                    error: erroredConversion,
                    timeout: 10000
                });
            };

            /**
             * Wraps the text currently in the link input field with square braces and sends
             * it to the convert() method.
             */
            var parseLinkText = function () {
                var text = $linkField.val();
                text = text.replace(/\[/g, "\\[").replace(/]/g, "\\]");
                convert("[" + text + "]", successfulConversion, erroredConversion);
            };

            // Define the available Link Browser Advanced tab methods.
            var tab = linkBrowser.tabs[key] = {
                createPanel: function (context) {
                    thisPanel = context.baseElement;
                    $linkField = thisPanel.find("input[name='advanced-link']");
                    $errorField = thisPanel.find("div[name='advanced-error']");
                    thisPanel.find("form").keydown(function (e) {
                        if (e.keyCode === 13 && !linkBrowser.isSubmitButtonEnabled()) {
                            e.preventDefault();
                        }
                    });
                    $linkField.keyup(function (e) {
                        clearTimeout(keyTimeout);
                        $errorField.text('');

                        if (!!$linkField.val()) {
                            keyTimeout = setTimeout(function () {
                                parseLinkText();
                            }, 200);
                        }
                    });
                },

                setLink: function (text) {
                    $linkField.val(text);
                    $linkField.keyup();
                    $linkField.change();
                },

                getLink: function () {
                    return $linkField.val();
                },

                // Called when the panel is selected
                onSelect: function () {
                    AJS.DarkFeatures.isEnabled(linkBrowser.OPEN_IN_NEW_WINDOW_DARK_FEATURE) && linkBrowser.showOpenInNewWindowCheckbox(true);
                    linkBrowser.moveLocationPanel(thisPanel.find("form:first"));

                    if (this.openedLink) {
                        if (this.openedLink.isShortcutLink()) {
                            tab.setLink(this.openedLink.getShortcut());

                        } else {
                            if (this.openedLink.getResourceId()) { //anchor to external page.
                                tab.setLink(this.openedLink.getDefaultAlias());
                            }
                            else { //anchor in the same page.
                                tab.setLink('#' + this.openedLink.getAnchor());
                            }
                        }

                        linkBrowser.setLink(this.openedLink);
                    }
                    // Defer focus to after LB is shown, gets around AJS.Dialog tabindex issues
                    setTimeout(function () {
                        $linkField.focus();
                    });
                },

                // Called when this panel is no longer selected
                onDeselect: function () {
                    linkBrowser.restoreLocationPanel();
                    linkBrowser.showOpenInNewWindowCheckbox(false);
                },

                handlesLink: function (linkObj) {
                    return linkObj.isShortcutLink() || linkObj.hasAnchor();
                }
            };
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-link-browser/link-browser-tab-advanced', function (LinkBrowserTabAdvanced) {
    var AJS = require('ajs');

    AJS.bind("dialog-created.link-browser", LinkBrowserTabAdvanced.dialogCreatedLinkBrowserAdvanced);
});