/**
 * LinkAdapter is an adapter between the RTE and a Link object.
 *
 * It should not know about the Link Browser or how the internals of a Link object work.
 * Any interaction with RTE nodes and Links should be via this object.
 */
define('confluence-link-browser/link-adapter', [
    'ajs',
    'jquery',
    'confluence/legacy',
    'tinymce'
], function(
    AJS,
    $,
    Confluence,
    tinymce
) {
    "use strict";

    return {

        /**
         * Inserts a link at the position of the current range.
         * @param linkObj a Confluence.Link instance
         */
        setLink: function (linkObj) {
            var ed = AJS.Rte.getEditor();
            var newLinkNode = $(ed.dom.create("a"), ed.getDoc());

            linkObj.fillNode(newLinkNode);
            var insertedNode = tinymce.confluence.NodeUtils.replaceSelection(newLinkNode);

            // CONFDEV-5203 - Prevent cursor loss when using arrow keys after inserting an
            // image. This is a FF-specific issue that occurs when adding or editing comments.
            if (!!$("#comments-section").length) {
                AJS.Rte.fixEditorFocus();
            }

            return insertedNode;
        },

        /**
         * Constructs a javascript object containing link information.
         * Invoked when the user tries to edit an existing link or make a piece of selected content into a link.
         *
         * Returns a new or existing link based on the current Editor selection.
         */
        getLink: function() {
            var selectedHtml;
            var selectedText;
            var rng;
            var ed = AJS.Rte.getEditor();
            var selection = ed.selection;
            var selectedNode = selection.getNode();
            var linkNode = $(selectedNode).parents().andSelf().filter("a[href]")[0];

            if (linkNode) {
                selection.select(linkNode);
                selectedText = selection.getContent({format : 'text'});

                return Confluence.Link.fromSelectedAnchor(linkNode, selectedText);
            } else {
                rng = selection.getRng(true);
                selectedText = selection.getContent({format : 'text'});
                selectedHtml = selection.getContent();
                return Confluence.Link.fromSelection(rng, selectedNode, selectedHtml, selectedText);
            }
        }

    };
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-link-browser/link-adapter', 'Confluence.Editor.LinkAdapter');