define('confluence-editor-reliable-save/reliable-save', [
    'ajs',
    'confluence/legacy',
    'underscore',
    'jquery',
    'window',
    'document',
    'confluence/api/constants',
    'confluence-editor/editor/page-editor-message',
    'confluence-editor/editor/page-editor-quit-dialog'
], function (
    AJS,
    Confluence,
    _,
    $,
    window,
    document,
    CONSTANTS,
    Message,
    QuitDialog
) {
    "use strict";

    var cleanupFunctions = [];

    // used to identify the source of and the action triggering an entity.stop or start
    var PUBLISH = 'confluence.editor.publish';

    function isSharedDraftsEnabled() {
        return AJS.Meta.get("shared-drafts");
    }

    function isNewPage() {
        return AJS.Meta.getBoolean("new-page");
    }

    function enableBar() {
        Confluence.Editor.UI.toggleSavebarBusy(false);
    }

    function displayGenericSaveMessage(error) {
        var errorMessage = AJS.I18n.getText("editor.offline.save.error.generic");
        Message.closeMessages(["generic-error"]);
        Message.handleMessage("generic-error", {
            type: "error",
            message: errorMessage
        }, bindEventsToDraftDialog);

        AJS.logError("Generic error: " + JSON.stringify(error));
    }

    function bindEventsToDraftDialog() {

        var $draft = $("#draft-messages");
        if ($draft.length > 0) {
            if ($draft.is(':visible')) {
                AJS.Confluence.Analytics.publish('rte.notification.draft');
            }

            $draft.find("a.use-draft").click(function (e) {
                e.stopPropagation();
                e.preventDefault();
                Confluence.Editor.Drafts.useDraft();
                AJS.Confluence.Analytics.publish('rte.notification.draft.resume');
            });
            $draft.find("a.discard-draft").click(function (e) {
                e.stopPropagation();
                e.preventDefault();
                SafeSave.Draft.discardDraft(AJS.Meta.get('page-id'), AJS.Meta.get('existing-draft-id'))
                        .done(SafeSave.Draft.onSuccessDiscardDraft)
                        .fail(SafeSave.Draft.onErrorDiscardDraft);
            });
        }
    }

    function bindEventsToRestoreFlag() {
        $("#editor-restore-title-link").click(function (e) {
            e.stopPropagation();
            e.preventDefault();
            $("#content-title").val(AJS.Meta.get("latest-published-page-title"));
            Message.closeMessages(["rename-during-limited-mode"]);
        });
    }

    function displayNoAuthorizedSaveMessage(triggerInvalidXsrfToken) {
        var errorMessage = AJS.I18n.getText("editor.offline.save.noauthorized");
        Message.closeMessages(["noauthorized"]);
        Message.handleMessage("noauthorized", {
            title: AJS.I18n.getText("editor.offline.save.noauthorized.title"),
            type: "error",
            message: errorMessage
        }, bindEventsToDraftDialog);
        if (triggerInvalidXsrfToken) {
            AJS.trigger("rte.safe-save.invalid-xsrf-token");
        }
    }

    function displayServerOfflineSaveMessage() {
        var errorMessage = AJS.I18n.getText("editor.offline.save.error");
        Message.closeMessages(["server-offline"]);
        Message.handleMessage("server-offline", {
            type: "error",
            message: errorMessage
        }, bindEventsToDraftDialog);
    }

    var SafeSave = {};

    SafeSave.Draft = {
        discardDraft: function (pageId, draftId) {

            var draftData = {
                draftId: draftId,
                pageId: pageId,
                type: AJS.Meta.get("draft-type"),
                spaceKey: AJS.Meta.get('space-key')
            };

            return $.ajax({
                type: "DELETE",
                url: CONSTANTS.CONTEXT_PATH + "/rest/tinymce/1/drafts/discard",
                data: $.toJSON(draftData),
                contentType: "application/json",
                dataType: "json"
            });
        },

        onSuccessDiscardDraft: function () {
            //Set draft-id to zero when discard draft in edit mode to make sure
            //if there is no draft at the time save button is clicked later on, we update content directly
            if (!isNewPage()) {
                AJS.Meta.set("draft-id", "0");
            }

            Message.closeMessages(["draft-message"]);
            Message.handleMessage("discarding-successfull", {
                type: "info",
                message: AJS.I18n.getText("editor.page.draft.discarding.info"),
                close: "auto"
            }, bindEventsToDraftDialog);
            AJS.Confluence.Analytics.publish('rte.notification.draft.discard');
        },
        onErrorDiscardDraft: function (errorData) {
            switch (errorData.status) {
                case 403:
                    displayNoAuthorizedSaveMessage(true);
                    break;
                case 404:
                    Message.handleMessage("draft-deleted", {
                        type: "info",
                        message: AJS.I18n.getText("editor.offline.draft.is.deleted")
                    }, bindEventsToDraftDialog);
                    break;
                case 422:
                    Message.handleMessage("discarding-invalid", {
                        type: "error",
                        message: AJS.I18n.getText("editor.offline.draft.is.invalid")
                    }, bindEventsToDraftDialog);
                    break;
                default:
                    Message.handleMessage("discarding-error", {
                        type: "error",
                        message: AJS.I18n.getText("discard.draft.unknown.error")
                    }, bindEventsToDraftDialog);
                    break;
            }
        }
    };

    SafeSave._internal = Confluence.SafeSafe && Confluence.SafeSave._internal ? Confluence.SafeSave._internal : {};

    // successful http response with no validation errors
    SafeSave._internal.onSuccessfulResponse = function (responseJSON) { // exposed for testing purposes

        $('#rte-button-overwrite').unbind("click.overwrite");

        var data = {
            dataType: 'json',
            contentId: AJS.Meta.get('content-id'),
            draftType: AJS.Meta.get('draft-type')
        };

        AJS.safe.post(CONSTANTS.CONTEXT_PATH + '/json/stopheartbeatactivity.action', data, function () {
            AJS.log('Stop heartbeat activity on', data.draftType, 'id', data.contentId);
        }, 'json').fail(function (xhr, status, err) {
            AJS.logError('Server error on stop heartbeat activity request:');
            AJS.log(err);
        }).always(function () {
            var newLocation = responseJSON._links.webui;
            if (!newLocation) {
                enableBar();
                return;
            }
            if (newLocation.indexOf('/') !== 0) { //relative address to current path
                window.location = newLocation;
            } else { // full virtual path, needs to be joined to context path.
                window.location = CONSTANTS.CONTEXT_PATH + newLocation;
            }
        });
    };

    SafeSave.initialize = function () {
        // Makes sure we are not in a comment
        if ($("#editpageform").length === 0 && $("#createpageform").length === 0) { return; }

        var allowedSaveErrorMessage = {
            duplicatedTitle: "A page with this title already exists",
            titleTooLong: "Title cannot be longer than 255 characters.",
            publishNewDraftDeprecated: "Unsupported call to publishNewDraft",
            existingDraftNotFound: "Could not find existing draft, perhaps you're trying to publish a personal draft?",
            renameDuringLimitedMode: "Unable to perform a page rename when limited mode is enabled"
        };

        var draftData = {
            existingDraftId: AJS.Meta.get("existing-draft-id") ? AJS.Meta.get("existing-draft-id") : 0,
            pageId: AJS.Meta.get("page-id"),
            type: AJS.Meta.get("draft-type"),
            spaceKey: AJS.Meta.get('space-key')
        };

        if (AJS.Meta.get("show-draft-message") === true) {
            if ($("#conflict-diffs").length > 0) {
                return;
            }
            $.ajax({
                type: "GET",
                url: CONSTANTS.CONTEXT_PATH + "/rest/tinymce/1/drafts/message",
                data: draftData,
                contentType: "application/json",
                dataType: "text json",
                success: function (data) {
                    if (data && data.draftData) {
                        var content = Confluence.Templates.Editor.Reliable.draftMessage({
                            existingDraft: data.draftData,
                            conflictFound: data.conflictFound,
                            mergeRequired: data.mergeRequired,
                            isNewPage: data.newPage,
                            pageId: AJS.Meta.get("page-id"),
                            spaceKey: AJS.Meta.get("space-key")

                        });
                        Message.handleMessage("draft-message", {
                            type: "info",
                            message: content
                        }, bindEventsToDraftDialog);
                    }
                }
            });
        }

        //This is only a temporary solution for legacy drafts: CONFDEV-37357
        //When reliable save is enabled, after saving page with conflicts case and automatically falling back to xwork actions,
        //if there is any action error messages displayed (like token validation...), we restore the default save mechanism (form submit => go through xwork action)
        //for save button to ensure merging conflict will be handled properly after that if user click save again.
        var isActionErrorMessagesDisplayed = $("#editor-notifications-container #all-messages .aui-message-error").length > 0;

        if (!isSharedDraftsEnabled()) {
            if (isActionErrorMessagesDisplayed) {
                Confluence.Editor.restoreDefaultSave();
            } else {
                Confluence.Editor.overrideSave(onSave);
            }
        } else {
            QuitDialog.init({saveHandler: onSave});
            Confluence.Editor.overrideSave(QuitDialog.process);
        }

        $('#rte-button-overwrite').bind("click.overwrite", onSave);

        var retries = 0;
        var MAX_RETRIES = 3;
        var RETRY_DELAY = 1000;
        var retrySaveTimeouts = [];

        function clearAllTimeouts() {
            while (retrySaveTimeouts.length) {
                clearTimeout(retrySaveTimeouts.shift());
            }
        }

        function onSave(e) {
            AJS.trigger('synchrony.stop', {id: PUBLISH});

            if (e) {
                e.preventDefault();
            }

            function computeParentData() {
                var parentData = {};
                parentData.id = AJS.Meta.get("parent-page-id") || "0";
                parentData.type = AJS.Meta.get("content-type");

                var parentPageIdInput = AJS.Meta.get('parent-page-id');

                //If space has been changed
                if (!parentPageIdInput || jsonData.space.key !== AJS.Meta.get("space-key")) {
                    parentData.id = "0";
                }

                //If parent page id is not empty and has been changed
                if (parentPageIdInput && parentData.id !== parentPageIdInput) {
                    parentData.id = ($("#parentPageString").val() === AJS.Meta.get("from-page-title"))
                            ? parentData.id
                            : parentPageIdInput;
                }

                return parentData;
            }

            // Run any cleanup functions that have been registered.
            function cleanupContent(content) {
                cleanupFunctions.forEach(function (cleanupFunc) {
                    content = cleanupFunc(content);
                });

                return content;
            }

            function retrySave(xhr) {
                if (retries < MAX_RETRIES) {
                    retries++;

                    //remember timeout ID so we can clear all timeout later
                    retrySaveTimeouts.push(setTimeout(function () {
                        onSave();
                    }, RETRY_DELAY));
                } else {
                    // TODO SHARED-DRAFTS. Once content reconciliation is working we could ask the server to retry
                    // reconciliation at this point.
                    AJS.trigger('analyticsEvent', {name: 'editor.save.error.conflict'});
                    retries = 0;

                    Message.handleMessage("page-conflict", {
                        title: AJS.I18n.getText("editor.page.conflict.title"),
                        type: "error",
                        message: AJS.I18n.getText("editor.page.conflict.message", jsonData.space.key)
                    }, bindEventsToDraftDialog);

                    afterFailedSaveAttempt(xhr);
                }
            }

            function afterFailedSaveAttempt(xhr) {
                AJS.trigger("rte.safe-save.error", {status: xhr.status});
            }

            var $contentTitle = $("#content-title");
            if ($contentTitle.hasClass("placeholded") || $contentTitle.val().trim() === "") {
                AJS.trigger("rte.safe-save.error");
                AJS.trigger('synchrony.start', {id: PUBLISH});

                Message.closeMessages(["title-too-long", "duplicate-title"]);
                Message.handleMessage("empty-title", {
                    title: AJS.I18n.getText("editor.title.is.empty.title"),
                    type: "error",
                    message: AJS.I18n.getText("editor.title.is.empty")
                }, bindEventsToDraftDialog);
                enableBar();
                return;
            }

            Confluence.Editor.Drafts.unBindUnloadMessage();

            var action;
            var jsonData = {};
            var draftId = AJS.Meta.get('draft-id');
            var contentId = AJS.Meta.get('content-id');
            var url = CONSTANTS.CONTEXT_PATH + "/rest/api/content";
            var sourceTemplateId = $("#sourceTemplateId").val();

            jsonData.status = "current";
            jsonData.title = $contentTitle.val();
            jsonData.space = {key: AJS.Meta.get('space-key')};
            jsonData.body = {
                editor: {
                    value: cleanupContent(AJS.Rte.getEditor().getContent()),
                    representation: "editor"
                }
            };

            if (sourceTemplateId) {
                jsonData.extensions = {sourceTemplateId : sourceTemplateId};
            }

            var datePicker = Confluence.Editor.UI.postingDatePicker;
            // datePicker.getDate() could be undefined in case the date isn't changed
            var publishedDate = datePicker && datePicker.getDate();

            if (draftData.type === "blogpost" && publishedDate && publishedDate.toISOString) {
                jsonData.history = {
                    "createdDate": publishedDate.toISOString() // supported by all major browsers and IE9
                };
            }

            //@Deprecated: `POST` is issued only for legacy drafts
            if (isNewPage() && !AJS.Meta.get("shared-drafts")) {
                action = "POST";

                if (AJS.Meta.get("is-blueprint-page")) {
                    url = url + "/blueprint/instance/" + draftId;
                }

                url = url + "?status=draft";

                jsonData.id = draftId;
                jsonData.type = AJS.Meta.get('content-type');
                jsonData.body.editor.content = {id: draftId};
            } else {
                action = "PUT";

                if (isNewPage() && AJS.Meta.get("is-blueprint-page")) {
                    url = url + "/blueprint/instance";
                }

                url = url + "/" + contentId;

                //`draft-id` is content id with `draft` status for shared drafts
                if (isNewPage() && isSharedDraftsEnabled()) {
                    jsonData.id = draftId;
                    jsonData.body.editor.content = {id: draftId};
                } else {
                    jsonData.id = AJS.Meta.get('page-id');
                    jsonData.body.editor.content = {id: AJS.Meta.get('page-id')};
                }

                //if there is no draft at the time save button is clicked, we update content directly
                if (draftId === "0") {
                    url = url + "?status=current";
                } else {
                    url = url + "?status=draft";
                }

                var versionNumber = AJS.Meta.getNumber("page-version") || 0;

                jsonData.type = AJS.Meta.get('content-type');
                jsonData.version = {
                    number: versionNumber + 1,
                    message: $("#versionComment").val(),
                    minorEdit: $("#notifyWatchers").is(":checked") ? false : true,
                    syncRev: $("#syncRev").val()
                };
            }

            var parentData = computeParentData();
            if (parentData.id !== "0") {
                jsonData.ancestors = [parentData];
            }

            $.ajax({
                type: action,
                url: url,
                contentType: "application/json; charset=utf-8",
                dataType: "json",
                data: JSON.stringify(jsonData),
                success: function (data) {
                    if (jsonData.type === 'page' || jsonData.type === 'blogpost') {
                        AJS.trigger('analytics', {name: "confluence.editor.close", data: {source: "publishButton"}});
                    }
                    AJS.trigger("rte.safe-save.success", data);
                    SafeSave._internal.onSuccessfulResponse(data);
                },
                error: function (xhr) {
                    var retrying = false;
                    enableBar();
                    switch (xhr.status) {
                        case 400:
                            Message.closeMessages(["empty-title", "duplicate-title", "title-too-long", "legacy-draft-deprecated"]);

                            if (xhr.responseText.indexOf(allowedSaveErrorMessage.duplicatedTitle) >= 0) {
                                Message.handleMessage("duplicate-title", {
                                    type: "error",
                                    message: AJS.I18n.getText("editor.offline.save.page.exist.content", AJS.escapeHtml($("#content-title").val()))
                                });
                            } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.titleTooLong) >= 0) {
                                Message.handleMessage("title-too-long", {
                                    type: "error",
                                    message: AJS.I18n.getText("editor.title.too.long")
                                }, bindEventsToDraftDialog);
                            } else if (
                                    xhr.responseText.indexOf(allowedSaveErrorMessage.publishNewDraftDeprecated) >= 0  //TODO SHARED DRAFTS Should be removed once we 100% collab
                                    || xhr.responseText.indexOf(allowedSaveErrorMessage.existingDraftNotFound) >= 0) {

                                // CONFDEV-47278 in case extra content is added after CE enabled, we can't migrate them to the new shared draft
                                // so we should ask user to copy their content and refresh the editor instead.
                                if (AJS.Meta.get('new-page') && isSharedDraftsEnabled() && !Confluence.Editor.hasContentChanged()) {
                                    AJS.trigger("rte.legacy-draft-can-be-migrated");

                                    Message.handleMessage("legacy-draft-deprecated", {
                                        type: "error",
                                        message: AJS.I18n.getText("editor.draft.can.be.migrated",
                                                CONSTANTS.CONTEXT_PATH + '/pages/resumedraft.action?draftId=' + AJS.Meta.get('draft-id'))
                                    }, bindEventsToDraftDialog);
                                } else {
                                    AJS.trigger("rte.legacy-draft-cannot-be-migrated");

                                    Message.handleMessage("legacy-draft-deprecated", {
                                        type: "error",
                                        message: AJS.I18n.getText("editor.page.legacy.draft.deprecated")
                                    }, bindEventsToDraftDialog);
                                }
                            } else {
                                displayGenericSaveMessage(xhr);
                            }
                            break;
                        case 403:
                            displayNoAuthorizedSaveMessage(true);
                            break;
                        case 404:
                            Message.closeMessages(["page-not-accessible", "noauthorized"]);
                            Message.handleMessage("page-not-accessible", {
                                title: AJS.I18n.getText("editor.page.can.not.be.accessed.title"),
                                type: "error",
                                message: AJS.I18n.getText("editor.page.can.not.be.accessed.message", jsonData.space.key)
                            }, bindEventsToDraftDialog);
                            break;
                        case 409:
                            if (isSharedDraftsEnabled()) {
                                retrying = true;
                                retrySave(xhr);
                            } else {
                                //In case of having conflicts when saving, restore the default save mechanism (form submit)
                                //then rely on XworkAction for displaying error message and merging conflicts because of
                                //at the moment ContentAPI doesn't do the work merging conflicts, it only throws ConflictException
                                Confluence.Editor.restoreDefaultSave();
                                Confluence.Editor.UI.saveButton.click();
                            }
                            break;
                        case 410:
                            Message.closeMessages(["page-deleted"]);
                            Message.handleMessage("page-deleted", {
                                title: AJS.I18n.getText("editor.page.is.trashed.title"),
                                type: "error",
                                message: AJS.I18n.getText("editor.page.is.trashed.message", jsonData.space.key)
                            }, bindEventsToDraftDialog);
                            break;
                        case 413:
                            Message.closeMessages(["page-too-big"]);
                            Message.handleMessage("page-too-big", {
                                type: "error",
                                message: AJS.I18n.getText("editor.page.is.too.big")
                            }, bindEventsToDraftDialog);
                            break;
                        case 0:
                        case 500:
                        case 503:
                            //CONFDEV-43951
                            clearAllTimeouts();
                            displayServerOfflineSaveMessage();
                            break;
                        case 501:
                            if (xhr.responseText.indexOf(allowedSaveErrorMessage.renameDuringLimitedMode) >= 0) {
                                Message.handleMessage("rename-during-limited-mode", {
                                    type: "error",
                                    message: AJS.I18n.getText("editor.title.rename.during.limited.mode")
                                }, bindEventsToRestoreFlag);
                            } else {
                                displayGenericSaveMessage(xhr);
                            }
                            break;
                        default:
                            displayGenericSaveMessage(xhr);
                            break;
                    }
                    if (!retrying) {
                        afterFailedSaveAttempt(xhr);
                    }
                    AJS.trigger('synchrony.start', {id: PUBLISH});
                }
            });

        }

        // -----------------------------
        // Heartbeat binding. Heartbeating takes care of enabling/disabling the save bar.
        // -----------------------------

        AJS.bind("rte.heartbeat-error", function (sender, err) {
            switch (err.status) {
                case 401:
                case 403:
                    if (!Message.isDisplayed(["page-not-accessible"])) {
                        displayNoAuthorizedSaveMessage(false);
                    }
                    break;
                case 0:
                case 404:
                case 500:
                case 503:
                    if (!Confluence.Editor.metadataSyncRequired()) {
                        displayServerOfflineSaveMessage();
                    }
                    break;
                default :
                    AJS.logError("Heartbeat action error: " + JSON.stringify(err));
            }
        });

        AJS.bind("rte.heartbeat", function (sender) {
            var isDisplayed = false;
            _.each(Message.displayedErrors(), function (error) {
                if (_.contains(["noauthorized", "server-offline", "page-not-accessible"], error)) {
                    isDisplayed = true;
                    Message.closeMessages([error]);
                }
            });

            if (isDisplayed) {
                Message.handleMessage("reconnect", {
                    type: "info",
                    title: AJS.I18n.getText("editor.offline.save.info.title"),
                    message: AJS.I18n.getText("editor.offline.save.info.content"),
                    close: "auto"
                }, bindEventsToDraftDialog);
            }
        });
    };

    SafeSave.registerCleanupFunction = function (cleanupFunction) {
        cleanupFunctions.push(cleanupFunction);
    };

    return SafeSave;
});

require('confluence/module-exporter').safeRequire('confluence-editor-reliable-save/reliable-save', function (SafeSave) {
    var AJS = require('ajs');

    //CONFDEV-38205
    var isCreatingPageFromRoadMapBar = window.document.referrer.indexOf('createDialog=true&flashId') > 0;

    if (!AJS.DarkFeatures.isEnabled('editor.ajax.save') ||
            AJS.Meta.get('remote-user') === '' ||
            isCreatingPageFromRoadMapBar) {
        // DO NOT BIND MODULE OR EXPOSE GLOBALS
    } else {
        var Confluence = require('confluence/legacy');

        AJS.bind("rte.init.ui", function () {
            SafeSave.initialize();

            Confluence.Editor = Confluence.Editor || {};
            Confluence.Editor.SafeSave = Confluence.Editor.SafeSave || {};
            Confluence.Editor.SafeSave.Draft = SafeSave.Draft;
            Confluence.Editor.SafeSave._internal = SafeSave._internal || {};
        });
    }
});
