define('confluence-dashboard/utils/event-manager', ['exports', 'module', 'backbone', 'marionette'], function (exports, module, _backbone, _marionette) {
	'use strict';

	// Refer to https://github.com/marionettejs/backbone.wreqr
	// for more details of usage

	// for now we have only one channel
	var globalChannel = _backbone.Wreqr.radio.channel('global');

	module.exports = {
		EventManager: globalChannel.vent,
		Commands: globalChannel.commands,
		ReqRes: globalChannel.reqres
	};
});

// importing marionette here because karma isn't loading Wreqr correctly;
define('confluence-dashboard/behaviors/aui-sidebar-resizable', ['exports', 'module', 'marionette', '../utils/event-manager', 'jquery', 'ajs', 'confluence/storage-manager'], function (exports, module, _marionette, _utilsEventManager, _jquery, _ajs, _confluenceStorageManager) {
    'use strict';

    var SIDEBAR_WIDTH_KEY = 'width';
    var COLLAPSED_SIZE = 55;
    var DEFAULT_SIZE = 280;

    var TOGGLE_SIDEBAR_WIDTH = 130;
    var MAXIMUM_SIDEBAR_WIDTH = 640;
    var MINIMUM_SIDEBAR_WIDTH = COLLAPSED_SIZE;

    // this value is the same we have on AUI sidebar.
    var FORCE_COLLAPSE_WIDTH = 1240;

    module.exports = _marionette.Behavior.extend({
        ui: {
            resizeHandle: '.aui-sidebar-handle'
        },

        events: {
            'mousedown @ui.resizeHandle': 'onResizeStart',
            'mouseup @ui.resizeHandle': 'onResizeEnd'
        },

        dragging: false,

        initialize: function initialize() {
            this.EXPANDED_SIZE = DEFAULT_SIZE;
            this.sidebarSettings = (0, _confluenceStorageManager)('confluence', 'sidebar');
            this.listenTo(this.view, 'sidebar-attached', this.createAuiSidebar, this);
            this.listenTo(_utilsEventManager.EventManager, 'window:resize', this.resetSidebar, this);
        },

        createAuiSidebar: function createAuiSidebar() {
            var _this = this;

            // central event to fire sidebar changes
            this.AUISidebar = _ajs.sidebar(this.view.$el.selector);

            this.AUISidebar.on('expand-start.sidebar', function () {
                return _this.onExpand();
            });
            this.AUISidebar.on('collapse-start.sidebar', function () {
                return _this.onCollapse();
            });

            this.$footer = (0, _jquery)('#footer, #studio-footer');

            this.currentWidth = parseInt(this.sidebarSettings.getItem(SIDEBAR_WIDTH_KEY));

            this.setInitialState();
        },

        resetSidebar: function resetSidebar() {
            var shouldAutoCollapse = (0, _jquery)('body').hasClass('aui-sidebar-collapsed');
            var sidebarWasOpenedWhenCollapsed = this.AUISidebar.$el.hasClass('aui-sidebar-fly-out');

            if (shouldAutoCollapse && !sidebarWasOpenedWhenCollapsed) {
                this.AUISidebar.collapse();
            } else if (!this.isFlyOut() && !this.AUISidebar.isCollapsed()) {
                // if the page is wide enough to not be in fly-out mode, and the sidebar is expanded,
                // we need to make sure the panel padding gets changed to make the sidebar appear "docked"
                // if we don't do this, the sidebar remains in flyout mode if you re-widen the window.
                this.setPanelWidth(this.EXPANDED_SIZE, this.EXPANDED_SIZE);
            }
        },

        setInitialState: function setInitialState() {
            if (this.currentWidth) {
                if (this.currentWidth > TOGGLE_SIDEBAR_WIDTH) {
                    this.EXPANDED_SIZE = this.currentWidth;
                }

                this.adjustSize();
            } else {
                this.currentWidth = DEFAULT_SIZE;
            }
        },

        onBeforeDestroy: function onBeforeDestroy() {
            this.AUISidebar.off('.sidebar');
        },

        triggerToggleSidebar: function triggerToggleSidebar(event) {
            _utilsEventManager.EventManager.trigger('sidebar:' + event);
        },

        isFlyOut: function isFlyOut() {
            return FORCE_COLLAPSE_WIDTH > window.innerWidth;
        },

        onExpand: function onExpand() {
            this.triggerToggleSidebar('expand');
            this.setPanelWidth(this.EXPANDED_SIZE, this.EXPANDED_SIZE);
            this.persistSidebarSize();
        },

        onCollapse: function onCollapse() {
            // if we collapses the sidebar after a resize, we should be able to return to the previous size
            // if the previous size is less than the TOGGLE_SIDEBAR_WIDTH we will just use the DEFAULT_SIZE
            this.EXPANDED_SIZE = this.currentWidth < TOGGLE_SIDEBAR_WIDTH ? DEFAULT_SIZE : this.currentWidth;

            this.triggerToggleSidebar('collapse');
            this.setPanelWidth(COLLAPSED_SIZE, 'inherit');
            this.persistSidebarSize();
        },

        onResizeStart: function onResizeStart(e) {
            var _this2 = this;

            e.preventDefault();

            this.AUISidebar.$body.on('mousemove.sidebar', function (e) {
                var currentWidth = e.pageX;

                _this2.dragging = true;

                if (currentWidth > TOGGLE_SIDEBAR_WIDTH) {
                    _this2.EXPANDED_SIZE = currentWidth;
                }

                _this2.adjustSize();

                if (currentWidth > MINIMUM_SIDEBAR_WIDTH && currentWidth < MAXIMUM_SIDEBAR_WIDTH) {
                    _this2.setPanelWidth(currentWidth);
                }
            });

            this.AUISidebar.$body.one('mouseup.sidebar', function () {
                return _this2.onResizeEnd();
            });
        },

        onResizeEnd: function onResizeEnd() {
            this.AUISidebar.$body.off('.sidebar');

            if (this.dragging) {
                this.adjustSize();
                this.persistSidebarSize();
                this.dragging = false;
            } else {
                // if it's not dragging, it's a click and should toggle
                this.AUISidebar.toggle();
            }
        },

        setPanelWidth: function setPanelWidth(width, maxWidth) {
            var widthPx = width + 'px';
            var paddingPx = widthPx;

            if (this.isFlyOut()) {
                paddingPx = COLLAPSED_SIZE + 'px';
            }

            this.AUISidebar.$wrapper.css({
                'width': widthPx,
                'maxWidth': maxWidth || widthPx
            });

            this.AUISidebar.$el.siblings('.aui-page-panel').css({
                'paddingLeft': paddingPx
            });

            this.$footer.css('padding-left', paddingPx);

            this.currentWidth = width;
        },

        persistSidebarSize: function persistSidebarSize() {
            this.sidebarSettings.setItemQuietly(SIDEBAR_WIDTH_KEY, this.currentWidth);
        },

        _collapse: function _collapse() {
            if (!this.AUISidebar.isCollapsed()) {
                this.AUISidebar.collapse();
            } else {
                // if the sidebar is not expanded, collapse won't fire, so we need to fire it manually.
                this.onCollapse();
            }
        },

        _expand: function _expand() {
            if (this.AUISidebar.isCollapsed()) {
                this.AUISidebar.expand();
            } else {
                // if the sidebar is not collapsed, expand won't fire, so we need to fire it manually.
                this.onExpand();
            }
        },

        adjustSize: function adjustSize() {
            if (this.currentWidth < TOGGLE_SIDEBAR_WIDTH) {
                this._collapse();
            } else {
                this._expand();
            }
        }
    });
});
define('configuration', ['ajs'], function(AJS) {
	var contextPath = AJS.contextPath();
	var pluginKey = 'com.atlassian.confluence.plugins.confluence-dashboard';
    var assetsResourceKey = 'confluence-dashboard-resources';

	return {

        // the 'limit' parameter to send to the server on each page request
		apiLimit: 20,

        visibleItemLimit: 200,

        pluginKey: pluginKey,

		staticResourceUrl: AJS.Meta.get('static-resource-url-prefix') + '/download/resources/' + pluginKey + ':' + assetsResourceKey,

		backboneHistoryConfig: {
			pushState: false,
			root: contextPath + '/dashboard.action',
			silent: false
		},

        // DARK FEATURES
        DARK_FEATURES: {
            USER_DISABLED_DASHBOARD_DARK_FEATURE: 'simple.dashboard.disabled',
			RECOMMENDED_FOR_YOU_DARK_FEATURE: 'recommended.for.you'
        },

		// ENDPOINTS
        endpoints: {
			ALL_UPDATES: contextPath + '/rest/dashboardmacros/1.0/updates',

			POPULAR_STREAM: contextPath + '/rest/popular/1/stream/content?days=7&pageSize=20',

			RECOMMENDED_STREAM: contextPath + '/rest/recommender/1.0/stream/content',

			RECENTLY_VIEWED: contextPath + '/rest/api/content/search',

			RECENTLY_WORKED: contextPath + '/rest/api/content/search',

			STARRED: contextPath + '/rest/api/content/search',

			FAVOURITE_SPACES: contextPath + '/rest/experimental/search?cql=type=space and space.type=favourite order by favourite desc&expand=space.icon&limit=100',

			ALL_SPACES: contextPath + '/rest/experimental/search?cql=type = space&expand=space.icon',

			TOGGLE_FAVOURITE_SPACE: function(key) {
				return contextPath + '/rest/experimental/relation/user/current/favourite/toSpace/' + key;
			},

			WEB_ITEMS: function(section) {
                return AJS.Meta.get('static-resource-url-prefix') + '/download/resources/com.atlassian.confluence.plugins.confluence-dashboard:confluence-dashboard-nav-items/api/web-items/' + section + '.json';
			},

			ADD_FAVOURITE: contextPath + '/json/addfavourite.action',

			REMOVE_FAVOURITE: contextPath + '/json/removefavourite.action',

            FEATURE_DISCOVERY: contextPath + '/rest/feature-discovery/latest/discovered'
		},

		sections: {
			webItems: {
				sidebar: {
					MY_WORK: 'my-work',
					DISCOVER: 'discover',
					MY_SPACES: 'my-spaces',
					MY_TEAM: 'my-team'
				}
			}
		},

		URLS: {
            LOGIN_PAGE: contextPath + '/login.action'
        }
	};
});

define('confluence-dashboard/utils/dark-features', ['exports', 'module', 'confluence/dark-features', 'confluence/api/ajax', 'ajs', 'jquery'], function (exports, module, _confluenceDarkFeatures, _confluenceApiAjax, _ajs, _jquery) {
    'use strict';

    var contextPath = _ajs.contextPath();

    function remoteDarkFeature(action, type, key) {
        var actions = {
            'enable': 'PUT',
            'disable': 'DELETE'
        };

        return _confluenceApiAjax.ajax({
            type: actions[action],
            contentType: 'application/json',
            url: contextPath + '/rest/feature/1/' + type + '/' + key,
            data: {} // ajax module needs it
        });
    }

    module.exports = _jquery.extend(_confluenceDarkFeatures, {
        remotely: {
            user: {
                enable: function enable(key) {
                    _confluenceDarkFeatures.enable(key);
                    return remoteDarkFeature('enable', 'user', key);
                },

                disable: function disable(key) {
                    _confluenceDarkFeatures.disable(key);
                    return remoteDarkFeature('disable', 'user', key);
                }
            }
        }
    });
});
define('confluence-dashboard/modules/nav/nav-definitions', ['exports', 'module', 'ajs', '../../utils/dark-features', 'configuration'], function (exports, module, _ajs, _utilsDarkFeatures, _configuration) {
    'use strict';

    var RECOMMENDED_FOR_YOU_DARK_FEATURE = _configuration.DARK_FEATURES.RECOMMENDED_FOR_YOU_DARK_FEATURE;

    var MY_WORK = [{
        "key": "recently-worked",
        "url": "recently-worked",
        "label": _ajs.I18n.getText("recently.worked.title"),
        "icon": "icon-recently-worked-on",
        "controllerModule": "confluence-dashboard/modules/recently-worked/recently-worked-controller",
        "controllerMethod": "recentWorked",
        "spa": true
    }, {
        "key": "recently-viewed",
        "url": "recently-viewed",
        "label": _ajs.I18n.getText("recently.viewed.title"),
        "icon": "icon-recently-viewed",
        "controllerModule": "confluence-dashboard/modules/recently-viewed/recently-viewed-controller",
        "controllerMethod": "recentViewed",
        "spa": true
    }, {
        "key": "starred",
        "url": "starred",
        "label": _ajs.I18n.getText("starred.title"),
        "icon": "icon-starred",
        "controllerModule": "confluence-dashboard/modules/starred/starred-controller",
        "controllerMethod": "starred",
        "spa": true
    }];

    var DISCOVER = [{
        "key": "all-updates",
        "url": "all-updates",
        "icon": "icon-all-updates",
        "label": _ajs.I18n.getText('all.updates.title'),
        "controllerModule": "confluence-dashboard/modules/all-updates/all-updates-controller",
        "controllerMethod": "allUpdates",
        "spa": true
    }, {
        "key": "popular-stream",
        "url": "popular",
        "icon": "icon-popular-stream",
        "label": _ajs.I18n.getText('popular.stream.title'),
        "controllerModule": "confluence-dashboard/modules/popular-stream/popular-stream-controller",
        "controllerMethod": "popularStream",
        "spa": true
    }];

    if (_utilsDarkFeatures.isEnabled(RECOMMENDED_FOR_YOU_DARK_FEATURE)) {
        DISCOVER.push({
            "key": "recommended-stream",
            "url": "recommended",
            "icon": "icon-recommended-stream",
            "label": "Recommended for you",
            "controllerModule": "confluence-dashboard/modules/recommended-stream/recommended-stream-controller",
            "controllerMethod": "recommendedStream",
            "spa": true
        });
    }

    module.exports = { MY_WORK: MY_WORK, DISCOVER: DISCOVER };
});
define('confluence-dashboard/soy-templates', ['exports', 'module'], function (exports, module) {
  // just a shim
  'use strict';

  module.exports = window['DashboardTemplates'];
});
define('confluence-dashboard/behaviors/tooltips', ['exports', 'module', 'marionette'], function (exports, module, _marionette) {
    'use strict';

    /**
     * Tooltips
     *
     * Add this behaviour to the view when you want a item to show tooltips
     *
     * Params:
     *  - selector: the element which should show the tooltip
     *  - configs: configs to be sent to $.tipsy
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *       tooltips: {
     *           behaviorClass: Tooltip,
     *           selector: '.tooltip-top',
     *           configs: {
     *               gravity: 'n'
     *           }
     *       }
     *   }
     * })
     *
     * TODO: write tests
     */

    module.exports = _marionette.Behavior.extend({
        defaults: {
            selector: '.tooltip',
            configs: {
                gravity: 's'
            }
        },

        onRender: function onRender() {
            this.$(this.options.selector).tooltip(this.options.configs);
        },

        onBeforeDestroy: function onBeforeDestroy() {
            // make sure we remove the tooltip so it won't be
            // floating forever alone in the screen
            this.$(this.options.selector).tooltip('hide');
        }
    });
});
define("confluence-dashboard/behaviors/tooltips.js", function(){});

define('confluence-dashboard/modules/nav/nav-item-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', '../../utils/event-manager', 'backbone', '../../behaviors/tooltips'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _utilsEventManager, _backbone, _behaviorsTooltips) {
    'use strict';

    var SELECTED_CLASS = 'aui-nav-selected';

    module.exports = _marionette.ItemView.extend({
        tagName: 'li',
        template: _confluenceDashboardSoyTemplates.Nav.item,
        behaviors: {
            tooltip: {
                behaviorClass: _behaviorsTooltips,
                selector: '.aui-nav-item',
                configs: {
                    gravity: 'w'
                }
            }
        },

        modelEvents: {
            'change selected': 'render'
        },

        events: {
            'click a': 'onClick'
        },

        onClick: function onClick() {
            this.model.set('selected', true);
        },

        initialize: function initialize() {
            this.onChangeRoute();
            this.listenTo(_utilsEventManager.EventManager, 'onRoute', this.onChangeRoute);
        },

        onChangeRoute: function onChangeRoute() {
            this.model.set('selected', this.model.get('url') === _backbone.history.fragment);
        },

        onBeforeRender: function onBeforeRender() {
            this.$el.attr({
                'class': 'nav-item-container-' + this.model.get('key') + ' ' + (this.model.get('selected') ? SELECTED_CLASS : '')
            });
        }
    });
});
define('confluence-dashboard/core/shared/base-collection', ['exports', 'module', 'backbone', 'jquery', 'ajs', 'underscore'], function (exports, module, _backbone, _jquery, _ajs, _underscore) {
    'use strict';

    var context = _ajs.contextPath();

    /**
     * Base-Collection
     * ===============
     *
     * It's a basic Backbone Collection that implements some common functionality
     * for the dashboard app.
     */
    module.exports = _backbone.Collection.extend({
        // Defined just to remember you that you need to implement it on your collection if you want grouping
        groupMethod: function groupMethod(item) {
            // implement your own group method
            throw 'groupMethod method missing - check this collection !!!';
        },

        // safe fetch abort previous requests (created by itself) if they haven't been completed yet to avoid race conditions
        safeFetch: function safeFetch(options) {
            // if there is a previous filterXHR object and its readyState is between 1 and 3, cancel it!
            if (this.filterXHR && /[1-3]/.test(this.filterXHR.readyState)) {
                this.filterXHR.abort();
                this.filterXHR = null;
            }

            this.filterXHR = this.fetch(options);

            return this.filterXHR;
        },

        sync: function sync(method, model, options) {
            // make sure results aren't cached (required for IE 10)
            if (!options.cache) {
                options.cache = false;
            }
            return _backbone.Collection.prototype.sync.apply(this, arguments);
        }
    });
});
define("confluence-dashboard/core/shared/base-collection.js", function(){});

define('confluence-dashboard/core/web-fragments/web-item/web-item-collection', ['exports', 'module', '../../shared/base-collection', 'backbone', 'configuration', 'underscore'], function (exports, module, _sharedBaseCollection, _backbone, _configuration, _underscore) {
	'use strict';

	var WebItemModel = _backbone.Model.extend({
		defaults: {
			weight: 1,
			urlWithoutContextPath: true
		}
	});

	module.exports = _sharedBaseCollection.extend({
		model: WebItemModel,

		setContainer: function setContainer(container) {
			this.container = container;
		},

		url: function url() {
			return _configuration.endpoints.WEB_ITEMS(this.container);
		},

		extractUrls: function extractUrls() {
			return this.invoke('pick', ['url', 'key']);
		}
	});
});
define('confluence-dashboard/core/web-fragments/web-item/web-item-views', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', './web-item-collection'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _webItemCollection) {
	'use strict';

	// represents an individual web-item, i.e: a link
	var WebItemView = _marionette.ItemView.extend({
		tagName: 'li',
		template: _confluenceDashboardSoyTemplates.WebFragments.webItem
	});

	// represents a list of web-items, i.e: a menu
	var WebItemCollectionView = _marionette.CollectionView.extend({
		tagName: 'ul',
		childView: WebItemView,

		initialize: function initialize(options) {
			if (!options.container) {
				throw 'A container should be defined for a web item';
			}

			this.collection = new _webItemCollection();
		}
	});

	// the same as WebItemCollectionView but allows you to add a
	// template for the container
	var WebItemCompositeView = _marionette.CompositeView.extend({
		childView: WebItemView,

		initialize: function initialize(options) {
			if (!options.container) {
				throw 'A container should be defined for a web item';
			}

			this.collection = new _webItemCollection();
		}
	});

	module.exports = { WebItemView: WebItemView, WebItemCollectionView: WebItemCollectionView, WebItemCompositeView: WebItemCompositeView };
});
define('confluence-dashboard/modules/nav/nav-composite-view', ['exports', 'module', 'confluence-dashboard/soy-templates', './nav-item-view', '../../core/web-fragments/web-item/web-item-views'], function (exports, module, _confluenceDashboardSoyTemplates, _navItemView, _coreWebFragmentsWebItemWebItemViews) {
  'use strict';

  module.exports = _coreWebFragmentsWebItemWebItemViews.WebItemCompositeView.extend({
    template: _confluenceDashboardSoyTemplates.Nav.container,
    childViewContainer: '.nav-items',
    className: 'aui-navgroup-inner',
    childView: _navItemView
  });
});
define("confluence-dashboard/utils/logger", ["exports", "module"], function (exports, module) {
  "use strict";

  module.exports = console;
});
define('confluence-dashboard/core/shared/base-controller', ['exports', 'module', 'marionette', 'underscore', '../../utils/logger'], function (exports, module, _marionette, _underscore, _utilsLogger) {
	'use strict';

	/**
  * Base-Controller
  * ===============
  *
  * It's a basic Marionette Object which add the hooks 'before' and 'after'
  * for any action specified at actionsToFilter property. If you don't need
  * filter methods, you don't need to extend this controller :)
  *
  * If you need to reuse any of the components outside of this app, change
  * your controller to extend Marionette.Object or copy this file.
  */

	module.exports = _marionette.Object.extend({
		initialize: function initialize() {
			// setup your views and router when needed
			this.wrapActions();
		},

		// list of actions to be wrapped, should be defined in each controller specifying the actions
		actionsToFilter: [],
		// noop methods to be implemented in each controller
		beforeAction: function beforeAction() {},
		afterAction: function afterAction() {},

		// wrap actions from this.actions including beforeAction and afterAction
		wrapActions: function wrapActions() {
			var _this = this;

			this.actionsToFilter.forEach(function (action) {
				if (!_this[action] || !_underscore.isFunction(_this[action])) {
					throw 'Method \'' + action + '\' not found!';
				}

				var originalMethod = _this[action];

				// wrap all the listed actions
				_this[action] = function () {
					// if beforeAction returns false, it stops the execution,
					// if you need to throw an error or show a dialog, do it
					// inside the beforeAction
					if (this.beforeAction.apply(this, arguments) !== false) {

						// call the original method with the original arguments
						originalMethod.apply(this, arguments);

						// and finally the afterAction
						this.afterAction.apply(this, arguments);
					}
				};
			});
		}
	});
});
define('confluence-dashboard/modules/nav/nav-controller', ['exports', 'module', 'configuration', './nav-definitions', '../../utils/event-manager', './nav-composite-view', '../../core/shared/base-controller', 'ajs', 'jquery'], function (exports, module, _configuration, _navDefinitions, _utilsEventManager, _navCompositeView, _coreSharedBaseController, _ajs, _jquery) {
    'use strict';

    module.exports = _coreSharedBaseController.extend({
        createDiscoverView: function createDiscoverView() {
            var discoverView = this.createDefaultNavContainer({
                container: _configuration.sections.webItems.sidebar.DISCOVER,
                templateHelpers: {
                    title: _ajs.I18n.getText('nav.controller.discover')
                }
            });

            discoverView.collection.add(_navDefinitions.DISCOVER);

            // Don't repeat it at home - quick fix for sidebar bug where the
            // sidebar doesn't fit the whole screen before you scroll.
            // TODO: investigate how to solve it properly
            var fixSidebar = function fixSidebar() {
                (0, _jquery)('body').animate({ scrollTop: 0 });
            };

            discoverView.on('dom:refresh', fixSidebar);

            return discoverView;
        },

        createMyWorkView: function createMyWorkView() {
            var myWorkView = this.createDefaultNavContainer({
                container: _configuration.sections.webItems.sidebar.MY_WORK,
                templateHelpers: {
                    title: _ajs.I18n.getText('nav.controller.my.work')
                }
            });

            myWorkView.collection.add(_navDefinitions.MY_WORK);

            return myWorkView;
        },

        createDefaultNavContainer: function createDefaultNavContainer(menuConfig) {
            var menuView = new _navCompositeView(menuConfig);

            this.listenTo(menuView.collection, 'add', this.bootstrapWebItem, this);

            menuView.collection.setContainer(menuConfig.container);

            return menuView;
        },

        // calls the factory to bind the route
        bootstrapWebItem: function bootstrapWebItem(webItemModel) {
            _utilsEventManager.Commands.execute('app:setupWebItem', webItemModel.toJSON());
        }
    });
});
define('confluence-dashboard/modules/nav-spaces/nav-spaces-collection', ['exports', 'module', '../../core/shared/base-collection', 'backbone', 'configuration', 'ajs'], function (exports, module, _coreSharedBaseCollection, _backbone, _configuration, _ajs) {
	'use strict';

	var contextPath = _ajs.contextPath();

	var SpaceModel = _backbone.Model.extend({
		idAttribute: 'key',

		url: function url() {
			return _configuration.endpoints.TOGGLE_FAVOURITE_SPACE(this.get('key'));
		},

		parse: function parse(data) {
			data.url = contextPath + data.url;
			data.logo = contextPath + data.space.icon.path;
			data.name = data.title;
			data.key = data.space.key;
			return data;
		}
	});

	var FavouriteSpacesCollection = _coreSharedBaseCollection.extend({
		url: _configuration.endpoints.FAVOURITE_SPACES,
		model: SpaceModel,

		parse: function parse(data) {
			return data.results;
		}
	});

	var AllSpacesCollection = FavouriteSpacesCollection.extend({
		url: _configuration.endpoints.ALL_SPACES
	});

	module.exports = { FavouriteSpacesCollection: FavouriteSpacesCollection, AllSpacesCollection: AllSpacesCollection };
});
define('confluence-dashboard/core/shared/base-composite-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', 'configuration', 'underscore'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _configuration, _underscore) {
    'use strict';

    /**
     * Base-Composite-View
     * ===============
     *
     * Marionette only supports emptyView. This view adds basic support for loadingView and noMatchesView.
     *
     * The current implementation is very specific for the dashboard use case, and if
     * needed outside we need to make it more generic.
     *
     * Basically, it will add 'fetched' state and isFiltered method to the views which decide when to show
     * 'emptyView', 'loadingView' or 'noMatchesView'.
     *
     */
    module.exports = _marionette.CompositeView.extend({
        // the view starts with a collection not fetched
        fetched: false,

        // we need to detect the first fetch and mark the collection
        collectionEvents: {
            'sync': 'onCollectionSync'
        },

        getOption: function getOption(name) {
            if (name === 'emptyViewOptions') {
                return this.selectEmptyViewOptions();
            }
            return _marionette.CompositeView.prototype.getOption.call(this, name);
        },

        isFiltered: function isFiltered() {
            return false;
        },

        onCollectionSync: function onCollectionSync() {
            var becameFetched = !this.fetched;

            // after sync we set it as fetched
            this.fetched = true;

            // re-render is required when switching between empty views (loadingView, emptyView, noMatchesView)
            if (becameFetched || this.collection.length === 0) {
                this._renderChildren();
            }
        },

        // Overrides CompositeView.getEmptyView to return the desired empty view
        getEmptyView: function getEmptyView() {
            var emptyViewType = this.selectEmptyViewType();
            var emptyView = this.getOption(emptyViewType);
            if (!emptyView) {
                throw 'Missing ' + emptyViewType;
            }
            return emptyView;
        },

        // Choose the empty view to use
        selectEmptyViewType: function selectEmptyViewType() {
            if (!this.fetched) {
                return 'loadingView';
            } else if (this.isFiltered()) {
                return 'noMatchesView';
            }
            return 'emptyView';
        },

        // Choose the empty view options to use
        selectEmptyViewOptions: function selectEmptyViewOptions() {
            var emptyViewType = this.selectEmptyViewType();
            return _marionette.CompositeView.prototype.getOption.call(this, emptyViewType + "Options");
        }
    });
});
define("confluence-dashboard/core/shared/base-composite-view.js", function(){});

define('confluence-dashboard/core/shared/loading-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', 'ajs', 'underscore'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _ajs, _underscore) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        tagName: 'li',
        className: 'empty-view loading-view',

        getTemplate: function getTemplate() {
            return _confluenceDashboardSoyTemplates.Default.loading;
        }
    });
});
define("confluence-dashboard/core/shared/loading-view.js", function(){});

define('confluence-dashboard/utils/analytics', ['exports', 'module', 'ajs', 'underscore', 'backbone'], function (exports, module, _ajs, _underscore, _backbone) {
    'use strict';

    var ANALYTICS_DASHBOARD_PREFIX = 'confluence.spa';

    module.exports = {
        publish: function publish(name) {
            var data = arguments.length <= 1 || arguments[1] === undefined ? {} : arguments[1];

            var tagName = ANALYTICS_DASHBOARD_PREFIX + '.' + name;

            data.currentSection = _backbone.history.fragment;

            _ajs.trigger('analytics', { name: tagName, data: data });
            // debug options to help making the whitelist
            //console.log(`"${tagName}":[],`);
            //console.log(`"${tagName}":[
            // _.keys(data)
            // ],`);
        }
    };
});
define('confluence-dashboard/behaviors/undo-remove', ['exports', 'module', 'marionette', 'jquery', '../utils/analytics'], function (exports, module, _marionette, _jquery, _utilsAnalytics) {
    'use strict';

    /**
     * Undo Remove
     *
     * Add this behaviour to the view if you want to keep it visible after removing.
     *
     * It will call the model.destroy but won't remove from the screen so you can add
     * it again simulating an 'undo' action.
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *       undoRemove: {
     *           behaviorClass: UndoRemove
     *       }
     *   }
     * })
     *
     * TODO: write tests
     */

    var $win = (0, _jquery)(window);
    var ON_UNLOAD_EVENT = 'beforeunload.remove';

    module.exports = _marionette.Behavior.extend({
        defaults: {
            timeToDestroy: 3000 // 10000
        },

        ui: {
            'remove': '.remove',
            'undo': '.undo-remove'
        },

        events: {
            'click @ui.remove': 'virtualRemove',
            'click @ui.undo': 'undoVirtualRemove'
        },

        modelEvents: {
            'change:virtuallyDeleted': 'onVirtualDelete'
        },

        onVirtualDelete: function onVirtualDelete() {
            this.view.render();
        },

        virtualRemove: function virtualRemove() {
            var _this = this;

            this.view.model.set('virtuallyDeleted', true);

            // set the model to be destroyed in a moment
            this.timeout = setTimeout(function () {
                _this.sendRemove();
            }, this.options.timeToDestroy);

            this.bindToUnload();

            _utilsAnalytics.publish(this.options.eventNamespace + '.remove-favourite.clicked');
        },

        undoVirtualRemove: function undoVirtualRemove() {
            // avoid the model destruction
            clearTimeout(this.timeout);

            this.unbindFromUnload();

            this.view.model.set('virtuallyDeleted', false);

            _utilsAnalytics.publish(this.options.eventNamespace + '.remove-favourite.undo');
        },

        // as the action happens after a few seconds, we need to make sure that if the user
        // leave the page before the time the action still happens.
        bindToUnload: function bindToUnload() {
            var _this2 = this;

            var id = this.view.model.get('id');

            // the event has a unique identifier so we can remove it if needed
            $win.on(ON_UNLOAD_EVENT + '.' + id, function () {
                _this2.sendRemove();

                // we return empty so the browser won't confirm anything
                return;
            });
        },

        // clean up to be called if undo or timeout happens
        unbindFromUnload: function unbindFromUnload() {
            var id = this.view.model.get('id');

            $win.off(ON_UNLOAD_EVENT + '.' + id);
        },

        sendRemove: function sendRemove() {
            this.view.model.destroy();
            this.unbindFromUnload();
        }
    });
});
define('confluence-dashboard/behaviors/list-item-animated', ['exports', 'module', 'marionette', '../utils/event-manager', 'jquery', 'underscore'], function (exports, module, _marionette, _utilsEventManager, _jquery, _underscore) {
    'use strict';

    /**
     * ListItemAnimated
     *
     * Add this behaviour when you want to animate list items
     *
     * It will delay the destroy method
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *       listItemAnimated: {
     *           behaviorClass: ListItemAnimated
     *       }
     *   }
     * })
     *
     * TODO: write tests
     */

    module.exports = _marionette.Behavior.extend({
        initialize: function initialize() {
            this.overrideViewDestroyMethod();
        },

        onRender: function onRender() {
            this.$el.addClass('animated show');
        },

        overrideViewDestroyMethod: function overrideViewDestroyMethod() {
            var self = this;

            // override view's destroy method to wait for the animation
            this.view.destroy = function () {
                var _this = this,
                    _arguments = arguments;

                if (this.isDestroyed) {
                    return this;
                }

                var originalDestroy = _marionette.View.prototype.destroy;
                var runAnimation = self.onBeforeDestroyWithAnimation();

                runAnimation.then(function () {
                    return originalDestroy.apply(_this, _arguments);
                }).fail(function () {
                    return originalDestroy.apply(_this, _arguments);
                });
            };

            // keep the scope bound to the view instance
            this.view.destroy.bind(this.view);
        },

        onBeforeDestroyWithAnimation: function onBeforeDestroyWithAnimation() {
            var _this2 = this;

            var defer = _jquery.Deferred();

            var transitionEnd = 'webkitTransitionEnd otransitionend oTransitionEnd msTransitionEnd transitionend';

            // resolve promises on transition end
            this.view.$el.one(transitionEnd, function () {
                defer.resolve();
                _utilsEventManager.EventManager.trigger('list:change', _this2.view);
            });

            // the class removing should be associated to an animation
            this.view.$el.removeClass('show').addClass('removing');

            // reject promise if nothing happen in 10 seconds
            setTimeout(function () {
                if (!defer.isResolved) {
                    defer.reject('time-out');
                }
            }, 10000);

            return defer.promise();
        }

    });
});
define("confluence-dashboard/behaviors/list-item-animated.js", function(){});

define('confluence-dashboard/behaviors/analytics-tracking', ['exports', 'module', 'marionette', '../utils/analytics', 'backbone', 'underscore'], function (exports, module, _marionette, _utilsAnalytics, _backbone, _underscore) {
    'use strict';

    /**
     * ListItemAnimated
     *
     * Add this behaviour when you want to animate list items
     *
     * It will delay the destroy method
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *       analyticsTracking: {
     *           behaviorClass: AnalyticsTraking
     *       }
     *   }
     * })
     */

    module.exports = _marionette.Behavior.extend({
        defaults: {
            prefix: ''
        },

        events: {
            'click a': 'trackItemClick'
        },

        trackItemClick: function trackItemClick() {
            var eventName = 'item.clicked';

            if (this.options.prefix) {
                eventName = this.options.prefix + '.' + eventName;
            }

            _utilsAnalytics.publish(eventName);
        }
    });
});
define('confluence-dashboard/utils/conditions', ['exports', 'module', 'confluence/meta', 'ajs'], function (exports, module, _confluenceMeta, _ajs) {
    'use strict';

    module.exports = {
        canShowDashboard: function canShowDashboard() {
            var simplifyDashboardPresent = document.querySelector('.confluence-dashboard');
            if (!simplifyDashboardPresent) {
                _ajs.log("Simplify dashboard disabled.", { simplifyDashboardPresent: simplifyDashboardPresent });
            }
            return simplifyDashboardPresent;
        },

        isAnonymousUser: function isAnonymousUser() {
            return !_confluenceMeta.get('remote-user');
        }
    };
});
define('confluence-dashboard/modules/nav-spaces/nav-spaces-item-view', ['exports', 'module', 'marionette', '../../behaviors/undo-remove', '../../behaviors/list-item-animated', '../../behaviors/tooltips', '../../behaviors/analytics-tracking', 'confluence-dashboard/soy-templates', '../../utils/conditions'], function (exports, module, _marionette, _behaviorsUndoRemove, _behaviorsListItemAnimated, _behaviorsTooltips, _behaviorsAnalyticsTracking, _confluenceDashboardSoyTemplates, _utilsConditions) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        template: _confluenceDashboardSoyTemplates.NavSpaces.space,

        tagName: 'li',

        className: 'item',

        templateHelpers: function templateHelpers() {
            return {
                isAnonymousUser: _utilsConditions.isAnonymousUser()
            };
        },

        behaviors: {
            undoRemove: {
                behaviorClass: _behaviorsUndoRemove,
                eventNamespace: 'global-sidebar.spaces-menu'
            },

            tooltip: {
                behaviorClass: _behaviorsTooltips,
                selector: '.aui-nav-item',
                configs: {
                    gravity: 'w'
                }
            }
        }
    });
});
define('confluence-dashboard/core/shared/no-content-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', '../../utils/analytics', 'confluence/legacy', 'confluence/meta'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _utilsAnalytics, _confluenceLegacy, _confluenceMeta) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        tagName: 'li',
        className: 'empty-view no-content-view',
        template: _confluenceDashboardSoyTemplates.Default.blank,

        templateHelpers: function templateHelpers() {
            return {
                userCanCreateContent: _confluenceMeta.getBoolean('user-can-create-content')
            };
        },

        events: {
            'click a': 'onActionClick',
            'click .create-page': 'openCreatePage'
        },

        openCreatePage: function openCreatePage() {
            _confluenceLegacy.Blueprint.loadDialogAndOpenTemplate({});
        },

        onActionClick: function onActionClick() {
            _utilsAnalytics.publish('blank-experience.action.clicked');
        }
    });
});
define("confluence-dashboard/core/shared/no-content-view.js", function(){});

define('confluence-dashboard/modules/nav-spaces/nav-spaces-empty-view', ['exports', 'module', 'confluence-dashboard/soy-templates', '../../core/shared/no-content-view', 'ajs'], function (exports, module, _confluenceDashboardSoyTemplates, _coreSharedNoContentView, _ajs) {
    'use strict';

    module.exports = _coreSharedNoContentView.extend({
        tagName: 'li',
        template: _confluenceDashboardSoyTemplates.NavSpaces.mySpacesBlank,
        className: 'nav-space-blank',
        templateHelpers: {
            text: _ajs.I18n.getText('nav.controller.my.spaces.blank'),
            callToActionText: _ajs.I18n.getText('nav.controller.my.spaces.blank.callToAction'),
            callToActionLink: _ajs.contextPath() + '/spacedirectory/view.action'
        }
    });
});
define('confluence-dashboard/modules/nav-spaces/nav-spaces-composite-view', ['exports', 'module', '../../core/shared/base-composite-view', '../../core/shared/loading-view', 'confluence-dashboard/soy-templates', './nav-spaces-item-view', './nav-spaces-empty-view', '../../behaviors/analytics-tracking', '../../utils/analytics'], function (exports, module, _coreSharedBaseCompositeView, _coreSharedLoadingView, _confluenceDashboardSoyTemplates, _navSpacesItemView, _navSpacesEmptyView, _behaviorsAnalyticsTracking, _utilsAnalytics) {
    'use strict';

    module.exports = _coreSharedBaseCompositeView.extend({
        template: _confluenceDashboardSoyTemplates.Nav.container,
        childViewContainer: '.nav-items',
        childView: _navSpacesItemView,
        emptyView: _navSpacesEmptyView,
        loadingView: _coreSharedLoadingView,
        className: 'aui-navgroup-inner',

        behaviors: {
            analyticsTracking: {
                behaviorClass: _behaviorsAnalyticsTracking,
                prefix: 'global-sidebar.spaces-menu'
            }
        },

        events: {
            'click .all-spaces-link': 'onSpaceDirectoryClick'
        },

        onSpaceDirectoryClick: function onSpaceDirectoryClick() {
            _utilsAnalytics.publish('space-directory.clicked');
        }
    });
});
define('confluence-dashboard/modules/nav-spaces/nav-spaces-controller', ['exports', 'module', 'configuration', './nav-spaces-collection', 'confluence-dashboard/soy-templates', './nav-spaces-composite-view', '../../core/shared/base-controller', 'ajs'], function (exports, module, _configuration, _navSpacesCollection, _confluenceDashboardSoyTemplates, _navSpacesCompositeView, _coreSharedBaseController, _ajs) {
    'use strict';

    module.exports = _coreSharedBaseController.extend({
        createAllSpacesView: function createAllSpacesView() {
            var spacesMenu = new _navSpacesCompositeView({
                container: _configuration.sections.webItems.sidebar.MY_SPACES,
                collection: new _navSpacesCollection.AllSpacesCollection(),
                templateHelpers: {
                    title: _ajs.I18n.getText('nav.controller.all.spaces'),
                    anchorName: _ajs.I18n.getText('nav.controller.my.spaces.anchor'),
                    anchorLink: _ajs.contextPath() + '/spacedirectory/view.action',
                    anchorTitle: _ajs.I18n.getText('sidebar.header.spaces.menu.view.all')
                }
            });

            return spacesMenu;
        },

        createMySpacesView: function createMySpacesView() {
            var spacesMenu = new _navSpacesCompositeView({
                container: _configuration.sections.webItems.sidebar.MY_SPACES,
                collection: new _navSpacesCollection.FavouriteSpacesCollection(),
                templateHelpers: {
                    title: _ajs.I18n.getText('nav.controller.my.spaces'),
                    anchorName: _ajs.I18n.getText('nav.controller.my.spaces.anchor'),
                    anchorLink: _ajs.contextPath() + '/spacedirectory/view.action',
                    anchorTitle: _ajs.I18n.getText('sidebar.header.spaces.menu.view.all')
                }
            });

            spacesMenu.collection.fetch();

            return spacesMenu;
        }
    });
});
define('confluence-dashboard/modules/sidebar/sidebar-view', ['exports', 'module', 'marionette', '../../behaviors/aui-sidebar-resizable', '../../modules/nav/nav-controller', '../../modules/nav-spaces/nav-spaces-controller', '../../utils/conditions'], function (exports, module, _marionette, _behaviorsAuiSidebarResizable, _modulesNavNavController, _modulesNavSpacesNavSpacesController, _utilsConditions) {
    'use strict';

    module.exports = _marionette.LayoutView.extend({
        template: false,

        el: '.aui-sidebar',

        regions: {
            'sidebar-discover': '#sidebar-discover',
            'sidebar-my-work': '#sidebar-my-work',
            'sidebar-spaces': '#sidebar-spaces'
        },

        behaviors: {
            AuiSidebarResizable: {
                behaviorClass: _behaviorsAuiSidebarResizable
            }
        },

        initialize: function initialize() {
            this.setNavContent();
            this.trigger('sidebar-attached');
        },

        setNavContent: function setNavContent() {
            var _this = this;

            var navController = new _modulesNavNavController();
            var navSpacesController = new _modulesNavSpacesNavSpacesController();

            this.getRegion('sidebar-discover').show(navController.createDiscoverView());

            if (!_utilsConditions.isAnonymousUser()) {
                this.getRegion('sidebar-my-work').show(navController.createMyWorkView());
                this.getRegion('sidebar-spaces').show(navSpacesController.createMySpacesView());
            } else {
                (function () {
                    // only show list of spaces for anonymous users if there are some spaces
                    var allSpacesView = navSpacesController.createAllSpacesView();

                    allSpacesView.collection.fetch().then(function () {
                        if (allSpacesView.collection.length > 0) {
                            _this.getRegion('sidebar-spaces').show(allSpacesView);
                        }
                    });
                })();
            }
        }
    });
});
define('confluence-dashboard/modules/welcome-message/welcome-message-view', ['exports', 'module', 'marionette', 'jquery', '../../utils/analytics'], function (exports, module, _marionette, _jquery, _utilsAnalytics) {
    'use strict';

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

    var _$ = _interopRequireDefault(_jquery);

    var HIGHLIGHTED_CLASS = 'welcome-message-highlight';
    var EDIT_BUTTON = '.aui-button.welcome-message-edit-button';

    module.exports = _marionette.ItemView.extend({
        el: '.welcome-message',

        template: false,

        events: {
            'mouseenter a.welcome-message-edit-button': 'highlightOn',
            'mouseleave a.welcome-message-edit-button': 'highlightOff',
            'click a.welcome-message-edit-button': 'onEditButtonClick'
        },

        highlightOn: function highlightOn() {
            this.$el.addClass(HIGHLIGHTED_CLASS);
        },

        highlightOff: function highlightOff() {
            this.$el.removeClass(HIGHLIGHTED_CLASS);
        },

        onEditButtonClick: function onEditButtonClick() {
            _utilsAnalytics.publish('edit-welcome-message.clicked');
        }
    });
});
define('confluence-dashboard/core/main/main-router', ['exports', 'module', 'marionette', '../../utils/event-manager', 'backbone', 'confluence/storage-manager', 'underscore', 'jquery', '../../utils/analytics'], function (exports, module, _marionette, _utilsEventManager, _backbone, _confluenceStorageManager, _underscore, _jquery, _utilsAnalytics) {
    'use strict';

    var dashboardHistory = (0, _confluenceStorageManager)('dashboard', 'route');

    var DEFAULT_PAGE = 'all-updates';

    // debouncing so we don't track redirects
    var trackPageView = _underscore.debounce(function (lastPath) {

        _utilsAnalytics.publish('view', {
            previousSection: lastPath
        });
    }, 100);

    module.exports = _marionette.AppRouter.extend({
        initialize: function initialize() {
            this.listenTo(_backbone.history, 'route', this.onRoute);
        },

        onRoute: function onRoute() {
            trackPageView(dashboardHistory.getItem('last'));

            var path = _backbone.history.fragment;

            dashboardHistory.setItemQuietly('last', path);
            _utilsEventManager.EventManager.trigger('onRoute', path);
        },

        appRoutes: {
            // '' -> empty key means root - the same defined on our main-app.js
            '': 'index'
        },

        controller: {
            index: function index() {
                var initialPage = DEFAULT_PAGE;

                if (dashboardHistory.doesContain('last')) {
                    initialPage = dashboardHistory.getItem('last');
                }

                _utilsAnalytics.publish('root-loaded', {
                    redirectedTo: initialPage
                });

                _backbone.history.navigate(initialPage, { trigger: true, replace: true });
            }
        },

        handleClicks: function handleClicks(evt) {
            var href = (0, _jquery)(evt.currentTarget).attr('href'); //.replace('/content', '');
            var protocol = window.location.protocol + '//';

            if (href.slice(protocol.length) !== protocol) {
                var metric = 'global-sidebar.clicked.using-meta';

                if (!evt.ctrlKey && !evt.metaKey) {
                    evt.preventDefault();
                    //need to slice the href to cause the fragment won't have the #
                    if (_backbone.history.fragment === href.slice(1)) {
                        _backbone.history.loadUrl();
                    } else {
                        this.navigate(href, true);
                    }

                    metric = 'global-sidebar.clicked';
                }

                _utilsAnalytics.publish(metric);
            }
        }
    });
});
define('confluence-dashboard/core/content/content-factory', ['exports', 'module'], function (exports, module) {
	'use strict';

	module.exports = function (options) {
		if (!options) {
			throw 'Missing options!';
		}

		var name = options.name;
		var scope = options.scope;
		var routes = options.routes;
		var controllerModule = options.controllerModule;

		if (!controllerModule) {
			throw 'Missing controller module!';
		}

		if (!routes) {
			throw 'Missing route definitions!';
		}

		// TODO: in production it should be async so we will load the controller when loading the web item
		var Controller = require(controllerModule);

		if (!Controller) {
			throw 'Controller not found!';
		}

		return new Controller({
			name: name,
			scope: scope || '/',
			routes: routes
		});
	};
});
define('confluence-dashboard/core/main/main-controller', ['exports', 'module', './main-router', '../../utils/event-manager', 'backbone', '../shared/base-controller', '../content/content-factory', 'underscore'], function (exports, module, _mainRouter, _utilsEventManager, _backbone, _sharedBaseController, _contentContentFactory, _underscore) {
	'use strict';

	function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

	module.exports = _sharedBaseController.extend({
		initialize: function initialize() {
			this.router = new _mainRouter();

			// handle the navigation between spa linsk
			this.on('navigate', this.onNavigate, this);

			// allow web-items to create their rotes
			_utilsEventManager.Commands.setHandler('app:setupWebItem', this.setupWebItem, this);
		},

		onNavigate: function onNavigate(href) {
			// all navigate events go through here
			this.router.navigate(href, true);
		},

		setupWebItem: function setupWebItem(webItem) {
			if (!webItem) {
				throw 'No web item data';
			}

			if (webItem.spa) {

				if (!webItem.url) {
					throw 'Web Item missing an url';
				}

				if (!webItem.controllerMethod) {
					throw 'Web Item missing its controller method';
				}

				if (!webItem.controllerModule) {
					throw 'Web Item missing controller';
				}

				// if it's an spa, create a content router
				var routes = _defineProperty({}, webItem.url, webItem.controllerMethod);

				// call the factoty. If everything is ok it will return a controller
				this.webItemController = (0, _contentContentFactory)({
					name: webItem.key,
					scope: webItem.scope,
					routes: routes,
					controllerModule: webItem.controllerModule
				});

				// make sure that this route triggers if it's created
				// after the router is started
				if (webItem.url === _backbone.history.fragment) {
					_backbone.history.loadUrl();
				}
			}
		}
	});
});
define('confluence-dashboard/core/content/content-region-animation', ['exports', 'module', 'marionette', 'jquery'], function (exports, module, _marionette, _jquery) {
    'use strict';

    /*eslint-disable */
    _jquery.extend(_jquery.easing, {
        def: 'easeOutQuad',
        easeOutQuad: function easeOutQuad(x, t, b, c, d) {
            return -c * (t /= d) * (t - 2) + b;
        }
    });
    /*eslint-enable */

    var animationOptions = {
        easing: 'easeOutQuad',
        duration: 200,
        queue: false // queue false to avoid blocking other animations
    };

    var scrollToTheTop = function scrollToTheTop() {
        // Scroll back to the top if the user has scrolled down more than the height of the header.
        // It returns a promise so we can queue with the other animations.
        var defer = _jquery.Deferred();
        var hh = (0, _jquery)('#header').height();

        var doc = document.documentElement;
        var scrollPosition = (window.pageYOffset || doc.scrollTop) - (doc.clientTop || 0);

        if (scrollPosition > hh) {
            var animationProperties = _jquery.extend({}, animationOptions, {
                complete: function complete() {
                    defer.resolve();
                }
            });

            (0, _jquery)('html, body').animate({
                scrollTop: hh
            }, animationProperties);
        } else {
            defer.resolve();
        }

        return defer.promise();
    };

    module.exports = _marionette.Region.extend({
        // animation that will run for content coming in
        // opacity is not here because it would cause a flash in slow
        // connections when the loading view appear before the content.
        // For changes in the opacity settings, go to
        // shared/base-composite-view.js#attachBuffer method
        'in': function _in(el) {
            el.css({
                position: 'relative',
                top: '15px',
                opacity: 0
            }).animate({
                top: 0,
                opacity: 1
            }, animationOptions);
        },

        // animation that will run when removing content
        out: function out(el) {
            var defer = _jquery.Deferred();

            el.animate({
                opacity: 0
            }, _jquery.extend({}, animationOptions, {
                complete: function complete() {
                    defer.resolve();
                }
            }));

            return defer.promise();
        },

        // overriding the Region show method so we can run animations in between each view
        show: function show() {
            var _this = this;

            var self = this;
            var args = arguments;

            if (this.currentView) {
                scrollToTheTop().then(function () {
                    _this.out(_this.$el.find(_this.options.selectorToAnimate)).then(function () {
                        _marionette.Region.prototype.show.apply(self, args);
                        _this['in'](_this.$el.find(_this.options.selectorToAnimate));
                    });
                });
            } else {
                _marionette.Region.prototype.show.apply(this, arguments);
                this['in'](this.$el.find(this.options.selectorToAnimate));
            }
        }
    });
});
define('confluence-dashboard/core/main/main-app', ['exports', 'module', 'marionette', '../../utils/event-manager', '../../modules/sidebar/sidebar-view', '../../modules/welcome-message/welcome-message-view', './main-controller', '../content/content-region-animation', 'underscore', 'jquery', 'ajs', '../../utils/conditions'], function (exports, module, _marionette, _utilsEventManager, _modulesSidebarSidebarView, _modulesWelcomeMessageWelcomeMessageView, _mainController, _contentContentRegionAnimation, _underscore, _jquery, _ajs, _utilsConditions) {
    'use strict';

    var $win = (0, _jquery)(window);

    module.exports = _marionette.Application.extend({
        el: '.PageContent',

        isStarted: false,

        regions: {
            'page': '.confluence-dashboard',
            'content': {
                el: '.content-body',
                selectorToAnimate: '.list-container',
                regionClass: _contentContentRegionAnimation
            },
            'dialogs': '.dialogs',
            'right-sidebar': '.content-sidebar',
            'welcome-message': '.welcome-message-wrapper'
        },

        initialize: function initialize() {
            this.controller = new _mainController();

            this.router = this.controller.router;

            // Returns this instance so you can get it from anywhere in the app
            _utilsEventManager.ReqRes.setHandler('app', this.getApp, this);

            // Global command called from the controllers to switch the layout
            _utilsEventManager.Commands.setHandler('main-app:swapContent', this.swapContent, this);

            // Global command called from the controllers to switch the layout
            _utilsEventManager.Commands.setHandler('main-app:showDialog', this.showDialog, this);

            // binding an event to window scroll, so instead of listening to scroll you can listen to window:scroll.
            $win.on('scroll.window.main-layout', this.triggerWindowScroll);

            // the same goes for the resize event. Just listen to window:resize on your view
            $win.on('resize.window.main-layout', _underscore.debounce(this.triggerWindowResize.bind(this), 16));
        },

        getApp: function getApp() {
            return this;
        },

        onStart: function onStart() {
            this.isStarted = true;
            this.attachServerRenderedViews();
            this.attachWelcomeMessage();
        },

        onBeforeDestroy: function onBeforeDestroy() {
            $win.off('scroll.window.main-layout');
            $win.off('resize.window.main-layout');
        },

        triggerWindowScroll: function triggerWindowScroll() {
            // TODO: if we need to calculate the scroll position in more than two places it make sense to send it here as parameter
            _utilsEventManager.EventManager.trigger('window:scroll', window);
        },

        triggerWindowResize: function triggerWindowResize() {
            // TODO: if we need to calculate the window size in more than two places it make sense to send it here as parameter
            _utilsEventManager.EventManager.trigger('window:resize', window);
        },

        // Global command called from the controllers to switch the layout
        swapContent: function swapContent(contentView) {
            this.getRegion('content').show(contentView);
        },

        // Global command called from anywhere to add a dialog
        showDialog: function showDialog(dialogView) {
            this.getRegion('dialogs').show(dialogView);
        },

        attachServerRenderedViews: function attachServerRenderedViews() {
            this.getRegion('page').attachView(new _modulesSidebarSidebarView());
        },

        attachWelcomeMessage: function attachWelcomeMessage() {
            this.getRegion('welcome-message').attachView(new _modulesWelcomeMessageWelcomeMessageView());
        }
    });
});
define('confluence-dashboard/utils/single-flag', ['exports', 'module', 'confluence/flag', 'underscore', 'jquery'], function (exports, module, _confluenceFlag, _underscore, _jquery) {
    'use strict';

    /**
     * Single flag makes sure that you won't show more than one flag each time.
     * The last created takes precedence and removes the last one.
     *
     * @type {{currentFlag: null, closePreviousFlags: Function, create: Function}}
     */

    var SingleFlag = {
        currentFlag: null,

        closePreviousFlags: function closePreviousFlags() {
            if (SingleFlag.currentFlag) {
                SingleFlag.currentFlag.flag.close();
                SingleFlag.currentFlag = null;
            }
        },

        shouldShow: function shouldShow(options) {
            if (!SingleFlag.currentFlag || !SingleFlag.currentFlag.flag) {
                return true;
            }

            var isOpen = (0, _jquery)(SingleFlag.currentFlag.flag).attr('aria-hidden') === 'false';
            var hasTheSameProps = _underscore.isEqual(SingleFlag.currentFlag.options, options);

            // compare the set of options and if they are the same, don't show
            if (isOpen && hasTheSameProps) {
                return false;
            }

            return true;
        },

        create: function create(options) {
            var defaults = {
                type: 'error',
                close: 'auto',
                persistent: false,
                stack: 'dashboard'
            };

            var config = _underscore.extend(defaults, options);

            if (!this.shouldShow(config)) {
                return;
            }

            SingleFlag.closePreviousFlags();

            SingleFlag.currentFlag = {
                flag: (0, _confluenceFlag)(config),
                options: config
            };

            if (config.callback && _underscore.isFunction(config.callback)) {
                config.callback();
            }
        }
    };

    module.exports = SingleFlag;
});
define('confluence-dashboard/utils/navigation', ['exports', 'module', 'window'], function (exports, module, _window) {
    'use strict';

    module.exports = {
        /**
         * Provides an easy to test facade for window.location.href = 'foo-bar';
         */
        redirect: function redirect(url) {
            _window.location.href = url;
        }
    };
});
define('confluence-dashboard/utils/error-handlers', ['exports', 'module', 'backbone', 'jquery', 'ajs', 'underscore', './analytics', './single-flag', './navigation', 'configuration'], function (exports, module, _backbone, _jquery, _ajs, _underscore, _analytics, _singleFlag, _navigation, _configuration) {
    'use strict';

    var $win = (0, _jquery)(window);
    var contextPath = _ajs.contextPath();
    var EVENT_NAMESPACE = 'dashboard-error-handler';

    var FLAG_DEFINITIONS = {
        NOT_FOUND: {
            type: 'error',
            title: _ajs.I18n.getText('error.notfound.title'),
            close: 'auto'
        },

        SERVER_ERROR: {
            type: 'error',
            title: _ajs.I18n.getText('error.server-error.title'),
            body: _ajs.I18n.getText('error.server-error.body'),
            close: 'auto'
        },

        SESSION_EXPIRED: {
            type: 'error',
            title: _ajs.I18n.getText('error.session.expired.title'),
            body: _ajs.I18n.getText('error.session.expired.body'),
            close: 'never'
        },

        OFFLINE: {
            type: 'error',
            title: _ajs.I18n.getText('error.connection.lost.title'),
            body: _ajs.I18n.getText('error.connection.lost.body'),
            close: 'never'
        },

        ONLINE: {
            type: 'success',
            title: _ajs.I18n.getText('error.connection.recovered.title'),
            body: '<button class="aui-button aui-button-link btn-reload-content">' + _ajs.I18n.getText('error.connection.recovered.action') + '</button>',
            close: 'auto',
            callback: function callback() {
                (0, _jquery)(_singleFlag.currentFlag).on('click', '.btn-reload-content', function () {
                    _singleFlag.closePreviousFlags();
                    _backbone.history.loadUrl();
                });
            }
        }
    };

    var handle404 = function handle404() {
        _singleFlag.create(FLAG_DEFINITIONS.NOT_FOUND);
        _analytics.publish('error-handler.not-found');
    };

    var handleServerErrors = function handleServerErrors() {
        _singleFlag.create(FLAG_DEFINITIONS.SERVER_ERROR);
        _analytics.publish('error-handler.server-error');
    };

    var handleSessionIssues = function handleSessionIssues(xhr) {
        // request not completed and alike
        if (xhr.status === 0) {
            handleServerErrors();
        } else if (/(401|403)/.test(xhr.status)) {
            _singleFlag.create(FLAG_DEFINITIONS.SESSION_EXPIRED);
            _analytics.publish('error-handler.session.expired');

            var destination = window.location.pathname.replace(contextPath, '');
            _navigation.redirect(_configuration.URLS.LOGIN_PAGE + '?os_destination=' + destination);
        }
    };

    var handleConnectionIssues = function handleConnectionIssues() {
        _singleFlag.create(FLAG_DEFINITIONS.OFFLINE);
        _analytics.publish('error-handler.connection.lost');

        // after showing a message saying you're offline, we listen to the online event;
        // when it happens, we remove the message and reload the current view
        $win.one('online.' + EVENT_NAMESPACE, function () {
            _singleFlag.create(FLAG_DEFINITIONS.ONLINE);
            _analytics.publish('error-handler.connection.recovered');
        });
    };

    var BackboneAjaxBackup = _backbone.ajax;

    module.exports = {
        interceptBackboneErrors: function interceptBackboneErrors() {
            // overriding backbone.ajax to include statusCode error handlers.
            // this way we only affect ajax requests started by backbone.
            _backbone.ajax = function () {
                var options = arguments.length <= 0 || arguments[0] === undefined ? {} : arguments[0];

                options.statusCode = _jquery.extend({
                    401: handleSessionIssues,
                    403: handleSessionIssues,
                    404: handle404,
                    400: handleServerErrors,
                    500: handleServerErrors
                }, options.statusCode || {});

                return _backbone.$.ajax.call(_backbone.$, options);
            };
        },

        interceptConnectionErrors: function interceptConnectionErrors() {
            $win.on('offline.' + EVENT_NAMESPACE, handleConnectionIssues);
        },

        stopIntercepting: function stopIntercepting() {
            $win.off('.' + EVENT_NAMESPACE);
            _backbone.ajax = BackboneAjaxBackup;
        }
    };
});
define('confluence-dashboard/app', ['exports', 'module', 'backbone', 'ajs', 'jquery', './core/main/main-app', './utils/error-handlers', './utils/analytics', 'configuration'], function (exports, module, _backbone, _ajs, _jquery, _coreMainMainApp, _utilsErrorHandlers, _utilsAnalytics, _configuration) {
    'use strict';

    // Create our Application
    var app = new _coreMainMainApp();

    app.on('before:start', function () {
        // collection / Api errors 401, 403, 404 and 500
        _utilsErrorHandlers.interceptBackboneErrors();
        // offline / online
        _utilsErrorHandlers.interceptConnectionErrors();

        // start the router
        _backbone.history.start(_configuration.backboneHistoryConfig);
    });

    // Start history when our application is ready
    app.on('start', function () {

        // send all the links to the router
        (0, _jquery)(document).on('click', '.confluence-dashboard .aui-sidebar-body .spa a', app.router.handleClicks.bind(app.router));

        _utilsAnalytics.publish('rendered');
    });

    module.exports = app;
});
define('confluence-dashboard/index', ['exports', './app', './utils/conditions', 'ajs'], function (exports, _app, _utilsConditions, _ajs) {
    'use strict';

    _ajs.toInit(function () {
        if (_utilsConditions.canShowDashboard()) {
            _app.start();
        }
    });
});
define('confluence-dashboard/core/shared/no-matches-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        tagName: 'li',
        className: 'empty-view no-matches-view',
        template: _confluenceDashboardSoyTemplates.Default.noMatches
    });
});
define("confluence-dashboard/core/shared/no-matches-view.js", function(){});

define('confluence-dashboard/core/shared/cql-base-collection', ['exports', 'module', './base-collection', 'jquery', 'ajs', 'underscore'], function (exports, module, _baseCollection, _jquery, _ajs, _underscore) {
    'use strict';

    var context = _ajs.contextPath();

    /**
     * Cql-Base-Collection
     * ===================
     *
     * It's a basic Backbone Collection that reads data from an Atlassian REST API using a CQL query.
     *
     * This implementation will provide the methods for infinite loading, filtering
     * and parsing the results
     */

    module.exports = _baseCollection.extend({
        // generic url for collections. It will concatenate all the attributes inside apiParams. Override it on your collection if you need something different.
        url: function url() {
            return this.buildUrlFromParams();
        },

        // build your url from params. Your collection should have an attribute
        // like the following one:
        //      ...
        //      apiParams: {
        //          url: endpoints.RECENTLY_WORKED,
        //          params: 'limit=20',
        //          expansions: [
        //              'container',
        //              'metadata.currentuser.lastmodified'
        //          ],
        //          cql: 'type in (page,blogpost) and contributor=currentUser()',
        //      },
        //      ...
        // it will make easier and more readable building compound urls that rely on
        // a certain order of the parameters
        buildUrlFromParams: function buildUrlFromParams(options) {
            if (!this.apiParams) {
                throw 'apiParams attribute missing';
            }

            var _getApiParams = this.getApiParams(options);

            var url = _getApiParams.url;
            var params = _getApiParams.params;
            var expansions = _getApiParams.expansions;
            var cql = _getApiParams.cql;
            var cqlOrder = _getApiParams.cqlOrder;

            var queryAttributes = [];

            if (!url) {
                throw 'CqlBaseCollection:buildUrlFromParams: missing URL!';
            }

            if (params) {
                queryAttributes.push(_jquery.param(params));
            }

            if (expansions) {
                queryAttributes.push('expand=' + expansions.join(','));
            }

            if (cql) {
                if (options && options.cqlQuery) {
                    cql += ' ' + options.cqlQuery;
                }
                if (cqlOrder) {
                    cql += ' order by ' + cqlOrder;
                }

                queryAttributes.push('cql=' + cql);
            }

            if (queryAttributes.length) {
                url = url + '?' + queryAttributes.join('&');
            }

            return url;
        },

        getApiParams: function getApiParams(options) {
            return _underscore.isFunction(this.apiParams) ? this.apiParams(options) : this.apiParams;
        },

        // default parse method for Atlassian REST response format. It also detects if there is a next page and persist this information
        parse: function parse(data, options) {
            this.parseNext(data, options);
            return data.results;
        },

        // Used inside the parse method to detect if there is a next page. This follows the Atlassian REST response format and should be overwritten on your collection if you expect a different format of data.
        parseNext: function parseNext(data, options) {
            if (data._links && data._links.next) {
                this.hasNext = true;
                this.nextUrl = context + data._links.next;
            } else {
                this.hasNext = false;
                this.nextUrl = null;
            }
        },

        // Used by the infinite loading if the parseNext detected the next page. It will return a promise. If there is no next page it will reject the promise immediatelly.
        loadMore: function loadMore() {
            if (this.hasNext) {
                return this.safeFetch({
                    url: this.nextUrl,
                    remove: false,
                    loadingMore: true
                });
            } else {
                var defer = _jquery.Deferred();
                defer.reject();
                return defer.promise();
            }
        },

        // Used by the filter. It assumes you have cql on your endpoint and just adds one more clause at the end. If your endpoint doesn't have cql on the url, override this method on your collection with the right path
        search: function search(params) {
            // params.query should be CQL safe (see utils/strings.normalizeForCQL)
            var url = this.buildUrlFromParams({ cqlQuery: 'and title ~ \'' + params.query + '*\'', isFiltered: true });

            return this.safeFetch({
                url: url,
                isFiltered: true
            });
        }
    });
});
define("confluence-dashboard/core/shared/cql-base-collection.js", function(){});

define('confluence-dashboard/core/shared/base-collection-with-function', ['exports', 'module', './cql-base-collection', 'configuration', 'jquery', 'underscore'], function (exports, module, _cqlBaseCollection, _configuration, _jquery, _underscore) {
    'use strict';

    /**
     * Base-Collection-With-Function
     * =============================
     *
     * A Backbone Collection that implements common functionality to handle infinite scrolling and filtering
     * appropriately when content ids are read from a cql function. The following special handling is required when
     * start and limit are specified within the function:
     *
     * - Infinite scrolling needs to pass the start and limit to the function manually without using the API _links.next url
     * - Filtering and infinite scrolling don't work well together, so infinite scrolling is disabled for filtering
     *
     * To use, pass the start and limit parameters to the cql function:
     *
     * apiParams(options) {
     *   cql: `type in (page,blogpost) and id in function(${this.getPageLimit(options)}, ${this.getPageOffset(options)})`
     * }
     */

    module.exports = _cqlBaseCollection.extend({
        // the currently displayed page for infinite scrolling
        page: 0,

        sync: function sync(method, model, options) {
            // if not an infinite scroll load (eg. collection.fetch is called manually), start from page 1
            if (!options.loadingMore) {
                this.page = 0;
            }
            return _cqlBaseCollection.prototype.sync.apply(this, arguments);
        },

        parseNext: function parseNext(data, options) {
            // the next infinite scroll load will fetch the next page
            this.page += 1;
            this.hasNext = !(options && options.isFiltered) && data.results.length >= _configuration.apiLimit && this.getPageLimit(options) > 0;
            this.nextUrl = this.hasNext ? this.buildUrlFromParams(options) : null;
        },

        getPageOffset: function getPageOffset(options) {
            // infinite scrolling doesn't play nice with offset+limit within the function, and filtering
            return options && options.isFiltered ? 0 : this.page * _configuration.apiLimit;
        },

        getPageLimit: function getPageLimit(options) {
            // infinite scrolling doesn't play nice with offset+limit within the function, and filtering
            return options && options.isFiltered ? _configuration.visibleItemLimit : _underscore.min([_configuration.apiLimit, _configuration.visibleItemLimit - this.getPageOffset(options)]);
        }
    });
});
define("confluence-dashboard/core/shared/base-collection-with-function.js", function(){});

define('confluence-dashboard/core/shared/base-dialog', ['exports', 'module', 'marionette', 'ajs'], function (exports, module, _marionette, _ajs) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        tagName: 'section',

        className: 'aui-layer aui-dialog2 aui-dialog2-medium confluence-dialog-no-chrome confluence-dialog-centered',

        attributes: {
            'aria-hidden': true
        },

        openDialog: function openDialog() {
            var _this = this;

            this.initDialog();
            this.dialog.on('show', function () {
                return _this.delegateEvents();
            });
            this.dialog.on('hide', function () {
                return _this.destroy();
            });
            this.dialog.show();
        },

        closeDialog: function closeDialog() {
            if (this.dialog) {
                this.dialog.hide();
            }
        },

        onRender: function onRender() {
            this.initDialog();
        },

        initDialog: function initDialog() {
            if (!this.dialog) {
                this.dialog = _ajs.dialog2(this.el);
            }
        }
    });
});
define("confluence-dashboard/core/shared/base-dialog.js", function(){});

define('confluence-dashboard/core/shared/scope-router', ['exports', 'module', 'marionette', 'backbone', 'underscore', '../../utils/logger'], function (exports, module, _marionette, _backbone, _underscore, _utilsLogger) {
	'use strict';

	/**
  * Scoped Router
  * =============
  *
  * Let you define routes per scope avoiding a monolitic huge router;
  *
  * Usage:
  *
  * import ScopeRouter from '../shared/scope-router';
  *
  * let Router = ScopeRouter.extend({
  * 		scope: 'name-of-the-feature',
  * 		scopedRoutes: {
  * 			'action1': 'doAction1'
  * 		},
  * 		controller: new MyFeatureController()
  * })
  *
  * It will create the url 'name-of-the-feature/action1' and bound to the
  * method 'doAction1' from the controller 'MyFeatureController'.
  *
  * If you need to reuse any of the components outside of this app, change your controller
  * to extend 'Marionette.AppRouter' and change the key from 'scopedRoutes' to 'appRoutes'
  * or copy this file.
  */

	// Overriding history.route to avoid duplicated routes.
	_backbone.History.prototype.route = function (route, callback) {
		// We can't compare regexps so let's convert it to string
		var name = String(route);
		// Now we can use it to search in the existent routes
		var handler = _underscore.findWhere(this.handlers, { name: name });

		// if there is a route for the current handler, just update the handler
		if (handler) {
			handler.callback = callback;
		} else {
			this.handlers.unshift({ name: name, route: route, callback: callback });
		}
	};

	module.exports = _marionette.AppRouter.extend({
		initialize: function initialize(options) {
			var scope = this.options.scope || this.scope;
			var scopedRoutes = this.options.scopedRoutes || this.scopedRoutes;

			if (!scope) {
				throw 'Scope router requires a scope!';
			}

			if (!scopedRoutes) {
				throw 'Scope router requires a scopedRoutes object!';
			}

			if (!this.controller && !this.options.controller) {
				throw 'Scope router requires a controller!';
			}

			if (scope === '/') {
				scope = '';
			} else {
				scope = scope + '/';
			}

			var appRoutes = {};

			_underscore.each(scopedRoutes, function (method, route) {
				appRoutes['' + scope + route] = method;
			});

			this.scope = scope;
			this.appRoutes = appRoutes;
		}

	});
});
// scopedRoutes: {},

// controller: featureController;
define("confluence-dashboard/core/shared/scope-router.js", function(){});

define('confluence-dashboard/core/content/content-as-grouped-list-view', ['exports', 'module', 'jquery', 'underscore', '../shared/base-composite-view', 'confluence-dashboard/soy-templates', '../../utils/event-manager', '../../behaviors/stickable/stickable', '../../behaviors/infinite-loading', '../../behaviors/filterable', '../../behaviors/progress-indicator', '../../behaviors/analytics-tracking', '../shared/no-content-view', '../shared/loading-view', '../shared/no-matches-view', '../../modules/group/group-composite-view', '../../modules/group/group-collection', '../../modules/list-item/list-item-view'], function (exports, module, _jquery, _underscore, _sharedBaseCompositeView, _confluenceDashboardSoyTemplates, _utilsEventManager, _behaviorsStickableStickable, _behaviorsInfiniteLoading, _behaviorsFilterable, _behaviorsProgressIndicator, _behaviorsAnalyticsTracking, _sharedNoContentView, _sharedLoadingView, _sharedNoMatchesView, _modulesGroupGroupCompositeView, _modulesGroupGroupCollection, _modulesListItemListItemView) {
    'use strict';

    module.exports = _sharedBaseCompositeView.extend({
        template: _confluenceDashboardSoyTemplates.Content.groupedListWithFilter,
        childViewContainer: '.list-container',

        childView: _modulesGroupGroupCompositeView,
        childViewOptions: {
            childView: _modulesListItemListItemView
        },

        emptyView: _sharedNoContentView,
        loadingView: _sharedLoadingView,
        noMatchesView: _sharedNoMatchesView,
        noMatchesViewOptions: {
            templateHelpers: function templateHelpers() {
                return {
                    filter: this._parent.getFilterString()
                };
            }
        },

        ui: {
            filter: '[name=filter]'
        },

        behaviors: {
            // adds ios like appearance to the headers
            stickMainHeader: {
                behaviorClass: _behaviorsStickableStickable,
                element: '.content-header',
                autoRun: true
            },

            // infinite loading based on the collection methods / response
            infiniteLoading: {
                behaviorClass: _behaviorsInfiniteLoading,
                target: '.list-container'
            },

            // progress indicator (spinner) based on the collection methods / response
            progressIndicator: {
                behaviorClass: _behaviorsProgressIndicator,
                dataSource: 'collection',
                // once loadingView is embedded into groupedListWithFilter / streamList, the page gets two
                // 'spinner-container' elements so just take the first one
                container: '.spinner-container:first',
                size: 'medium'
            },

            // adding filter capabilities
            filterable: {
                behaviorClass: _behaviorsFilterable
            },

            analyticsTracking: {
                behaviorClass: _behaviorsAnalyticsTracking
            }
        },

        events: {
            'submit .content-filter': 'onFilterSubmit'
        },

        // if we are using a group collection inside, it won't trigger the default event collection:render so we should trigger it ourselves
        initialize: function initialize() {
            var collection = this.collection;

            this.collection = new _modulesGroupGroupCollection([], {
                collectionToGroup: collection
            });
        },

        isFiltered: function isFiltered() {
            return this.getFilterString().length > 0;
        },

        getFilterString: function getFilterString() {
            return this.ui.filter.val();
        },

        onFilterSubmit: function onFilterSubmit(e) {
            // disable filter submit, including on enter keypress
            e.preventDefault();
        }
    });
});
define("confluence-dashboard/core/content/content-as-grouped-list-view.js", function(){});

define('confluence-dashboard/core/content/content-as-stream-view', ['exports', 'module', '../shared/base-composite-view', 'confluence-dashboard/soy-templates', '../../behaviors/stickable/stickable', '../../behaviors/infinite-loading', '../../behaviors/progress-indicator', '../../behaviors/analytics-tracking', '../shared/no-content-view', '../shared/loading-view', '../shared/no-matches-view', '../../modules/stream-item/stream-item-view'], function (exports, module, _sharedBaseCompositeView, _confluenceDashboardSoyTemplates, _behaviorsStickableStickable, _behaviorsInfiniteLoading, _behaviorsProgressIndicator, _behaviorsAnalyticsTracking, _sharedNoContentView, _sharedLoadingView, _sharedNoMatchesView, _modulesStreamItemStreamItemView) {
    'use strict';

    module.exports = _sharedBaseCompositeView.extend({
        template: _confluenceDashboardSoyTemplates.Content.streamList,
        childViewContainer: '.list-container',

        emptyView: _sharedNoContentView,
        loadingView: _sharedLoadingView,
        noMatchesView: _sharedNoMatchesView,
        childView: _modulesStreamItemStreamItemView,

        behaviors: {
            // adds ios like appearance to the headers
            stickMainHeader: {
                behaviorClass: _behaviorsStickableStickable,
                element: '.content-header',
                autoRun: true
            },

            // progress indicator (spinner) based on the collection methods / response
            progressIndicator: {
                behaviorClass: _behaviorsProgressIndicator,
                dataSource: 'collection',
                // once loadingView is embedded into groupedListWithFilter / streamList, the page gets two
                // 'spinner-container' elements so just take the first one
                container: '.spinner-container:first',
                size: 'medium'
            },

            infiniteLoading: {
                behaviorClass: _behaviorsInfiniteLoading,
                target: '.list-container'
            },

            analyticsTracking: {
                behaviorClass: _behaviorsAnalyticsTracking
            }
        }
    });
});
define("confluence-dashboard/core/content/content-as-stream-view.js", function(){});

define('confluence-dashboard/core/content/content-controller', ['exports', 'module', '../../utils/event-manager', '../shared/scope-router', '../shared/base-controller'], function (exports, module, _utilsEventManager, _sharedScopeRouter, _sharedBaseController) {
	'use strict';

	module.exports = _sharedBaseController.extend({
		beforeAction: function beforeAction() {},

		afterAction: function afterAction() {
			_utilsEventManager.Commands.execute('main-app:swapContent', this.view);
		},

		initialize: function initialize(options) {
			if (!options || !options.routes) {
				throw 'ContentController is missing its route';
			}

			this.wrapActions();

			this.router = new _sharedScopeRouter({
				scopedRoutes: options.routes,
				scope: options.scope || '/',
				controller: this
			});
		}
	});
});
define("confluence-dashboard/core/content/content-controller.js", function(){});

define('confluence-dashboard/behaviors/stickable/stickable', ['exports', 'module', 'marionette', '../../utils/event-manager', 'underscore', 'ajs'], function (exports, module, _marionette, _utilsEventManager, _underscore, _ajs) {
    'use strict';

    /**
     * Stickable
     *
     *
     * Add this behaviour to the view you want to stick in the screen.
     *
     * Params:
     *  - element: if you want a child node to stick instead of the view.$el
     *  - autoRun: stick it immediatelly.
     *  - offset: distance from the top when it should stick
     *  - gap: Safe margin added to the offset to calculate the position it should stick.
     *         This value won't be added to the top position.
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *       stickable: {
     *           behaviorClass: Stickable,
     *           element: '.content-header',
     *           offset: 60
     *       }
     *   }
     * })
     *
     * * NOTE: It requires the sticky-headers.css file with at least this rules:
     *
     *
     * TODO: write tests
     */

    module.exports = _marionette.Behavior.extend({
        defaults: {
            offset: 0,
            gap: 0,
            autoRun: false
        },

        onRender: function onRender() {
            var opts = this.options;

            if (!opts.element) {
                throw 'Behavior:Stickable: Missing `element` attribute';
            }

            this.$target = this.view.$el.find(opts.element);
            this.target = this.$target[0];

            this.listenTo(_utilsEventManager.EventManager, 'window:resize', this.onResize, this);
            this.listenTo(_utilsEventManager.EventManager, 'window:scroll', this.stick, this);
            this.listenTo(_utilsEventManager.EventManager, 'list:change', this.stick, this);

            if (opts.autoRun) {
                _underscore.defer(this.stick.bind(this));
            }

            this.view.$el.addClass('stickable');
        },

        stick: function stick() {
            var _options = this.options;
            var gap = _options.gap;
            var offset = _options.offset;

            var doc = document.documentElement;
            var scrollPosition = (window.pageYOffset || doc.scrollTop) - (doc.clientTop || 0);
            var positionFromTop = this.view.$el.offset().top;
            var limitBottom = positionFromTop + this.view.$el.outerHeight();
            var collisionPosition = scrollPosition + offset + gap;

            // if the scroll is reaching the container (this.view.el)
            if (collisionPosition >= positionFromTop && collisionPosition <= limitBottom) {
                var width = this.target.clientWidth;
                // creates a clone with the same dimension as when you have position fixed the element is dettached from its container
                if (!this.clone) {
                    this.clone = this.$target.clone();
                    this.$target.before(this.clone);
                    this.$target.addClass('stick').css({
                        width: width + 'px'
                    });
                }

                // if you are close to the bottom of the container, changes its class to .stuck (position: absolute; bottom: 0). We expect the container to have position: relative;
                if (collisionPosition >= limitBottom - this.$target.outerHeight()) {
                    this.$target.removeClass('stick').addClass('stuck');

                    // if it's not close to the bottom of the container, it will have the class .stick (position: fixed) and the top will be the offset
                } else {
                        this.$target.removeClass('stuck').addClass('stick').css({
                            top: offset + 'px',
                            width: width + 'px'
                        });
                    }
            } else {
                this.removeClone();
            }
        },

        onResize: function onResize() {
            if (this.clone) {
                var width = this.clone.width();
                this.$target.css({
                    width: width + 'px'
                });
            }
        },

        removeClone: function removeClone(isDestroying) {
            if (this.clone) {
                this.$target.removeClass('stick').removeClass('stuck');
                this.$target.css({
                    width: '100%'
                });
                this.clone.remove();
                this.clone = null;
            }
        },

        onBeforeDestroy: function onBeforeDestroy() {
            this.removeClone();
        }
    });
});
define("confluence-dashboard/behaviors/stickable/stickable.js", function(){});

define('confluence-dashboard/behaviors/infinite-loading', ['exports', 'module', 'marionette', '../utils/event-manager', 'jquery', 'underscore'], function (exports, module, _marionette, _utilsEventManager, _jquery, _underscore) {
    'use strict';

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

    var _$ = _interopRequireDefault(_jquery);

    /**
     * Infinite Loading
     *
     * Add this behaviour to the view you want infinite loading for the current collection.
     *
     * It will call the fetch method of the collection with the next offset when reach the bottom;
     *
     * Params:
     *  - target: the list container that displays the scrollable data
     *  - offset: distance from the bottom when it should load more
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *      infiniteLoading: {
                behaviourClass: InfiniteLoading,
                target: '.list-container'
            }
     *   }
     * })
     */

    module.exports = _marionette.Behavior.extend({
        defaults: {
            offset: 50
        },

        initialize: function initialize() {
            this.listenToOnce(this.view, 'render:collection', this.setup);
        },

        setup: function setup() {
            var _this = this;

            if (!this.view.collection.loadMore) {
                return;
            }
            this.$target = (0, _$['default'])(this.options.target, this.$el);
            this.$window = (0, _$['default'])(window);

            // basic status to skip execution when there is one request running;
            this.loading = false;

            // check for more loading on scroll
            this.listenTo(_utilsEventManager.EventManager, 'window:scroll', this.checkPosition, this);

            // if there is space after a fetch, fetch some moar and then some moar..
            this.listenTo(this.view.collection, 'sync', this.checkPosition);

            // if there is space immediately, load more.. but only after the sync event has finished raised to other behaviours
            _underscore.defer(function () {
                return _this.checkPosition();
            });
        },

        checkPosition: function checkPosition() {
            var _this2 = this;

            if (this.loading) {
                return;
            }

            var scrollPosition = this.$window.scrollTop() + this.$window.height();
            var targetHeight = this.$target.height();

            var positionFromTop = this.$target.offset().top + targetHeight;

            var waypoint = positionFromTop - this.options.offset;

            if (scrollPosition > waypoint) {
                this.loadMore().then(function () {
                    return _this2.checkPosition();
                });
            }
        },

        loadMore: function loadMore() {
            var _this3 = this;

            // lock while performing this request
            this.loading = true;
            return this.view.collection.loadMore()
            // unlock on complete or reject
            .always(function () {
                _this3.loading = false;
            });
        }
    });
});
define("confluence-dashboard/behaviors/infinite-loading.js", function(){});

define('confluence-dashboard/behaviors/filterable', ['exports', 'module', 'marionette', '../utils/event-manager', 'underscore', 'jquery', 'ajs', '../utils/analytics', '../utils/strings'], function (exports, module, _marionette, _utilsEventManager, _underscore, _jquery, _ajs, _utilsAnalytics, _utilsStrings) {
    'use strict';

    /**
     * Filterable
     *
     * Add this behaviour to the view you want add a remote filter
     *
     * It will call the search method of the collection with the text from the field
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *      filterable: {
                behaviourClass: Filterable
            }
     *   }
     * })
     *
     */

    module.exports = _marionette.Behavior.extend({
        ui: {
            filter: '[name=filter]'
        },

        events: {
            'input @ui.filter': 'doFilter'
        },

        initialize: function initialize() {
            // behaviors are initialized before the view so collections and models are not available yet. Because of that we should use options.model / options.collection within initialize.

            if (!this.view.options.collection || !this.view.options.collection.search) {
                console.warn('Behavior:Filterable: The collection is missing or not implementing a search method');
            }

            this.doFilter = _underscore.debounce(this.doFilter, 400);
        },

        onShow: function onShow() {
            var _this = this;

            this.listenTo(this.view.collection, 'sync', this.updateFilterString);
            this.listenTo(this.view.collection, 'error', function () {
                return _this.view.collection.reset([]);
            });
        },

        updateFilterString: function updateFilterString(options) {
            // this field is used for the webdriver test - do not remove;
            this.view.$el.find('.list-container').attr('data-filter-string', this.view.getFilterString());
            this.fixAuiSidebar();
        },

        fixAuiSidebar: function fixAuiSidebar() {
            // hack to fix the sidebar not re-docking itself after the page height decreases and becomes non-scrolling
            _ajs.sidebar('.aui-sidebar').setPosition();
        },

        doFilter: function doFilter() {
            var query = this.ui.filter.val();
            query = (0, _utilsStrings.normalizeForCQL)(query);
            var view = this.view;

            if (query.length) {
                if (query === this.lastQuery) {
                    return;
                }

                this.lastQuery = query;

                view.collection.search({
                    query: query
                });

                _utilsAnalytics.publish('filter.submit');
            } else {
                // if the search field is empty, let's load the original results
                view.collection.safeFetch();
                _utilsAnalytics.publish('filter.clear');

                this.lastQuery = null;
            }
        }
    });
});
define("confluence-dashboard/behaviors/filterable.js", function(){});

define('confluence-dashboard/behaviors/progress-indicator', ['exports', 'module', 'marionette', 'ajs', 'underscore'], function (exports, module, _marionette, _ajs, _underscore) {
    'use strict';

    /**
     * Progress Indicator
     *
     * Add this behaviour to the view you want to add a spinner in sync with your model or collection
     *
     * It will start the spinner when the event `request` is fired and remove it when `sync` is fired.
     *
     * Params:
     *  - dataSource: collection or model - the kind of data to bind the events
     *  - container: the html selector where to add the spinner
     *  - size: 'small | medium | large' - see https://docs.atlassian.com/aui/latest/docs/spinner.html
     *
     * Usage:
     * ItemView.extend({
     *  behaviors: {
     *      progressIndicator: {
                behaviorClass: ProgressIndicator,
                target: 'collection' (or model)
                container: '.spinner-container',
                size: 'medium'
            }
     *   }
     * })
     *
     * TODO: write tests
     */
    module.exports = _marionette.Behavior.extend({
        defaults: {
            size: 'small',
            dataSource: 'collection'
        },

        // will become modelEvents or collectionEvents depending on the dataSource option
        dataEvents: {
            'request': 'onRequest',
            'sync': 'spinStop',
            'error': 'spinStop'
        },

        initialize: function initialize() {
            if (!this.options.dataSource) {
                throw "Behavior:ProgressIndicator: A dataSource should be defined. Use 'collection' or 'model'";
            }

            // define modelEvents or collectionEvents depending on the dataSource
            this[this.options.dataSource + 'Events'] = this.dataEvents;
        },

        onAttach: function onAttach() {
            if (this.options.autoRun) {
                this.spin();
            }
        },

        onRequest: function onRequest(model, xhr, options) {
            if (options.type !== 'POST' && options.type !== 'DELETE') {
                this.spin();
            }
        },

        spin: function spin() {
            var _this = this;

            // if spin is called before the view is available, listen to the view 'show' event to show the loader
            if (!this.view._isShown) {
                this.listenToOnce(this.view, 'show', this.spin, this);
            } else {
                // required for webdriver tests
                this.$el.addClass('loading');

                // avoid showing the spinner for requests that finish quickly
                this._currentSpinDelay = setTimeout(function () {
                    _ajs.$(_this.options.container, _this.view.el).spin({
                        size: _this.options.size,
                        zIndex: 2000
                    });
                }, 200);
            }
        },

        spinStop: function spinStop() {
            // clear the spin delay, if one is set
            if (this._currentSpinDelay) {
                clearTimeout(this._currentSpinDelay);
                this._currentSpinDelay = null;
            }

            // if spinStop is called before the view is available, remove the listener to the view 'show' so it won't show the loader
            if (!this.view._isShown) {
                this.stopListening(this.view, 'show', this.spin, this);
            }

            this.options.autoRun = false;

            // required for webdriver tests
            this.$el.removeClass('loading');

            _ajs.$(this.options.container, this.view.el).spinStop();
        }
    });
});
define("confluence-dashboard/behaviors/progress-indicator.js", function(){});

define('confluence-dashboard/modules/stream-item/stream-item-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', 'confluence/hover-user'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _confluenceHoverUser) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        template: _confluenceDashboardSoyTemplates.Stream.item,
        tagName: 'li',
        className: 'grouping',

        onDomRefresh: function onDomRefresh() {
            (0, _confluenceHoverUser)();
        }
    });
});
define("confluence-dashboard/modules/stream-item/stream-item-view.js", function(){});

define('confluence-dashboard/modules/list-item/list-item-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates) {
    'use strict';

    module.exports = _marionette.ItemView.extend({
        template: _confluenceDashboardSoyTemplates.Lists.item,
        className: 'item',
        tagName: 'li'
    });
});
define("confluence-dashboard/modules/list-item/list-item-view.js", function(){});

define('confluence-dashboard/modules/group/group-composite-view', ['exports', 'module', 'marionette', 'confluence-dashboard/soy-templates', '../../utils/event-manager', '../../modules/list-item/list-item-view', '../../behaviors/stickable/stickable'], function (exports, module, _marionette, _confluenceDashboardSoyTemplates, _utilsEventManager, _modulesListItemListItemView, _behaviorsStickableStickable) {
    'use strict';

    module.exports = _marionette.CompositeView.extend({
        template: _confluenceDashboardSoyTemplates.Groups.group,
        childViewContainer: '.group-container',
        tagName: 'li',
        className: 'group-wrapper',

        behaviors: {
            stickSubHeaders: {
                behaviorClass: _behaviorsStickableStickable,
                element: '.sticky-header',
                autoRun: true,
                offset: 75
            }
        },

        initialize: function initialize() {
            this.collection = this.model.get('items');
        },

        onRenderCollection: function onRenderCollection() {
            this.trigger('group:rendered');
        }
    });
});
define("confluence-dashboard/modules/group/group-composite-view.js", function(){});

define('confluence-dashboard/modules/group/group-collection', ['exports', 'module', '../../core/shared/base-collection', 'backbone', 'underscore', '../../utils/date-utils'], function (exports, module, _coreSharedBaseCollection, _backbone, _underscore, _utilsDateUtils) {
    'use strict';

    /**
     * Group Collection
     *
     * It's a clean way to group a collection and keep the normal behaviour of marionette collection/composite views when dealing with its models.
     *
     * Basically we need to group the original collection and return a new one. The problem is that the new one doesn't have the attributes of the full one. So we proxy all the actions to the full one and listen to changes there to update here.
     *
     * It's not quite straight forward
     */

    module.exports = _coreSharedBaseCollection.extend({
        idAttribute: 'periodKey',

        comparator: 'periodOrder',

        // define the methods that should be proxy'ed to the collectionToGroup
        methodsToProxy: ['url', 'fetch', 'safeFetch', 'loadMore', 'search'],

        initialize: function initialize(col, options) {
            if (!options.collectionToGroup) {
                throw 'Grouped Collection requires a collection to group';
            }

            if (!options.collectionToGroup.groupMethod) {
                throw 'The target collection should implement a groupMethod';
            }

            this.collectionToGroup = options.collectionToGroup;

            this.proxyMethods();
            this.proxyEvents();
        },

        // in order to keep marionette bindings, this collection should have all these methods and just proxy to the sub collection
        proxyMethods: function proxyMethods() {
            var groupInstance = this;

            this.methodsToProxy.forEach(function (method) {
                groupInstance[method] = function () {
                    return groupInstance.collectionToGroup[method].apply(groupInstance.collectionToGroup, arguments);
                };
            });
        },

        proxyEvents: function proxyEvents() {
            this.listenTo(this.collectionToGroup, 'request', this._request);
            this.listenTo(this.collectionToGroup, 'sync', this._sync);
            this.listenTo(this.collectionToGroup, 'remove', this._remove);
            this.listenTo(this.collectionToGroup, 'add', this._add);
            this.listenTo(this.collectionToGroup, 'error', this._error);
        },

        _request: function _request(model, xhr, options) {
            this.trigger('request', this, xhr, options);
        },

        _sync: function _sync(model, data, options) {
            this._group();
            this.trigger('sync', this, data, options);
        },

        _remove: function _remove(model, originalCollection) {
            var existent = this.filter(function (periodModel) {
                return periodModel.get('items').indexOf(model) !== -1;
            });

            if (existent.length) {
                var collection = existent[0].get('items');
                collection.remove(model);

                if (!collection.length) {
                    existent[0].destroy();
                }
            }
        },

        _add: function _add() {
            // add will be triggered many times at once so we need to debounce to do only one grouping.
            _underscore.debounce(this._group, 50);
        },

        _error: function _error() {
            this.trigger('error');
        },

        _group: function _group() {
            var _this = this;

            var periods = {};

            var groups = this.collectionToGroup.groupBy(this.collectionToGroup.groupMethod);

            // join all the days in groups by period
            _underscore.each(groups, function (models, day) {
                var period = _utilsDateUtils.getPeriod(day);
                periods[period] = periods[period] || [];
                periods[period] = periods[period].concat(models);
            });

            // create a model for each period
            _underscore.each(periods, function (items, period) {
                var model = _this.findWhere({ periodKey: period });

                if (model) {
                    model.get('items').set(periods[period], { remove: false });
                } else {
                    _this.add({
                        periodKey: period,
                        periodOrder: _utilsDateUtils.getPeriodOrder(period),
                        title: _utilsDateUtils.getPeriodTitle(period),
                        items: new _backbone.Collection(periods[period], {
                            comparator: _this.collectionToGroup.comparator
                        })
                    });
                }
            });
        }
    });
});
define("confluence-dashboard/modules/group/group-collection.js", function(){});

define('confluence-dashboard/utils/date-utils', ['exports', 'module', 'underscore', 'ajs'], function (exports, module, _underscore, _ajs) {
    'use strict';

    function _today() {
        var date = new Date();
        return date;
    }

    function parseUserTime(userTime) {
        return new Date(userTime).toDateString();
    }

    function weekPeriod(time) {
        // 0 = sunday - 7 = saturday
        var weekDay = time.getDay();
        // determine when was sunday
        time.setDate(time.getDate() - weekDay);

        var sunday = new Date(time.getTime());
        sunday.setHours(0, 0, 0, 0);

        var saturday = new Date(time.getTime() + 60 * 60 * 24 * 6 * 1000);
        saturday.setHours(23, 59, 59, 999);

        return {
            begin: sunday,
            end: saturday
        };
    }

    var periodTitles = {
        today: _ajs.I18n.getText("period.today"),
        yesterday: _ajs.I18n.getText("period.yesterday"),
        lastSevenDays: _ajs.I18n.getText("period.lastSevenDays"),
        lastThirtyDays: _ajs.I18n.getText("period.lastThirtyDays"),
        older: _ajs.I18n.getText("period.older")
    };

    // function monthPeriod
    var DateUtils = {
        today: function today(userTime) {
            var parsedUserTime = parseUserTime(userTime);
            var time = _today();
            return parsedUserTime === time.toDateString();
        },

        yesterday: function yesterday(userTime) {
            var parsedUserTime = parseUserTime(userTime);
            var time = _today();
            time.setDate(time.getDate() - 1);
            return parsedUserTime === time.toDateString();
        },

        lastSevenDays: function lastSevenDays(userTime) {
            var time = _today();
            var sevenDaysAgo = _today().setDate(time.getDate() - 7);
            var userDate = new Date(userTime);
            return userDate >= sevenDaysAgo && userDate <= time;
        },

        lastThirtyDays: function lastThirtyDays(userTime) {
            var time = _today();
            var thirtyDaysAgo = _today().setDate(time.getDate() - 30);
            var userDate = new Date(userTime);
            return userDate >= thirtyDaysAgo && userDate <= time;
        },

        getPeriod: function getPeriod(userTime) {
            var methods = _underscore.pick(DateUtils, 'today', 'yesterday', 'lastSevenDays', 'lastThirtyDays');

            var match = _underscore.find(methods, function (method, key) {
                return method(userTime);
            });

            if (match) {
                var key = _underscore.invert(methods)[match];
                return key;
            }

            return 'older';
        },

        getPeriodTitle: function getPeriodTitle(period) {
            return periodTitles[period];
        },

        getPeriodOrder: function getPeriodOrder(period) {
            return _underscore.keys(periodTitles).indexOf(period);
        },

        /**
         * Returns 0 if two dates are equal, 1 if the first date is greater than the second, or -1 if the first date is
         * less than the second. This is useful when sorting timestamps in descending order.
         */
        compareTimestamps: function compareTimestamps(firstDate, secondDate) {
            return Math.max(-1, Math.min(1, firstDate - secondDate));
        },

        toISODate: function toISODate(dateObj) {
            var paddedDate = dateObj.getDate();
            paddedDate = paddedDate <= 9 ? "0" + paddedDate : paddedDate;
            var paddedMonth = dateObj.getMonth() + 1;
            paddedMonth = paddedMonth <= 9 ? "0" + paddedMonth : paddedMonth;

            // Constructing a Date object with an ISO format date without a timezone makes it assume UTC.
            // When the local timezone is less than UTC that will lead to the date changing to yesterday.
            // However, we want this to be in the current user's timezone, so we append the local timezone so the date is
            // always midnight today in local time.
            var localTimezoneOffset = new Date().getTimezoneOffset();
            var timezoneHours = Math.abs(Math.floor(localTimezoneOffset / 60));
            timezoneHours = timezoneHours <= 9 ? "0" + timezoneHours : timezoneHours;
            var timezoneMinutes = Math.abs(localTimezoneOffset % 60);
            timezoneMinutes = timezoneMinutes <= 9 ? "0" + timezoneMinutes : timezoneMinutes;
            var timezoneString = (localTimezoneOffset < 0 ? "+" : "-") + timezoneHours + ":" + timezoneMinutes;
            return dateObj.getFullYear() + "-" + paddedMonth + "-" + paddedDate + "T00:00:00" + timezoneString;
        }
    };

    module.exports = DateUtils;
});
define("confluence-dashboard/utils/date-utils.js", function(){});

define('confluence-dashboard/utils/ensure-component', ['exports', 'module', 'jquery'], function (exports, module, _jquery) {
    'use strict';

    var executeUntilTrue = function executeUntilTrue(obj) {
        if (!obj.condition()) {
            setTimeout(function () {
                executeUntilTrue(obj);
            }, obj.interval);
        } else {
            obj.callback();
        }
    };

    var isResolved = function isResolved(component) {
        // The current version of skate on confluence uses __skate as class to identify that
        // the component has been resolved while new versions of skate are adding a resolved
        // attribute;
        return (0, _jquery)(component).hasClass('__skate') || component.hasAttribute('resolved');
    };

    module.exports = function (component) {
        var interval = arguments.length <= 1 || arguments[1] === undefined ? 200 : arguments[1];

        var deferred = _jquery.Deferred();

        if (isResolved(component)) {
            deferred.resolve();
        } else {

            executeUntilTrue({
                interval: interval,

                condition: function condition() {
                    return isResolved(component);
                },

                callback: function callback() {
                    deferred.resolve();
                }
            });
        }

        return deferred.promise();
    };
});
define('confluence-dashboard/utils/feature-discovery', ['exports', 'module', 'confluence/legacy', 'confluence/storage-manager'], function (exports, module, _confluenceLegacy, _confluenceStorageManager) {
    'use strict';

    var storage = (0, _confluenceStorageManager)('dashboard', 'feature-discovery');

    module.exports = {
        forPlugin: function forPlugin(pluginKey) {
            var dashboardFeatureDiscovery = _confluenceLegacy.FeatureDiscovery.forPlugin(pluginKey);

            // The original method is shouldShow but it only allows ONE feature discovery per page pop.
            // The new dashboard is a single page app and we want to show many features at once, so this method
            // will just return true if the item has not been discovered yet.
            dashboardFeatureDiscovery.canShow = function (feature) {
                if (storage.getItem(feature)) {
                    return false;
                }

                return !dashboardFeatureDiscovery.listDiscovered().some(function (item) {
                    return item === feature;
                });
            };

            dashboardFeatureDiscovery.markDiscoveredSafe = function (featureKey, callback) {
                // to avoid showing features again when the back button is pressed, we save it on local storage too.
                var oneDayInSeconds = 60 * 60 * 24;
                storage.setItemQuietly(featureKey, true, oneDayInSeconds);

                dashboardFeatureDiscovery.markDiscovered(featureKey, callback);
            };

            return dashboardFeatureDiscovery;
        }
    };
});
define('confluence-dashboard/utils/module-starter', ['exports', 'module', './event-manager'], function (exports, module, _eventManager) {
    'use strict';

    /**
     * ModuleStarter module
     * @module confluence-dashboard/utils/module-starter
     */

    module.exports = {
        /**
         * Registers a callback to run after the app is started.
         * @param initMethod - The function to be called.
         */
        register: function register(initMethod) {
            var app = _eventManager.ReqRes.request('app');

            if (!(initMethod && typeof initMethod === 'function')) {
                throw 'ModuleStarter.register needs a function as callback';
            }

            if (!app) {
                throw 'ModuleStarter is being called before the app is available, please review your dependencies';
            }

            if (app.isStarted) {
                // if it' already started we start the module immediately
                initMethod();
            } else {
                // if not, we listen to the event that will be fired when it starts
                app.on('start', initMethod);
            }
        }
    };
});

define("confluence-dashboard", function(){});

require(["confluence-dashboard/index"]);
