define('acm/app/views/data-supplier-view', [
    'ajs',
    'backbone',
    'csvgenerator'
],
function (
    AJS,
    Backbone,
    Csvgenerator
) {
    'use strict';

    var DataSupplierView = Backbone.View.extend(
        {
            el: '#cluster-monitoring-data-container',
            headerEl: '.header-container',
            actionsEl: '.actions-container',
            tableEl: '.data-supplier-table-container',

            containerView: Cluster.Monitoring.Templates.supplierContainerView,
            headerView: Cluster.Monitoring.Templates.headerContainerView,
            actionsView: Cluster.Monitoring.Templates.actionsView,
            supplierDataTableView: Cluster.Monitoring.Templates.supplierDataTable,

            events: {
                'click .export-link': 'exportCsv',
                'click .refresh-link': 'refreshView',
                'click .clustering-view-link': 'goBack'
            },

            initialize: function () {
                var onErrorCallback = function onError () {
                    this.renderActions(true, true, false);
                };
                this.listenTo(this.model, 'error', onErrorCallback);
                this.listenTo(this.model, 'sync', this.renderSupplierData);
            },

            //clearing all dom updates, can't call "this.remove" because it will remove container el :-(
            goBack: function (e) {
                this.$el.empty();

                this.undelegateEvents();
                this.stopListening();

                this.model.destroy();
            },

            refreshView: function (e) {
                e.preventDefault();
                
                //unclick button, won't happen automatically
                this.$(e.currentTarget).blur();
                
                this.model.fetch();
            },

            exportCsv: function (e) {
                var jsonModel = this.model.toJSON();

                var csv = Csvgenerator.generate(jsonModel.data.columns, jsonModel.data.rows);

                var uri = 'data:text/csv;charset=utf-8,' + encodeURI(csv);

                this.$el.find('.export-link').attr('href', uri);
                this.$el.find('.export-link').attr('download', 'export.csv');

                //unclick button, won't happen automatically
                this.$(e.currentTarget).blur();
            },

            renderViewContainer: function() {
                AJS.trigger('analyticsEvent', {
                    name: 'monitoringDataSupplierView.rendered',
                    data: {
                        pluginKey: this.model.pluginKey,
                        moduleKey: this.model.moduleKey
                    }
                });

                this.$el.empty();

                var containerView = this.containerView({
                    pluginKey: this.model.pluginKey,
                    moduleKey: this.model.moduleKey
                });

                this.$el.html(containerView);
            },

            renderActions: function (displayClusteringViewLink, displayRefreshLink, displayExportLink) {
                this.renderViewContainer();

                this.$el.find(this.actionsEl).html(this.actionsView({
                        displayClusteringViewLink: displayClusteringViewLink,
                        displayRefreshLink: displayRefreshLink,
                        displayExportLink: displayExportLink
                    }
                ));
            },

            renderSupplierData: function () {
                this.renderActions(true, true, true);

                var jsonModel = this.model.toJSON();

                this.$el.find(this.headerEl).html(this.headerView({title: jsonModel.supplier.i18nNameLabel}));

                var supplierView = this.supplierDataTableView({
                    pluginKey: this.model.pluginKey,
                    moduleKey: this.model.moduleKey,
                    nodeId: this.model.nodeId,
                    label: jsonModel.supplier.i18nNameLabel,
                    columns: jsonModel.data.columns,
                    rows: jsonModel.data.rows
                });

                var tableEl = this.$el.find(this.tableEl);

                tableEl.html(supplierView);

                this.customizeDataTable(tableEl);
            },

            //makes table/filter aui compliant
            customizeDataTable: function(el) {
                el.find('.data-supplier-table').dataTable(
                    {
                        asStripClasses: [],
                        paging: false,
                        bPaginate: false,
                        oLanguage: {
                            //Removing label text
                            sSearch: ''
                        }
                    }
                );

                var dataTablesFilter = el.find('.dataTables_wrapper .dataTables_filter');
                //making field look like aui
                dataTablesFilter.find('input').addClass('text');
                //adding watermark text
                dataTablesFilter.find('input').attr('placeholder', 'Search');

                //making field to look even more aui
                dataTablesFilter.find('label').addClass('field-group');
            }
        }
    );

    return DataSupplierView;
});