define('hipchat/invite/service', [
    'hipchat/admin/configure-integration-data',
    'jquery',
    'underscore'
], function (
    integrationData,
    $,
    _
) {

    var MAX_INVITATIONS_PER_REQUEST = 24;
    var MAX_USER_PRE_POPULATION_COUNT = 12;

    var users = [];
    var events = $({});
    var uiConfig = {};

    var routes = {
        inviteUsersURL: AJS.contextPath() + '/rest/hipchat/integration/latest/users/invite'
    };

    function getMaxInvitationsPerRequest() {
        return MAX_INVITATIONS_PER_REQUEST;
    }

    function getUsers() {
        return users;
    }

    function findUser(username) {
        return _.find(users, function(userInList) {
            if (userInList.username == username) {
                return userInList;
            }
        });
    }

    function addUser(user) {
        user.username = $("<div/>").text(user.username).html();
        var foundUser = findUser(user.username);
        if (!_.contains(users, foundUser)) {
            users.push(user);
            events.trigger('user-added', user);
            checkInvitationLimit();
        }
    }

    function removeUser(username) {
        var user = findUser(username);

        if(user.isSuggested) {
            AJS.trigger('analyticsEvent', {name: 'notifications.hipchat.user.invite.remove.suggested', data: {}});
        }

        if (_.contains(users, user)) {
            users = _.filter(users, function(userInList) {
                return userInList.username != username;
            });
            events.trigger('user-removed', user.username);
            checkInvitationLimit();
        }
    }

    function checkInvitationLimit() {
        var userCount = users.length;
        if (userCount >= MAX_INVITATIONS_PER_REQUEST) {
            $("#invite-users-picker-container").hide();
            $("#inviation-limit-message").show();
        } else {
            $("#inviation-limit-message").hide();
            $("#invite-users-picker-container").show();
        }
    }

    function clear() {
        users = [];
        events.off();
    }

    function sendInvitations() {
        AJS.$("#invite-users-picker-container input[type='text']").prop("disabled", true);
        AJS.$('#hipchat-invite-form').on('select2-opening', function(e) { e.preventDefault(); });
        uiConfig.invitationProcessingStartedRenderer.call();
        var usernames = _.map(getUsers(), function(user) {
            return user.username;
        });
        var userNamesJson = JSON.stringify({"user-names": usernames});
        $.ajax({
            url: routes.inviteUsersURL,
            type: "POST",
            contentType: 'application/json',
            data: userNamesJson
        })
        .done(function(invitationResults) {
            for (var i = 0; i < invitationResults.resultSet.length; i++) {
                var result = invitationResults.resultSet[i];
                uiConfig.resultRenderer(result.userName, result.errorMessage);
            }
            uiConfig.renderDoneStep.call();
        })
        .fail(function(xhr) {
            var title = AJS.I18n.getText("plugin.hipchat.admin.invite.error.title");
            var messageText = AJS.escapeHtml(xhr.responseText);
            var messageReason = AJS.escapeHtml(AJS.I18n.getText("plugin.hipchat.admin.invite.resource.error.reason", xhr.status, xhr.statusText));
            uiConfig.errorRenderer.call(this, title, messageText, messageReason);
        })
        .always(function() {
            uiConfig.invitationProcessingCompletedRenderer.call();
        });
    }

    function handleDone() {
        var adminConfigurationPageURI = $("#admin-configuration-page-uri").val();
        window.location = adminConfigurationPageURI;
    }

    function initialise(
            invitationProcessingStartedRenderer,
            invitationProcessingCompletedRenderer,
            resultRenderer,
            renderDoneStep,
            errorRenderer) {
        uiConfig.invitationProcessingStartedRenderer = invitationProcessingStartedRenderer;
        uiConfig.invitationProcessingCompletedRenderer = invitationProcessingCompletedRenderer;
        uiConfig.resultRenderer = resultRenderer;
        uiConfig.renderDoneStep = renderDoneStep;
        uiConfig.errorRenderer = errorRenderer;
        routes.findInviteUsersURL = $("#hipchat-find-invite-users-uri").val() + "?max-results=" + MAX_USER_PRE_POPULATION_COUNT;
        $.ajax(routes.findInviteUsersURL, {
            type: "GET"
        })
            .done(function(userReferenceSet) {
                var listHasBeenManuallyPopulated = users.length > 0;
                if (listHasBeenManuallyPopulated) {
                    // QA decision: don't auto populate the list since it would be disruptive to the user.
                } else {
                    if (userReferenceSet.users) {
                        var userCount = userReferenceSet.users.length;
                        var usersToDisplay = Math.min(userCount, MAX_USER_PRE_POPULATION_COUNT);
                        for (var i = 0; i < usersToDisplay; i++) {
                            var userReference = userReferenceSet.users[i];
                            var uiUser = {
                                username: userReference.name,
                                avatarUrl: userReference.extraSmallPictureURI,
                                fullname: userReference.displayName,
                                email: userReference.email,
                                isSuggested: true
                            };
                            addUser(uiUser);
                        }
                    }
                }
            });
    }

    return {
        getMaxInvitationsPerRequest: getMaxInvitationsPerRequest,
        getUsers: getUsers,
        addUser: addUser,
        removeUser: removeUser,
        clear: clear,
        initialise: initialise,
        sendInvitations: sendInvitations,
        handleDone: handleDone,
        events: events
    }
});
