UPM.define('AddonReviewView',
    [
        'jquery',
        'AddonReviewTemplate',
        'BaseView',
        'UpmFormats',
        'UpmMessageFactory',
        'UpmStrings'
    ], function($,
                addonReviewTemplate,
                BaseView,
                UpmFormats,
                UpmMessageFactory,
                UpmStrings) {

    "use strict";

    /**
     * Provides a UI element for adding a review, or for viewing/updating the user's previous review of this add-on.
     * Getting the review data requires a separate REST request, so the view first initializes itself in an empty
     * state, then makes the REST request via the add-on model and finishes rendering itself when ready.
     */
    return BaseView.extend({

        template: addonReviewTemplate,
        
        events: {
            'click span.star': '_selectStar',
            'mouseover span.star': '_startHoveringOnStar',
            'mouseout span.star': '_stopHoveringOnStar',
            'click .upm-review': '_submitReview'
        },

        _getData: function() {
            var review;
            if (!(this.model.isApplicationPlugin && this.model.isApplicationPlugin())) {
                review = this.reviewModel && this.reviewModel.toJSON();
            }

            return {
                plugin: this.model.toJSON(),
                reviewModel: review
            };
        },

        _postRender: function() {
            var me = this;
            if (!this.reviewModel) {
                this.model.loadReviewModel()
                    .done(function(reviewModel) {
                        me.reviewModel = reviewModel;
                        me.render();
                        UPM.trace('review-container-loaded');
                    });
            }
        },

        _showSuccessMessage: function(selector) {
            var $msg = this.$el.find(selector);
            $msg.removeClass('hidden');
            setTimeout(function() {
                    $msg.fadeOut(2000, function() {
                        $msg.addClass('hidden').show();
                    });
                }, 3000);
        },

        _hideMessages: function() {
            this.$el.find('.plugin-review-messages').empty();
        },

        _isBusy: function() {
            return this.$el.find('.plugin-review-container').hasClass('busy');
        },

        _setBusy: function(flag) {
            this.$el.find('.plugin-review-container').toggleClass('busy', flag);
        },

        _setVisibleStars: function(rating) {
            var $container = this.$el.find('.rating-container');
            $container.removeClass('rated-1 rated-2 rated-3 rated-4');
            if (rating) {
                $container.addClass('rated-' + rating);
            }
        },

        _selectStar: function(e) {
            var me = this,
                rating = $(e.target).attr('data-rating');

            e.preventDefault();
            if (this._isBusy()) {
                return;
            }
            this._setVisibleStars(rating);
            this._setBusy(true);
            this._hideMessages();
            this.reviewModel.updateStars(rating)
                .always(function() {
                    me._setBusy(false);
                })
                .done(function() {
                    me._showMessage(UpmMessageFactory.newSuccessMessage(null, UpmStrings['upm.messages.rating.saved']));
                })
                .fail(_.bind(this._onReviewUpdateFailure, this));
        },

        _showMessage: function(msgView) {
            this.$el.find('.plugin-review-messages').append(msgView.render().$el);
        },

        _startHoveringOnStar: function(e) {
            if (!this._isBusy()) {
                var target = $(e.target),
                    rating = target.attr('data-rating');
                this._setVisibleStars(rating);
            }
        },

        _stopHoveringOnStar: function(e) {
            if (!this._isBusy()) {
                this._setVisibleStars(this.reviewModel.getStars());
            }
        },

        _submitReview: function(e) {
            var rating = this.reviewModel.getStars(),
                review = this.$el.find('.plugin-review').val().trim(),
                me = this;

            e.preventDefault();
            if (this._isBusy()) {
                return;
            }
            this._hideMessages();

            if (!rating) {
                this._showMessage(UpmMessageFactory.newErrorMessage(null, UpmStrings['upm.messages.review.error.rate.required']));
            } else if (!review) {
                this._showMessage(UpmMessageFactory.newErrorMessage(null, UpmStrings['upm.messages.review.error.review.required']));
            } else {
                this._setBusy(true);
                this.reviewModel.updateReview(review, rating)
                    .always(function() {
                        me._setBusy(false);
                    })
                    .done(function() {
                        me._showMessage(UpmMessageFactory.newSuccessMessage(null, UpmStrings['upm.messages.review.saved']));
                    })
                    .fail(_.bind(this._onReviewUpdateFailure, this));
            }
        },

        _onReviewUpdateFailure: function(request) {
            this._setVisibleStars(this.reviewModel.getStars());
            if (this.reviewModel.getMarketplaceLink()) {
                this._showMessage(UpmMessageFactory.newErrorMessage(null,
                    UpmFormats.format(UpmStrings['upm.messages.review.submit.error'], this.reviewModel.getMarketplaceLink())
                ));
            } else {
                this.model.signalAjaxError(request);
            }
        }
    });
});
