/*
This code maintains a cache of calendar data, so we do not need to refetch data from the server unnecessarily,
especially when the user is scrolling quickly through the timeline view. The cache for each calendar contains a
start and end date that defines the range of data that is held by the cache. More data can be added to the cache
for a particular calendar by calling addDataToCalendar(), which will make sure to check for duplicate events, and
update the start and end dates for that cache.

NOTE: as the cache only maintains a single range it is possible to create gaps in the cache by pressing the 'Today'
button, therefore all caches must be removed when the 'Today' button is pressed.

Whenever a calendar is modified, e.g. an event is moved, deleted etc then the cache for that calendar should be
removed by calling removeCalendar().
 */

define("tc/calendar-cache", ["jquery"], function ($) {
    function CalendarCache() {
        var calendarCache = [];

        function appendDataToCalendar(calendar, data) {
            var dupe = false;
            //We need to filter out any duplicate events
            for (var i = 0; i < data.length; i++) {
                for (var j = 0; j < calendarCache[calendar].data.length; j++) {
                    if (data[i].id === calendarCache[calendar].data[j].id) {
                        dupe = true;
                        break;
                    }
                }

                if (!dupe) {
                    calendarCache[calendar].data.push(data[i]);
                }

                dupe = false;
            }
        }

        return {
            hasCalendar: function(calendarId) {
                return calendarCache[calendarId];
            },

            getCalendarStart: function(calendarId) {
                if (calendarCache[calendarId]) {
                    return calendarCache[calendarId].start;
                }

                return null;
            },

            getCalendarEnd: function(calendarId) {
                if (calendarCache[calendarId]) {
                    return calendarCache[calendarId].end;
                }

                return null;
            },

            addDataToCalendar: function(calendarId, data, start, end) {

                if (!calendarCache[calendarId]) {
                    calendarCache[calendarId] = {"data": data};
                } else {
                    appendDataToCalendar(calendarId, data);
                }

                if (!calendarCache[calendarId].start || start.getTime() < calendarCache[calendarId].start.getTime()) {
                    calendarCache[calendarId].start = start;
                }

                if (!calendarCache[calendarId].end || end.getTime() > calendarCache[calendarId].end.getTime()) {
                    calendarCache[calendarId].end = end;
                }
            },

            removeCalendar: function(calendarId) {
                if ($.isArray(calendarId)) {
                    var self = this;
                    $.each(calendarId, function(idIdx, calendarId) {
                        if ($.isPlainObject(calendarId) && calendarId.id && typeof calendarId.id === "string") { // A child sub-calendar object
                            self.removeCalendar(calendarId.id);
                        } else {
                            self.removeCalendar(calendarId.toString());
                        }
                    });
                } else {
                    delete calendarCache[calendarId];
                }
            },

            removeAllCalendars: function() {
                calendarCache = [];
            },

            getCalendarData: function(calendarId) {
                if (!calendarCache[calendarId]) {
                    return null;
                }

                return calendarCache[calendarId].data;
            }
        };
    }

    return CalendarCache;
});