define("tc/manage-event",
    [
        "jquery",
        "tc/event-delete-dialog",
        "tc/calendar-util"
    ],
    function (
        $,
        deleteEventDialog,
        CalUtil
        )
    {
        "use strict";
        var ManageEvent = function(CalendarPlugin, cache) {
            var eventDisplayDialog;

            //drop or resize event
            function eventDropOrResizeCallback(calendarDiv, event, dayDelta, minuteDelta, allDay, revertFunc) {
                if (!event.end && !allDay)
                    event.end = new Date(event.start.getTime() + (1000 * 60 * 60 * 2));

                CalendarPlugin.setProcessingEvent(calendarDiv, true);
                CalendarPlugin.setSuppressEventToolTip(calendarDiv, true);
                ManageEvent(CalendarPlugin, cache).updateEvent(calendarDiv, (function() {
                    var eventSubCalendar = CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId),
                        data = {
                            "subCalendarId" : eventSubCalendar.parentId,
                            "uid" : event.id,
                            "eventType" : event.eventType,
                            "originalStartDate" : event.originalStart || "",
                            "what" : event.title,
                            "url" : event.workingUrl, //TEAMCAL-993
                            "dragAndDropUpdate" : true,
                            "start" : $.fullCalendar.formatDate(event.start, "ddMMyyyyHHmm"), // I should probably use "u", but FullCalendar sends the wrong time zone
                            "end" : $.fullCalendar.formatDate(event.end || event.start, "ddMMyyyyHHmm"),
                            "allDayEvent" : allDay,
                            "where" : event.where,
                            "description" : event.description,
                            "freq" : event.recur ? event.recur.freq : "",
                            "byday" : event.recur ? event.recur.byDay : "",
                            "interval" : event.recur ? event.recur.interval : "",
                            "until" : event.recur ? event.recur.localizedUntil || "" : "",
                            "recurrenceId" : event.recurId || "",
                            "editAllInRecurrenceSeries" : "false",
                            "customEventTypeId": event.customEventTypeId,
                            "childSubCalendarId": event.subCalendarId || ""
                        };

                    if (eventSubCalendar.eventInviteesSupported) {
                        data.person = $.map(event.invitees || [], function(invitee) {
                            return invitee.id;
                        });
                    }

                    return data;

                })(), function(XMLHttpRequest, textStatus, errorThrown) {
                    CalendarPlugin.showAjaxError(calendarDiv, XMLHttpRequest, textStatus, errorThrown, CalendarPlugin.ERROR_CLASS_EVENT_UPDATE);
                    revertFunc();
                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                    CalendarPlugin.setProcessingEvent(calendarDiv, false);
                }, function(responseEntity) {
                    if (responseEntity.success) {
                        CalendarPlugin.setGenericErrors(calendarDiv, null, CalendarPlugin.ERROR_CLASS_EVENT_UPDATE);
                        var updatedEvent = responseEntity.event;
                        // Due to subscriptions, an event may appear more than once a calendar. We need to update all instances of it.
                        $.each(CalendarPlugin.getCalendarPanel(calendarDiv).fullCalendar("clientEvents", event.id), function(eventIdx, event) {
                            event.localizedStartDate = updatedEvent.localizedStartDate;
                            event.localizedStartTime = updatedEvent.localizedStartTime;
                            event.localizedEndDate = updatedEvent.localizedEndDate;
                            event.localizedEndTime = updatedEvent.localizedEndTime;
                            event.recurId = updatedEvent.recurId;
                            event.invitees = updatedEvent.invitees;
                            event.iconUrl = updatedEvent.iconUrl;
                            event.confluenceFormattedStartDate = updatedEvent.confluenceFormattedStartDate;
                            event.expandDates = updatedEvent.expandDates;
                            //update jira product release event when drap drop
                            if (updatedEvent.extraProperties) {
                                event.extraProperties = updatedEvent.extraProperties;
                            }
                        });

                        if (event.calendarReloadRequiredOnUpdate) { // Only reload if required, as with JIRA calendar events with some old version of TC.
                            CalendarPlugin.reloadSubCalendar(calendarDiv, [event.subCalendarId]);
                        }
                    } else {
                        CalendarPlugin.setGenericErrors(calendarDiv, AJS.I18n.getText("calendar3.error.unknown"), CalendarPlugin.ERROR_CLASS_EVENT_UPDATE);
                        revertFunc(); // Won't happen, but just in case, we revert it.
                    }

                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                    CalendarPlugin.setProcessingEvent(calendarDiv, false);
                });
            }

            return {
                //get event detail dialog
                getEventDetailsDialog : function(calendarDiv, event, target, inlineDialogOptions) {
                    var eventSubCalendar = CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId),
                        parentSubCalendar = CalendarPlugin.getSubCalendar(calendarDiv, eventSubCalendar.parentId),
                        hideSubCalendarName = (function() {
                            return ($.inArray(CalendarPlugin.getCalendarPanel(calendarDiv).fullCalendar("getView").name, [ "month", "agendaWeek" ]) !== -1
                                ) || CalendarPlugin.getParameter(calendarDiv, "showSubCalendarNameInEventPopup") !== "true";
                        })(),
                        getPeriodEvent = function() {
                            var period = null;
                            var calendar = parentSubCalendar ? parentSubCalendar : eventSubCalendar;
                            if(CalUtil.isCustomEventType(event.eventType)) {
                                var customEventTypes = calendar.customEventTypes;
                                $.each(customEventTypes, function(index, customEventType) {
                                    if(customEventType.customEventTypeId === event.customEventTypeId && customEventType.periodInMins > 0) {
                                        period = AJS.format(AJS.I18n.getText('calendar3.reminder.event.click.info'), CalUtil.showInforReminder(customEventType.periodInMins));
                                        return false;
                                    }
                                });
                            } else {
                                var sanboxEventTypeReminders = calendar.sanboxEventTypeReminders;
                                if(sanboxEventTypeReminders && sanboxEventTypeReminders.length > 0) {
                                    var eventType = event.eventType;
                                    if(CalUtil.isJiraType(eventType)) {
                                        eventType += "-calendar";
                                    }
                                    $.each(sanboxEventTypeReminders, function(index, sanboxEventTypeReminder){
                                       if(sanboxEventTypeReminder.eventTypeId === eventType && sanboxEventTypeReminder.periodInMins > 0) {
                                           period = AJS.format(AJS.I18n.getText('calendar3.reminder.event.click.info'), CalUtil.showInforReminder(sanboxEventTypeReminder.periodInMins));
                                           return false;
                                       }
                                    });
                                }
                            }
                            return period;
                        },
                        renderEventPopupContent = function() {
                            return $("<div/>").html(
                                Confluence.TeamCalendars.Templates.eventPopup({
                                    "contextClass" : hideSubCalendarName ? "mycalendars" : "",
                                    "baseUrl" : CalendarPlugin.getParameter(calendarDiv, "baseUrl"),
                                    "event" : $.extend(
                                        $.extend({}, event),
                                        {
                                            "colour" : eventSubCalendar.color,
                                            "subCalendarDisplayName" : parentSubCalendar && !Confluence.TeamCalendars.isSubscriptionSubCalendar(eventSubCalendar) ? parentSubCalendar.name : eventSubCalendar.name,
                                            "subCalendarId" : parentSubCalendar ? parentSubCalendar.id : eventSubCalendar.id,
                                            "inviteesSupported" : eventSubCalendar.inviteesSupported,
                                            "period": getPeriodEvent()
                                        }
                                    )
                                })
                            );
                        },

                        eventDisplayDialog = AJS.InlineDialog(target, CalendarPlugin.dialogSequence++, function(contents, trigger, showPopup) {
                            var eventDetailsContainer = renderEventPopupContent();
                            $(".field-duration", eventDetailsContainer).each(function() {
                                var durationField = $(this);
                                if (event.localizedStartDate !== event.localizedEndDate) {
                                    //TEAMCAL-1711: replace "formating..." by spinner
                                    durationField.empty().html("<span class='aui-icon aui-icon-wait'></span>");

                                    // Multiday event. We use _start and _end because fullcalendar seems to change the start date for
                                    // it's own purposes.
                                    CalendarPlugin.formatDate(calendarDiv, [ event._start, event._end ], "dd MMM", function(formattedDates) {
                                        if (typeof formattedDates === "string") {
                                            durationField.text(formattedDates);
                                            return;
                                        }

                                        if (event.allDayFlipped ? !event.allDay : event.allDay) {
                                            durationField.empty().text(AJS.format("{0} - {1}", formattedDates[0], formattedDates[1]));
                                        } else {
                                            durationField.empty().text(AJS.format("{0}, {1} - {2}, {3}", formattedDates[0], event.localizedStartTime, event.localizedEndTime, formattedDates[1]));
                                        }
                                    });
                                } else {
                                    // Single day event
                                    if (!((event.allDayFlipped ? !event.allDay : event.allDay))) {
                                        durationField.empty().text(event.localizedStartTime + " - " + event.localizedEndTime);
                                    }
                                }
                            });

                            var editControls = $(".edit-controls", eventDetailsContainer);
                            if (CalendarPlugin.isEventEditable(calendarDiv, event)) {
                                editControls.each(function() {
                                    if (CalendarPlugin.getParameter(calendarDiv, "redirectEditInEventPopup") === "true") {
                                        $(".event-edit", this).unbind("click").attr(
                                            "href",
                                            CalendarPlugin.getParameter(calendarDiv, "baseUrl").concat(
                                                "/calendar/mycalendar.action",
                                                "?startDayOfMonth=",
                                                $.fullCalendar.formatDate(event.start, "d"),
                                                "&startMonthOfYear=",
                                                (parseInt($.fullCalendar.formatDate(event.start, "M")) - 1),
                                                "&startYear=",
                                                $.fullCalendar.formatDate(event.start, "yyyy"),
                                                "&editSubCalendarId=",
                                                encodeURIComponent(CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId).id),
                                                "&editEventId=",
                                                encodeURIComponent(event.id),
                                                "&editView=month")
                                        );
                                    } else {
                                        $(".event-edit", this).unbind("click").click(function() {
                                            eventDisplayDialog.hide();
                                            var eventSubCalendar = CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId),
                                                dialog = Confluence.TeamCalendars.Dialogs.getEditEventDialog(
                                                    $.extend(
                                                        { },
                                                        event,
                                                        {
                                                            "eventType" : eventSubCalendar.subscriptionType || eventSubCalendar.type,
                                                            "subCalendarId" : eventSubCalendar.parentId,
                                                            "originalEventSubCalendarId" : eventSubCalendar.id,
                                                            "childSubCalendarId": event.subCalendarId || ""
                                                        }
                                                    ),
                                                    null,
                                                    CalendarPlugin,
                                                    calendarDiv);

                                            dialog.show();
                                            return false;
                                        });
                                    }

                                    var deleteControl = $(".event-delete", this);
                                    if (CalendarPlugin.isEventDeletable(calendarDiv, event) && CalendarPlugin.getParameter(calendarDiv, "hideDeleteInEventPopup") === "false") {
                                        deleteControl.click(function() {
                                            eventDisplayDialog.hide();
                                            deleteEventDialog(calendarDiv, event, CalendarPlugin).show();
                                            return false;
                                        });
                                    } else {
                                        deleteControl.remove();
                                    }
                                });
                            } else {
                                editControls.each(function() {
                                    $(".edit-control", this).remove();
                                });
                            }

                            // Clean up leading/trailing separators
                            editControls.each(function() {
                                $(".event-hyperlink-separator:first-child, .event-hyperlink-separator:last-child", this).remove();
                            });

                            eventDetailsContainer.appendTo(contents.empty());
                            showPopup();
                        }, (function() {
                            var defaultOptions = {
                                hideDelay: 0,
                                noBind : true,
                                isRelativeToMouse : true,
                                showDelay : 0,
                                width : Confluence.TeamCalendars.isPDLEnabled() && $.inArray("jira-version", event.className) !== -1 ? 380 : 360,
                                hideCallback : function() {
                                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                                    if (AJS.InlineDialog.current && AJS.InlineDialog.current.popup) {
                                        // hide current inline dialog if have
                                        AJS.InlineDialog.current.popup.hide();
                                    }
                                },
                                initCallback : function() {
                                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, true);

                                    $(".event-field.reminder-icon").tooltip({gravity: 's'});
                                }
                            };

                            return inlineDialogOptions ? $.extend(defaultOptions, inlineDialogOptions) : defaultOptions;
                        })());

                    eventDisplayDialog.addClass("event-display-dialog");
                    return eventDisplayDialog;
                },

                //call rest for create/update event
                updateEvent : function(calendarDiv, data, errorCallback, successCallback) {
                    data.userTimeZoneId = Confluence.TeamCalendars.getUserTimeZone();

                    var spinnerDefer = CalendarPlugin.setSubCalendarSpinnerIconVisible(true);
                    /*
                     * Data format
                     * 1) originalSubCalendarName - The name of the sub-calendar that contains the event to be moved. Leave blank if the update does not move the event to a different sub-calendar
                     * 2) subCalendarName - The name of the event's new containing sub-calendar
                     * 3) uid - The event's UID
                     * 4) originalStartDate - The originalStart property event object
                     * 5) what - New summary
                     * 6) dragAndDropUpdate - Flag indicating if the update is done via dragging and/or dropping
                     * 7) start - The new start date. Ignored if dragAndDropUpdate is false.
                     * 8) end - The new end date. Ignored if dragAndDropUpdate is false.
                     * 9) startDate - The new start date. Ignored if dragAndDropUpdate is true.
                     * 10) startTime - The new start time. Ignored if dragAndDropUpdate is true.
                     * 11) endDate - The new end date. Ignored if either dragAndDropUpdate or allDayEvent is true.
                     * 12) endTime - The new end time. Ignored if either dragAndDropUpdate or allDayEvent is true.
                     * 13) allDayEvent - Flag indicating if the event should be an all day event. If this is false, startTime and endTime must be set.
                     * 14) where - New location (optional)
                     * 15) description - New description (optional)
                     * 16) url - New URL (optional)
                     * 17) freq - The event's new repeating frequency (e.g. daily, weekly). See iCalendar RFC. Optional
                     * 18) byday - The days in the frequency (e.g. mo, tu, we). See iCalendar RFC. Optional. Ignored if freq is not defined.
                     * 19) interval - The frequency interval (_2_ weeks once, _2_ days once, etc). See iCalendar RFC. Optional. Ignored if freq is not defined.
                     * 20) repeatEnds - The flag indicating if the recurring event would end. Optional. Ignored if freq is not defined.
                     * 21) until - The repeat end date. Optional. Ignored if freq is not defined.
                     * 22) recurrenceId - The ID of the rescheduled recurring event
                     * 23) editAllInRecurrenceSeries - Whether the edit would apply to all instances of a recurring event (ignored if event is not of a recurring set)
                     */
                    $.ajax({
                        cache : false,
                        data : data,
                        dataFilter : function(data) {
                            var responseEntity = $.parseJSON(data);

                            if (responseEntity.success) {
                                Confluence.TeamCalendars.mergeSubCalendarObjectsToArray([ responseEntity.subCalendar ]);
                            }

                            return responseEntity;
                        },
                        dataType : "text",
                        error : function(XMLHttpRequest, textStatus, errorThrown) {
                            if (errorCallback) {
                                errorCallback(XMLHttpRequest, textStatus, errorThrown);
                            }
                        },
                        success : function(responseEntity) {
                            var eventType = data.eventType,
                                calendarContext = CalendarPlugin.getParameter(calendarDiv, "calendarContext");
                            var property;
                            if(eventType === "custom") {
                                if(responseEntity.subCalendar && responseEntity.subCalendar.customEventTypes) {
                                    $.each(responseEntity.subCalendar.customEventTypes, function(index, item){
                                        if(item.customEventTypeId === data.customEventTypeId) {
                                            property = {
                                                customEventTypeName: item.title
                                            };
                                        }
                                    });
                                }
                            }
                            if (data.uid) {
                                // Fire analytics event for event update
                                Confluence.TeamCalendars.fireEventForAnalytics("event.update.default." +  eventType + ".context." + calendarContext, property);
                            } else {
                                // Fire analytics event for event creation
                                Confluence.TeamCalendars.fireEventForAnalytics("event.create.default." +  eventType + ".context." + calendarContext, property);
                            }

                            cache.removeCalendar(data.subCalendarId);
                            cache.removeCalendar(data.originalSubCalendarId);
                            if (successCallback) {
                                successCallback(responseEntity);
                            }
                            CalendarPlugin.reloadSubCalendarDisableEvent(calendarDiv, data);
                        },
                        complete: function(){
                            if(spinnerDefer) spinnerDefer.resolve();
                        },
                        type : "PUT",
                        timeout : CalendarPlugin.ajaxTimeout,
                        url : CalendarPlugin.getCalendarServiceBaseUrl(calendarDiv, "/events.json")
                    });
                },
                //call rest for delete event
                deleteEvent : function(calendarDiv, params, errorCallback, successCallback) {
                    $.ajax({
                        cache : false,
                        data : params,
                        dataType : "json",
                        error : function(XMLHttpRequest, textStatus, errorThrown) {
                            if (errorCallback) {
                                errorCallback(XMLHttpRequest, textStatus, errorThrown);
                            }
                        },
                        success : function(responseEntity) {
                            cache.removeCalendar(params.subCalendarId);
                            if (successCallback) {
                                successCallback(responseEntity);
                            }
                        },
                        type : "DELETE",
                        timeout: CalendarPlugin.ajaxTimeout,
                        url : CalendarPlugin.getCalendarServiceBaseUrl(calendarDiv, "/events.json")
                    });
                },

                //trigger event click in fullcalendar.js
                eventClick : function(event, jsEvent, calendarDiv) {
                    CalendarPlugin.setEventTooltipVisible(calendarDiv);

                    if (jsEvent.shiftKey) {
                        if (CalendarPlugin.isEventEditable(calendarDiv, event)) {
                            var eventSubCalendar = CalendarPlugin.getSubCalendar(calendarDiv, event.subCalendarId);
                            Confluence.TeamCalendars.Dialogs.getEditEventDialog(
                                $.extend(
                                    { },
                                    event,
                                    {
                                        "eventType" : eventSubCalendar.subscriptionType || eventSubCalendar.type,
                                        "subCalendarId" : eventSubCalendar.parentId,
                                        "originalEventSubCalendarId" : eventSubCalendar.id,
                                        "childSubCalendarId": event.subCalendarId || ""
                                    }
                               ),
                               null,
                               CalendarPlugin,
                               calendarDiv).show();
                        }
                        return false;
                    }

                    // Ctrl click in a Mac brings up the context menu - we need another key
                    // See TEAMCAL-993 for why this is called workingUrl
                    if ((jsEvent.ctrlKey || jsEvent.metaKey) && event.workingUrl) {
                        window.location = event.workingUrl;
                        return false;
                    }

                    CalendarPlugin.getEventDetailsDialog(calendarDiv, event, $(jsEvent.currentTarget)).show(jsEvent);
                    return false;
                },
                //trigger eventRender in fullcalendar.js
                eventRender : function(event, eventDiv, view, calendarDiv) {
                    if (event.disableResizing) {
                        $(".ui-resizable-handle", eventDiv).hide();
                    }

                    eventDiv.mousemove(function(jsEvent) {
                        eventDiv.data("mouseCoords", { pageX : jsEvent.pageX, pageY : jsEvent.pageY });
                    }).hover(
                            function() {
                                setTimeout(function() {
                                    var mouseCoords = eventDiv.data("mouseCoords");
                                    if (!CalendarPlugin.isEventToolTipSuppressed(calendarDiv) && mouseCoords) {
                                        CalendarPlugin.setEventTooltipVisible(calendarDiv, event, mouseCoords, true);
                                    }
                                }, 5000);
                            },
                            function() {
                                eventDiv.removeData("mouseCoords");
                                CalendarPlugin.setEventTooltipVisible(calendarDiv);
                                CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                            }
                    );
                    $(".fc-event-time, .fc-event-title", eventDiv).css("color", eventDiv.css("color"));

                    if ("agendaWeek" === view.name) {
                        eventDiv.parent().addClass(event.allDay ? "fc-agenda-head" : "fc-agenda-body");
                        if (!event.allDay) {
                            $(".fc-event-inner", eventDiv).css({
                                "border-right-color": event.secondaryBorderColor || event.borderColor,
                                "border-bottom-color": event.secondaryBorderColor || event.borderColor,
                                "border-left-color": event.secondaryBorderColor || event.borderColor
                            });
                        }
                    }

                    if ("month" === view.name || "agendaWeek" === view.name) {
                        var renderedEventTitle;


                        if ($.inArray("other", event.className) !== -1 || event.eventType === "custom") {
                            if (event.shortTitle) {
                                renderedEventTitle = event.shortTitle + ": " + event.title;
                            } else {
                                if (event.invitees && event.invitees.length) {
                                    renderedEventTitle = event.invitees[0].displayName + ": " + event.title;
                                } else {
                                    renderedEventTitle = event.title;
                                }
                            }
                        } else {
                            renderedEventTitle = event.shortTitle || event.title;
                        }

                        $(".fc-event-title", eventDiv).text(renderedEventTitle);

                        //add image event for printing
                        if("month" === view.name || ("agendaWeek" === view.name && event.allDay)) {
                            $(".fc-event-inner", eventDiv).before(Confluence.TeamCalendars.Templates.iconEventTypePrinting({"eventType" : event.className, "isCustom" : event.eventType === "custom" ? true : false}));
                        }
                    }
                },
                //trigger event drop in fullcalendar.js
                eventDrop : function(event, dayDelta, minuteDelta, allDay, revertFunc, calendarDiv) {
                    if (CalendarPlugin.isEventEditable(calendarDiv, event) && !CalendarPlugin.isProcessingEvent(calendarDiv)) {
                        eventDropOrResizeCallback(calendarDiv, event, dayDelta, minuteDelta, allDay, revertFunc);
                    } else {
                        revertFunc();
                    }
                },
                //trigger event resize in fullcalendar.js
                eventResize : function(event, dayDelta, minuteDelta, revertFunc, calendarDiv) {
                    if (CalendarPlugin.isEventEditable(calendarDiv, event) && !CalendarPlugin.isProcessingEvent(calendarDiv)) {
                        eventDropOrResizeCallback(calendarDiv, event, dayDelta, minuteDelta, event.allDay, revertFunc);
                    } else {
                        revertFunc();
                    }
                },
                //trigger event drap start in fullcalendar.js
                eventDragStart : function(calendarDiv) {
                        CalendarPlugin.setSuppressEventToolTip(calendarDiv, true);
                        CalendarPlugin.setEventTooltipVisible(calendarDiv);
                        CalendarPlugin.hideInlineAuiDialogs();
                },
                //trigger event drap end in fullcalendar.js
                eventDragEnd : function(calendarDiv) {
                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                },
                //trigger event resize start in fullcalendar.js
                eventResizeStart : function(calendarDiv) {
                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, true);
                    CalendarPlugin.setEventTooltipVisible(calendarDiv);
                    CalendarPlugin.hideInlineAuiDialogs();
                },
                //trigger event resize end in fullcalendar.js
                eventResizeStop : function(calendarDiv) {
                    CalendarPlugin.setSuppressEventToolTip(calendarDiv, false);
                }
            };
        };
        return ManageEvent;
    }
);
