/**
 * This code creates the FieldHandler objects. Each FieldHandler object must implement the following methods:
 *   add() - This method adds a field to the fieldsDiv and sets up any Javascript needed to make the field work.
 *   init() - This initializes the field to either its default value, or a value taken from the event object.
 *   getData() - This returns the data contained in the fields as key-value pairs.
 *
 * Each field handler expects the field object to have different properties defined.
 *
 * See also:
 *   event-types.js - This is currently where the mappings of fields to event types lives.
 *   event-edit-dialog.js - The code that uses these FieldHandlers.
 *   event-fields.soy - The soy templates for the fields.
 */
define("tc/event-field-handlers",
    [
        "jquery",
        "tc/jira-event-field-process",
        "tc/calendar-util"
    ],
    function (
        $,
        JiraEventFieldProcess,
        CalUtil
    )
    {
        "use strict";
        $.fn.tcDatePicker = $.fn.datepicker.noConflict();
        var tcDatePicker = $.datepicker.noConflict();

        function createDatePicker(datePickerElement, options){
            var $datePickerElement = datePickerElement instanceof $ ? datePickerElement : $(datePickerElement);
            $datePickerElement.tcDatePicker(options);
        }

        function createLabelInterval(selectedFreq, interval, intervalSpan) {
            if (/^[1-9]\d*$/.test(interval) || interval === "0") {
                if (selectedFreq === "DAILY") {
                    intervalSpan.text(
                        parseInt(interval) > 1 ? AJS.I18n.getText("calendar3.repeat.interval.days") : AJS.I18n.getText("calendar3.repeat.interval.day"));
                } else if (selectedFreq === "WEEKLY") {
                    intervalSpan.text(
                        parseInt(interval) > 1 ? AJS.I18n.getText("calendar3.repeat.interval.weeks") : AJS.I18n.getText("calendar3.repeat.interval.week"));
                } else if (selectedFreq === "MONTHLY") {
                    intervalSpan.text(
                        parseInt(interval) > 1 ? AJS.I18n.getText("calendar3.repeat.interval.months") : AJS.I18n.getText("calendar3.repeat.interval.month"));
                } else if (selectedFreq === "YEARLY") {
                    intervalSpan.text(
                        parseInt(interval) > 1 ? AJS.I18n.getText("calendar3.repeat.interval.years") : AJS.I18n.getText("calendar3.repeat.interval.year"));
                }
            } else {
                intervalSpan.text("");
            }
        }

        var eventFieldHandlers = {

            /**
             * A simple text field. Expects the field object to have the following properties:
             *   id - The id of the field.
             *   label - A label for the field.
             *   required - Is the field required?
             *   property - The property on the event object that the field value is taken from.
             *
             *   TODO: make id and property the same. Need to fix in event object.
             */
            text : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.text({
                                "id" : field.id,
                                "label" : field.label,
                                "required" : field.required,
                                "readOnly" : field.readOnly
                            })
                        );
                    },
                    init : function () {
                        $("#" + field.id + "-text", fieldsDiv).val(event[field.property] || "").keydown(function() {
                            // This is to set a flag into the input field indicating that the value has been changed by the user.
                            // The use of this is by JIRA event types to not change the event series name after the user
                            // has typed in something for the name.
                            var nameInputField = $(this),
                                name = nameInputField.val();

                            if (!nameInputField.data("valueChangedByUser")) {
                                setTimeout(function() {
                                    if (name !== nameInputField.val()) {
                                        nameInputField.data("valueChangedByUser", true);
                                    }
                                }, 300);
                            }
                        });
                    },
                    setData: function(valueDefault) {
                        eventEditDialog.getField(fieldsDiv, field.id).val(valueDefault);
                    },
                    getData : function() {
                        var data = {};

                        data[field.id] = eventEditDialog.getField(fieldsDiv, field.id).val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-text-" + field.id;
                    }
                };
            },

            /**
             * A textarea field. Expects the field object to have the following properties:
             *   id - The id of the field.
             *   size - The size of the textarea in lines.
             *   label - A label for the field.
             *   required - Is the field required?
             *   property - The property on the event object that the field value is taken from.
             *
             *   TODO: make id and property the same. Need to fix in event object.
             */
            textarea : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.textarea({
                                "id" :  field.id,
                                "size" : field.size,
                                "required" : field.required,
                                "label" : field.label,
                                "readOnly" : field.readOnly
                            })
                        );
                    },
                    init : function() {
                        $("#" + field.id + "-textarea", fieldsDiv).val(event[field.property] || "");
                    },
                    getData : function() {
                        var data = {};
                        data[field.id] = $("textarea[name=" + field.id + "]", fieldsDiv).val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-textarea-" + field.id;
                    }
                };
            },

            /**
             * A Page field, can be used to enter a url or confluence page. Includes confluence page drop down.
             * Expects the field object to have the following properties:
             *   id - The id of the field.
             *   label - A label for the field.
             *   required - Is the field required?
             *   property - The property on the event object that the field value is taken from.
             *
             *   TODO: make id and property the same. Need to fix in event object.
             */
            page : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(
                            Confluence.TeamCalendars.Templates.Fields.page({
                                "id" : field.id,
                                "label" : field.label,
                                "required" : field.required,
                                "readOnly" : field.readOnly
                            })
                        );
                    },
                    init : function () {
                        $("#" + field.id + "-text", fieldsDiv).val(event[field.property] || "");

                        CalendarPlugin.makeAutoCompleteSearch(
                                calendarDiv,
                                $("input[name='" + field.id + "']", fieldsDiv),
                                { type: [ "page", "blogpost" ] },
                                $("." + field.id + "-suggestions", fieldsDiv),
                                function(searchField, suggestionLink) {
                                    searchField.val(suggestionLink.attr("href"));
                                }
                        );

                    },
                    getData : function() {
                        var data = {},
                            theField = eventEditDialog.getField(fieldsDiv, field.id);

                        data[field.id] = theField.hasClass("with-hint") ? "" : theField.val();
                        return data;
                    },
                    getElementId : function() {
                        return "#field-page-" + field.id;
                    }
                };
            },

            /**
             * Adds a 'when' field to the form. Consists of a start and end date and time pickers, and an all-day checkbox.
             * Expects the field object to have the following properties:
             *   id - id of the field.
             *
             */
            when : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        var startTimeField, endTimeField, allDayEventCheckbox, timeSuggestions;

                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.when({
                            "id" : field.id,
                            "disableStartDateTime" : field.disableStartDateTime,
                            "disableEndDateTime" : field.disableEndDateTime,
                            "disableAllDay" : field.disableAllDay
                        }));

                        startTimeField = eventEditDialog.getField(fieldsDiv, "startTime");
                        endTimeField = eventEditDialog.getField(fieldsDiv, "endTime");
                        allDayEventCheckbox = eventEditDialog.getField(fieldsDiv, "allDayEvent");

                        //Setup autocomplete for time fields
                        timeSuggestions = CalendarPlugin.getParameter(calendarDiv, "timeSuggestion");
                        startTimeField.autocompleteTC({ source : timeSuggestions, minLength : 2 });
                        endTimeField.autocompleteTC({ source : timeSuggestions, minLength : 2 });

                        //Set up All Day Event checkbox
                        allDayEventCheckbox.click(function() {
                            if (this.checked) {
                                startTimeField.addClass("hidden");
                                endTimeField.addClass("hidden");
                            } else {
                                startTimeField.removeClass("hidden");
                                endTimeField.removeClass("hidden");
                                if (!startTimeField.val()) {
                                    startTimeField.val(CalendarPlugin.getDefaultStartTime(calendarDiv));
                                }
                                if (!endTimeField.val()) {
                                    endTimeField.val(CalendarPlugin.getDefaultEndTime(calendarDiv));
                                }
                            }
                        });
                    },
                    init : function() {
                        $(".start, .end", fieldsDiv).each(function() {
                            var datePicker = $(this);
                            createDatePicker(datePicker, {
                                "dateFormat" : "ddmmyy",
                                "dayNames" : CalendarPlugin.getDayNames(),
                                "dayNamesMin": CalendarPlugin.getDayNamesShort(),
                                "dayNamesShort" : CalendarPlugin.getDayNamesShort(),
                                "firstDay" : CalendarPlugin.getParameter(calendarDiv, "firstDayOfWeek"),
                                "monthNames" : CalendarPlugin.getMonthNames(),
                                "monthNamesShort" : CalendarPlugin.getMonthNamesShort(),
                                "beforeShow" : function() {
                                    $("#ui-datepicker-div").addClass("tc-datepicker");
                                },
                                "onClose" : function() {
                                    $("#ui-datepicker-div").removeClass("tc-datepicker");
                                },
                                "onSelect": function(dateText) {
                                    var selectedDate = tcDatePicker.parseDate("ddmmyy", dateText);

                                    if (datePicker.hasClass("start")) {
                                        var endDatePicker = $(".end", fieldsDiv),
                                            oldStartDate = new Date(parseInt(datePicker.data("date"))),
                                            newEndDate = new Date(endDatePicker.data("date") - (oldStartDate.getTime() - selectedDate.getTime()));

                                        endDatePicker.tcDatePicker("option", "defaultDate", newEndDate);
                                        endDatePicker.tcDatePicker("setDate", newEndDate);

                                        endDatePicker.data("date", newEndDate.getTime()).val(function() {
                                            CalendarPlugin.formatDate(calendarDiv, newEndDate, "date", function(formattedDate) {
                                                endDatePicker.val(formattedDate);
                                            });
                                            return AJS.I18n.getText("calendar3.formattingdate");
                                        });

                                    }

                                    datePicker.tcDatePicker("option", "defaultDate", selectedDate);
                                    datePicker.tcDatePicker("setDate", selectedDate);

                                    datePicker.data("date", selectedDate.getTime()).val(function() {
                                        CalendarPlugin.formatDate(calendarDiv, selectedDate, "date", function(formattedDate) {
                                            datePicker.val(formattedDate);
                                        });
                                        return AJS.I18n.getText("calendar3.formattingdate");
                                    });
                                },
                                "showAnim" : ""
                            });
                        });

                        var startTimeField = eventEditDialog.getField(fieldsDiv, "startTime"),
                            endTimeField = eventEditDialog.getField(fieldsDiv, "endTime"),
                            allDayEventField = eventEditDialog.getField(fieldsDiv, "allDayEvent"),

                            setDateField = function(name, date, localisedDate) {
                                var dateField = $("input[name=" + name + "]", fieldsDiv);

                                dateField.tcDatePicker("option", "defaultDate", date);
                                dateField.tcDatePicker("setDate", date);

                                dateField.data("date", date.getTime()).val(localisedDate || function() {
                                    CalendarPlugin.formatDate(calendarDiv, date, "date", function(value) {
                                        dateField.val(value);
                                    });
                                    return AJS.I18n.getText("calendar3.formattingdate");
                                });
                            };

                        event.start && setDateField("startDate", event.start, event.localizedStartDate);
                        setDateField("endDate", event.end || event.start, event.localizedEndDate);

                        if (event.allDay) {
                            startTimeField.val("").addClass("hidden");
                            endTimeField.val("").addClass("hidden");
                            allDayEventField.attr("checked", "checked");
                        } else {
                            startTimeField.val(event.originalStartTime || event.localizedStartTime || function() {
                                        CalendarPlugin.formatDate(calendarDiv, event.start, "time", function(value) {
                                            startTimeField.val(value);
                                        });
                                        return AJS.I18n.getText("calendar3.formattingdate");
                                    }
                            ).removeClass("hidden");
                            endTimeField.val(event.originalEndTime || event.localizedEndTime || function() {
                                        CalendarPlugin.formatDate(calendarDiv, event.end || new Date(event.start.getTime() + 3600000), "time", function(value) {
                                            endTimeField.val(value);
                                        });
                                        return AJS.I18n.getText("calendar3.formattingdate");
                                    }
                            ).removeClass("hidden");

                            allDayEventField.removeAttr("checked", "checked");
                        }
                    },
                    getData : function() {
                        return {
                            startDate : eventEditDialog.getField(fieldsDiv, "startDate").val(),
                            startTime : eventEditDialog.getField(fieldsDiv, "startTime").val(),
                            endDate : eventEditDialog.getField(fieldsDiv, "endDate").val(),
                            endTime : eventEditDialog.getField(fieldsDiv, "endTime").val(),
                            allDayEvent : eventEditDialog.getField(fieldsDiv, "allDayEvent").is(":checked")
                        };
                    },
                    getElementId : function() {
                        return "#field-when";
                    }
                };
            },

            /**
             * Adds a 'repeats' field to the dialog. Contains inputs and logic for repeating events.
             * Expects the field object to have the following properties:
             *   id - id of the field.
             */
            repeats : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        var subCalendarSelect,
                            freqSelect,
                            intervalInput,
                            recurEndsContainer,
                            recurEndNever,
                            recurEndUntil,
                            untilInput;

                        fieldsDiv.append(
                                Confluence.TeamCalendars.Templates.Fields.repeats({id : field.id})
                        );

                        subCalendarSelect = $("select[name='calendar']", fieldsDiv);
                        freqSelect = $("select[name='freq-select']", fieldsDiv);
                        intervalInput = $("input[name='interval']", fieldsDiv);
                        recurEndsContainer = $(".recur-ends", fieldsDiv);
                        recurEndNever = $(".recur-ends-never", recurEndsContainer);
                        recurEndUntil = $(".recur-ends-until", recurEndsContainer);
                        untilInput = $("input[name='until']", recurEndsContainer);

                        var setDateField = function(name, date, localisedDate) {
                            var dateField = $("input[name=" + name + "]", fieldsDiv);

                            dateField.tcDatePicker("option", "defaultDate", date);
                            dateField.tcDatePicker("setDate", date);

                            dateField.data("date", date.getTime()).val(localisedDate);
                        };

                        $("input[name='editthisinstanceonly']", fieldsDiv).change(function() {
                            if ($(this).is(":checked")) {
                                eventEditDialog.disableCalSelect();
                                eventEditDialog.showToolTip();
                                subCalendarSelect.attr("disabled", "disabled");
                                freqSelect.attr("disabled", "disabled");
                                intervalInput.attr("readonly", "readonly");
                                recurEndNever.attr("disabled", "disabled");
                                recurEndUntil.attr("disabled", "disabled");
                                untilInput.attr("readonly", "readonly").tcDatePicker("disable");
                                setDateField("startDate", event.start, event.localizedStartDate);
                                setDateField("endDate", event.end || event.start, event.localizedEndDate);
                            } else {
                                eventEditDialog.enableCalSelect();
                                eventEditDialog.hiddenToolTip();
                                freqSelect.removeAttr("disabled");
                                intervalInput.removeAttr("readonly");
                                if ($("option:selected", freqSelect).val()) {
                                    recurEndNever.removeAttr("disabled");
                                    recurEndUntil.removeAttr("disabled");
                                    if (recurEndUntil.is(":checked")) {
                                        untilInput.removeAttr("readonly").tcDatePicker("enable");
                                    } else {
                                        untilInput.tcDatePicker("disable");
                                    }
                                }

                                var originalStartDateTime = tcDatePicker.parseDate("yy-mm-dd", event.originalStartDateTime.split("T")[0]);
                                var originalEndDateTime = tcDatePicker.parseDate("yy-mm-dd", event.originalEndDateTime.split("T")[0]);
                                setDateField("startDate", originalStartDateTime, event.localizedOriginalStartDate);
                                setDateField("endDate", originalEndDateTime, event.localizedOriginalEndDate);
                            }
                        });

                        freqSelect.change(function() {
                            var selectedFreq = $("option:selected", this).val();
                            var freqInput = $("input[name='freq']", fieldsDiv);
                            var byDayInput = $("input[name='byday']", fieldsDiv);
                            var intervalSelectContainer = $(".interval-container", fieldsDiv);

                            if (selectedFreq) {
                                if (selectedFreq === "WEEKLY-BYDAY") {
                                    intervalInput.attr("readonly", "readonly").val("1");
                                    intervalSelectContainer.addClass("hidden");
                                    freqInput.val("WEEKLY");
                                    byDayInput.val("MO,TU,WE,TH,FR");
                                } else {
                                    intervalInput.removeAttr("readonly").val("1");
                                    intervalSelectContainer.removeClass("hidden");
                                    freqInput.val(selectedFreq);
                                    byDayInput.val("");
                                }

                                recurEndsContainer.removeClass("hidden");
                                recurEndNever.removeAttr("disabled");
                                if (recurEndUntil.removeAttr("disabled").is(":checked")) {
                                    untilInput.removeAttr("disabled").removeAttr("readonly");
                                }
                            } else {
                                intervalInput.val("1");
                                intervalSelectContainer.addClass("hidden");
                                recurEndsContainer.addClass("hidden");
                                recurEndNever.attr("disabled", "disabled");
                                recurEndUntil.attr("disabled", "disabled");
                                untilInput.val("").attr("disabled", "disabled").attr("readonly", "readonly");
                                freqInput.val("");
                                byDayInput.val("");
                            }

                            intervalInput.trigger("keydown");
                        });

                        intervalInput.keydown(function() {
                            setTimeout(function() {
                                var selectedFreq = $("option:selected", freqSelect).val();
                                var intervalSpan = $(".interval-label", fieldsDiv);
                                var interval = $.trim(intervalInput.val());
                                createLabelInterval(selectedFreq, interval, intervalSpan);
                            }, 300);
                        });

                        $(".recur-ends-never", fieldsDiv).click(function() {
                            untilInput.val("").attr("disabled", "disabled").attr("readonly", "readonly").tcDatePicker("disable");
                        });
                        $(".recur-ends-until", fieldsDiv).click(function() {
                            untilInput.removeAttr("disabled", "disabled").removeAttr("readonly").tcDatePicker("enable");
                        });
                    },
                    init : function() {
                        var $untilField = $(".recur-ends .datepicker-field", fieldsDiv);
                        createDatePicker($untilField, {
                            "dateFormat" : "ddmmyy",
                            "dayNames" : CalendarPlugin.getDayNames(),
                            "dayNamesMin": CalendarPlugin.getDayNamesShort(),
                            "dayNamesShort" : CalendarPlugin.getDayNamesShort(),
                            "firstDay" : CalendarPlugin.getParameter(calendarDiv, "firstDayOfWeek"),
                            "monthNames" : CalendarPlugin.getMonthNames(),
                            "monthNamesShort" : CalendarPlugin.getMonthNamesShort(),
                            "beforeShow" : function() {
                                $("#ui-datepicker-div").addClass("tc-datepicker");
                            },
                            "onClose" : function() {
                                $("#ui-datepicker-div").removeClass("tc-datepicker");
                            },
                            "onSelect": function(dateText) {
                                var datePicker = $(this),
                                    selectedDate = tcDatePicker.parseDate("ddmmyy", dateText);

                                datePicker.tcDatePicker("option", "defaultDate", selectedDate);
                                datePicker.tcDatePicker("setDate", selectedDate);

                                datePicker.data("date", selectedDate.getTime()).val(function() {
                                    CalendarPlugin.formatDate(calendarDiv, selectedDate, "date", function(formattedDate) {
                                        datePicker.val(formattedDate);
                                    });

                                    return AJS.I18n.getText("calendar3.formattingdate");
                                });
                            },
                            "showAnim" : ""
                        });

                        var isNewEvent = function(event){
                            return !event.id;
                        };

                        var populateDateRangeFields = function() {
                            eventEditDialog.getField(fieldsDiv, "recurrenceId").val(event.recurId || ""); //todo - hidden field
                            var freqInput = eventEditDialog.getField(fieldsDiv, "freq").val("");
                            var byDayInput = eventEditDialog.getField(fieldsDiv, "byday").val("");
                            var intervalInput = eventEditDialog.getField(fieldsDiv, "interval").val("1").attr("readonly", "readonly");
                            var intervalContainer = intervalInput.closest(".interval-container");
                            var frequencySelectField = eventEditDialog.getSelect(fieldsDiv, "freq-select");

                            $("option", frequencySelectField.attr("disabled", "disabled")).removeAttr("selected");

                            var editThisInstanceOnly = eventEditDialog.getField(fieldsDiv, "editthisinstanceonly");
                            var editThisInstanceOnlyContainer = editThisInstanceOnly.closest(".edit-this-instance-only-container");
                            var recurEndsContainer = $(".recur-ends", fieldsDiv);
                            var recurEndRadioButtons = eventEditDialog.getField(fieldsDiv, "recur-end").removeAttr("checked").attr("disabled", "disabled");
                            var recurUntilInput = eventEditDialog.getField(fieldsDiv, "until").attr("disabled", "disabled").attr("readonly", "readonly").val("");

                            recurUntilInput.tcDatePicker("disable");

                            if (!isNewEvent(event) && event.recur) {
                                // populate data for existed event

                                eventEditDialog.disableCalSelect();
                                eventEditDialog.showToolTip();
                                editThisInstanceOnlyContainer.removeClass("hidden");
                                editThisInstanceOnly.attr("checked", "checked");

                                var freq = event.recur.freq;
                                var byDay = event.recur.byDay;
                                var interval = event.recur.interval;

                                recurEndsContainer.removeClass("hidden");
                                freqInput.val(freq || "");
                                byDayInput.val(byDay || "");

                                if (byDay) {
                                    $("option[value='WEEKLY-BYDAY']", frequencySelectField).attr("selected", "selected");
                                    intervalContainer.addClass("hidden");
                                } else {
                                    $("option[value='" + freq + "']", frequencySelectField).attr("selected", "selected");
                                    intervalContainer.removeClass("hidden");
                                    intervalInput.val(interval || "1");
                                }

                                //append label for field
                                var intervalSpan = $(".interval-label", fieldsDiv);
                                createLabelInterval(freq, interval, intervalSpan);

                                var repeatUntil = event.recur.until;
                                if (repeatUntil) {
                                    $(recurEndRadioButtons[1]).attr("checked", "checked");
                                    recurUntilInput.removeAttr("disabled");

                                    var repeatUntilDate = $.fullCalendar.parseDate(repeatUntil);
                                    recurUntilInput.tcDatePicker("setDate", repeatUntilDate);
                                    recurUntilInput.tcDatePicker("option", "defaultDate", repeatUntilDate);
                                    recurUntilInput.val(event.recur.localizedUntil || function() {
                                        CalendarPlugin.formatDate(calendarDiv, repeatUntilDate, "date", function(value) {
                                            recurUntilInput.val(value);
                                        });
                                        return AJS.I18n.getText("calendar3.formattingdate");
                                    });
                                } else {
                                    $(recurEndRadioButtons[0]).attr("checked", "checked");
                                }
                            } else if (isNewEvent(event) && event.recur) {
                                // populate default data for event
                                frequencySelectField.removeAttr("disabled");
                                frequencySelectField.val(event.recur.freq).change();
                                $(recurEndRadioButtons[0]).attr("checked", "checked");
                            } else {
                                eventEditDialog.enableCalSelect();
                                eventEditDialog.hiddenToolTip();
                                editThisInstanceOnlyContainer.addClass("hidden");
                                $("option:first", frequencySelectField.removeAttr("disabled")).attr("selected", "selected");
                                intervalContainer.addClass("hidden");
                                recurEndsContainer.addClass("hidden");
                                $(recurEndRadioButtons[0]).attr("checked", "checked");
                            }
                        };

                        populateDateRangeFields();
                    },
                    getData : function() {
                        var until = eventEditDialog.getField(fieldsDiv, "until");
                        return {
                            "freq" : eventEditDialog.getField(fieldsDiv, "freq").val(),
                            "byday" : eventEditDialog.getField(fieldsDiv, "byday").val(),
                            "interval" : eventEditDialog.getField(fieldsDiv, "interval").val() || "",
                            "until" : until.val(),
                            "repeatEnds" : !(until.attr("readonly") || until.attr("disabled")),
                            "recurrenceId" : eventEditDialog.getField(fieldsDiv, "recurrenceId").val(),
                            "editAllInRecurrenceSeries" : !eventEditDialog.getField(fieldsDiv, "editthisinstanceonly").is(":checked")
                        };
                    },
                    getElementId : function() {
                        return "#field-repeats";
                    }
                };
            },

            /**
             * Allows selection of multiple users.
             */
            user : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function() {
                        fieldsDiv.append (
                            Confluence.TeamCalendars.Templates.Fields.user({
                                id :  field.id,
                                label : field.label,
                                required : field.required
                            })
                        );
                        //This makes the user dropdown component work.
                        Confluence.runBinderComponents();
                    },
                    init : function() {
                        var inviteeList = $(".invitees", fieldsDiv),

                            deleteInviteeHandler = function() {
                                $(this).closest("li").remove();

                                if (!$("li", inviteeList).length) {
                                    inviteeList.addClass("hidden");
                                }
                            },

                            ellipsisInviteeNames = function() {
                                $(".invitee-left", inviteeList).filter(function() {
                                    return !$(this).attr("threedots");
                                }).ThreeDots({
                                    max_rows : 1,
                                    text_span_class: "invitee-name-display",
                                    whole_word: false
                                });
                            };

                        $("#" + field.id + "-user-picker", fieldsDiv).bind("selected.autocomplete-user", function(e, selection) {
                            var selectedUserId = selection.content.userKey || selection.content.username,
                                userSelection,
                                userAlreadyAdded = $(".invitee", inviteeList).filter(function() {
                                    return $(this).data("entity").id === selectedUserId;
                                }).length > 0;

                            if (!userAlreadyAdded) {
                                if ($("li", inviteeList).length === 3) {
                                    inviteeList.addClass("boxed");
                                }

                                userSelection = $(Confluence.TeamCalendars.Templates.Fields.userSelection({
                                    "fullName" : selection.content.title,
                                    "userName" : selection.content.username,
                                    "imgPath" : selection.content.thumbnailLink.href
                                }));

                                userSelection.data("entity", {
                                    "id" : selectedUserId,
                                    "displayName" : selection.content.title
                                }).find(".delete-invitee").click(deleteInviteeHandler);

                                inviteeList.prepend(userSelection).removeClass("hidden");

                                ellipsisInviteeNames();
                            }

                            // focus() to workaround https://studio.plugins.atlassian.com/browse/TEAMCAL-229
                            $(this).val("").focus();

                            var whoErrorDiv = $(".who-error", fieldsDiv);
                            if (whoErrorDiv.children().length) {
                                // Only empty and resize dialog if the error div has elements in it to fix the
                                // problem where the dialog footer is longer than it should be in Chrome 12 + OS X Lion
                                whoErrorDiv.empty();
                            }
                        });

                        if (event.invitees && event.invitees.length) {
                            $.each(event.invitees, function(inviteeIdx, invitee) {
                                inviteeList.append(
                                    $(Confluence.TeamCalendars.Templates.Fields.userSelection({
                                        "fullName" : invitee.displayName,
                                        "userName" : invitee.name,
                                        "imgPath" : invitee.avatarIconUrl
                                    })).data("entity", {
                                            "id" : invitee.id,
                                            "displayName" : invitee.displayName
                                    })
                                );
                            });

                            $(".delete-invitee", inviteeList.removeClass("hidden")).unbind("click").click(deleteInviteeHandler);
                            ellipsisInviteeNames();
                        }
                    },
                    getData : function() {
                        var data = {};
                        var invitees = $(".invitee", fieldsDiv);

                        if (invitees.length) {
                            data.person = $.map(invitees, function(invitee) {
                                return $(invitee).data("entity").id;
                            });
                        }

                        if (field.setTitle) {
                            data.what = invitees.length ? $.map(invitees, function(invitee) {
                                return $(invitee).data("entity").displayName;
                            }).join(", ") : "";
                        }

                        return data;
                    },
                    getElementId : function() {
                        return "#field-user-" + field.id;
                    }
                };
            },

            getEventField : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog, valueDefault) {
                var fieldHandler = eventFieldHandlers[field.type](field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);

                var fieldSelected = $(fieldHandler.getElementId(), fieldsDiv);
                //check if exist field, enable that field.
                if (fieldSelected.length) {
                    //check dose who field has required?
                    if (field.required) {
                        var fieldRequired = fieldSelected.find("label:first span.icon-required");
                        if (fieldRequired.length === 0) {
                            fieldSelected.find("label:first").append($("<span/>", {"class": "aui-icon icon icon-required" }));
                        }
                    } else {
                        fieldSelected.find("label:first span.icon-required").remove();
                    }
                    fieldsDiv.append(fieldSelected.show());

                    if (field.type === "repeats") {
                        var freqSelect = $("select[name='freq-select']", fieldsDiv);
                        var freqInterval = $("input[name='interval']", fieldsDiv);
                        var freqUntil = $("input[name=until]", fieldsDiv);
                        var isRepeatEnd = !(freqUntil.attr("readonly") || freqUntil.attr("disabled"));
                        var recurButtons = $("input[name=recur-end]", fieldsDiv);

                        if(field.defaultRecur) {
                            fieldSelected.data("repeatData", { repeat: freqSelect.val(), interval: freqInterval.val() , repeatEnd: isRepeatEnd, until: freqUntil.val()});
                            // Modify repeat controls appropriately.
                            freqSelect.val("YEARLY").change();
                        } else {
                            var repeatData = fieldSelected.data("repeatData");
                            if(repeatData) {
                                freqSelect.val(repeatData.repeat).change(); //for select repeat
                                freqInterval.val(repeatData.interval); //for interval repeat
                                if(repeatData.repeatEnd && repeatData.repeatEnd == true) {
                                    $(recurButtons[1]).attr("checked", "checked");
                                    freqUntil.removeAttr("readonly").val(repeatData.until).tcDatePicker("enable");; //for until repeat
                                } else {
                                    $(recurButtons[0]).attr("checked", "checked");
                                }

                                fieldSelected.removeData("repeatData");
                            }
                        }
                    }
                } else {
                    fieldHandler.add();
                    fieldHandler.init();
                }

                if(field.copyto && valueDefault !== undefined) {
                    fieldHandler.setData(valueDefault);
                }

                return fieldHandler;
            },

            reminder : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                   add : function () {
                       var eventTypeReminder = CalendarPlugin.getCurrentReminderForEventType(calendarDiv, eventEditDialog);
                       $(fieldsDiv).find("#field-text-" + field.id).remove(); //remove old field reminder
                       var hasReminderPeriod = eventTypeReminder && eventTypeReminder.periodInMins > 0;
                       //Add field reminder when edit/create
                       fieldsDiv.append(
                           Confluence.TeamCalendars.Templates.Fields.inforReminderLabel({
                               "id" : field.id,
                               "label" : field.label,
                               "reminderPeriodText" : hasReminderPeriod ? AJS.format(AJS.I18n.getText('calendar3.reminder.event.editdialog.period.text'), CalUtil.showInforReminder(eventTypeReminder.periodInMins)) : AJS.I18n.getText('calendar3.reminder.event.editdialog.period.none.text'),
                               "reminderPeriod": hasReminderPeriod ? eventTypeReminder.periodInMins : 0
                           })
                       );
                   },
                   init : function() {
                   },
                   getData : function() {
                       return {};
                   },
                   getElementId : function() {
                       return ""; //always create new field reminder when change event type
                   }
                };
            },

            /**
            * Adds hidden fields found on all JIRA event forms
            */
            "jira-hidden-fields" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.hiddenJiraFields());
                    },
                    init : function() {
                        var subCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event),
                            callbackHandler = CalendarPlugin.getRenderedMacroCallbackHandler(calendarDiv),
                            helper = Confluence.TeamCalendars.getEditCalendarDialogHelper(fieldsDiv, subCalendar, callbackHandler);

                        if (!subCalendar || !subCalendar.id) {
                            helper.setSubCalendarIdField("");
                            helper.setNameField("");
                            helper.setColorField("");
                            helper.setSpaceToDefault();
                        } else {
                            helper.setSubCalendarIdField(subCalendar.id);
                            helper.setNameField(subCalendar.name);
                            helper.setColorField(subCalendar.color);
                            helper.setSpaceKeyField(subCalendar.spaceKey);
                        }
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
            * A jira server selector
            */
            "jira-server-select" : function (field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.serverSelector());
                        JiraEventFieldProcess.getServerSelect(fieldsDiv).disable();
                    },
                    init : function() {
                        var serverSelect = JiraEventFieldProcess.getServerSelect(fieldsDiv),
                            spinner = $(".field-group-jira-server .spinner");

                        // clean up callbacks
                        AJS.unbind("server-select-ready.teamcalendars");
                        AJS.unbind("server-select-change.teamcalendars");

                        function populateDropDown (data) {
                            var subCalendar = JiraEventFieldProcess.getSubCalendarForEvent(CalendarPlugin, calendarDiv, event);

                            $.each(data.jiraLinks, function (jiraLinkIdx, jiraLink) {
                                var optionItem = $("<option/>", {
                                    "value":            jiraLink.id,
                                    "text":             jiraLink.name,
                                    "data-display-url": jiraLink.displayUrl
                                });

                                if (data.jiraLinks.length === 1) {
                                    //TEAMCAL-1719 - if there is only one option an then selected by default
                                    optionItem.prop('selected', true);
                                    var fieldGroup = serverSelect.parent();
                                    fieldGroup.children().hide();
                                    //stype layout when hide all children field
                                    fieldGroup.css("min-height", 0);
                                    // if already hide don't need to hide it again
                                    $('.oauth-error', fieldGroup).show();
                                }
                                optionItem.appendTo(serverSelect);
                            });

                            if (data.jiraLinks.length > 1)
                                $(".field-group-jira-server", fieldsDiv).removeClass("hidden");
                            
                            if(subCalendar) {
                                if(subCalendar.applicationId) {
                                    serverSelect.val(subCalendar.applicationId);
                                }
                                else if(subCalendar.sourceSubCalendar && subCalendar.sourceSubCalendar.applicationId){
                                        serverSelect.val(subCalendar.sourceSubCalendar.applicationId);
                                }
                            }
                        }

                        serverSelect.change(function() {
                            AJS.trigger("server-select-change.teamcalendars", fieldsDiv);
                        });

                        //Do ajax call to get available JIRA servers
                        $.ajax({
                            cache: false,
                            error: function(jqXHR, textStatus, errorThrown) {
                                CalendarPlugin.showAjaxError(fieldsDiv.parent().parent(), jqXHR, textStatus, errorThrown);
                            },
                            success: function(data) {
                                populateDropDown(data);
                                //Fire an event for anyone waiting for the drop down to be populated
                                AJS.trigger("server-select-ready.teamcalendars", fieldsDiv);
                            },
                            complete: function() {
                                serverSelect.enable();
                                spinner.hide();
                            },
                            timeout : Confluence.TeamCalendars.ajaxTimeout,
                            url: Confluence.TeamCalendars.getCalendarServiceBaseUrl("/jira/jiraLinks")
                        });
                        //Set default value
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
            * A JIRA project selector.
            */
            "jira-project-select" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.Fields.projectSelector());
                        $(".tc-jira-project", fieldsDiv).disable();
                    },
                    init : function() {
                        function serverChanged(context) {
                            if (context.id === fieldsDiv[0].id) { //event may have been fired from another panel
                                $(".oauth-approve", fieldsDiv).remove();
                                JiraEventFieldProcess.populateProjectList(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                            }
                        }

                        AJS.bind("server-select-ready.teamcalendars", function(event, context) {
                            serverChanged(context);
                        });

                        AJS.bind("server-select-change.teamcalendars", function(event, context) {
                            serverChanged(context);
                        });
                    },
                    getData : function() {
                        //Should never be called as JIRA events use a custom handler
                    }
                };
            },

            /*
             * Handles most of the fields on the JIRA issues date fields form. Lumped together for historical reasons.
             */
            "jira-dates" : function(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog) {
                return {
                    add : function () {
                        fieldsDiv.append(Confluence.TeamCalendars.Templates.editJiraCalendar());
                    },
                    init : function () {
                        //This is fired by the jira-server-select field in event-field-handlers.js
                        AJS.bind("server-select-ready.teamcalendars", function() {
                            JiraEventFieldProcess.initJiraDate(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                        });

                        AJS.bind("server-select-change.teamcalendars", function() {
                            JiraEventFieldProcess.initJiraDate(field, fieldsDiv, event, CalendarPlugin, calendarDiv, eventEditDialog);
                        });
                    },
                    getData : function() {
                         //Should never be called as JIRA events use a custom handler
                    }
                };
            }
        };

        return eventFieldHandlers;
    }
);
