AJS.toInit(function($) {
    var eventMacro = function(occurrence) {

        occurrence = occurrence || "";

        var eventMacroContainer = $('#event-macro-container-' + occurrence),
            attendeesCountMsgElement = $('#event-counter-' + occurrence),
            attendeesCounterElement = $('#attendeeCount-' + occurrence),
            messageContainer = $('#event-messages-' + occurrence),
            replyListDiv = $('#reply-list-' + occurrence, eventMacroContainer),
            waitingListDiv = $('#waiting-list-' + occurrence, eventMacroContainer),
            waitingListSize = parseInt($('#waitingListSize-' + occurrence).val()),
            hideReplies = eventMacroContainer.hasClass('hide-replies'),
            waitingListEnabled = $('#waitingListEnabled-' + occurrence).val() === 'true',
            replyLimit = $('#replyLimit-' + occurrence).val();

        /**
         * Given an element corresponding to a reply (or one of its children), returns the unique id of that reply
         * @method getReplyId
         * @param {HTMLElement} element DOM element of the reply or one of its children
         * @return {String} Reply id
         */
        function getReplyId(element) {
            var id = $(element).closest('tr').attr('id');
            id = id.substring(('event-'  + occurrence + '-reply-').length, id.length);
            return id;
        }

        /**
         * Displays a message at the top of the event macro
         * @method displayMessage
         * @param {String} message Message to show
         * @param {String} type Type of message, one of 'info' or 'error'
         */
        function displayMessage(message, type) {
            messageContainer.empty();
            AJS.messages[type](messageContainer, {
                body: message,
                closeable: true
            });
        }

        /**
         * Displays an error message at the top of the event macro
         * @method displayErrorMessage
         * @param {String} message Message to show
         */
        function displayErrorMessage(message) {
            displayMessage(message, 'error');
        }

        /**
         * Displays a success message at the top of the event macro
         * @method displaySuccessMessage
         * @param {String} message Message to show
         */
        function displaySuccessMessage(message) {
            displayMessage(message, 'success');
        }

        /**
         * Checks to see if a given string is of the form of an email address
         * @method isEmailAddress
         * @param {String} email
         * @return {Boolean}
         */
        function isEmailAddress(email) {
            return email.match(/^[a-zA-Z0-9\+._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$/)
        }

        /**
         * Creates and returns html for a reply row in the replies table
         * @method buildReplyElement
         * @param {Object} reply Object containing reply info
         * @return {String} Html for the reply row
         */
        function buildReplyElement(reply) {
            reply.checked = reply.confirm ? ' checked="checked"' : '';
            if (reply.email) {
                reply.email = 'mailto:' + reply.email;
                reply['nameHtml:html'] = Confluence.TeamCalendars.EventMacro.Templates.replyName({
                    email : reply.email,
                    name : reply.name
                });
            } else {
                reply['nameHtml'] = reply['name'];
            }
            if (reply.guests) {
                reply['guests'] = reply.guests;
            } else {
               reply['guests'] = 0;
            }

            $.each(reply.customColumnsMap, function(key, value) {
                reply[key] = value;
            });

            $.each(reply.customCheckboxsMap, function(key, value) {
                reply[key] = value ? AJS.I18n.getText("yes.name") : AJS.I18n.getText("no.name");
            });

            return AJS.template.load("reply-row-" + occurrence).fill(reply);
        }

        /**
         * Adds a new reply row to the replies table after a successful Attend.
         * @method addReplyElement
         * @param {Object} response Object containing reply info
         * @param {HTMLElement} container Reply table
         */
        function addReplyElement(response, container) {
            var row = buildReplyElement(response.reply);
            if (!response.reply.inWaitingList) {
                container.append(row);
                highlight(response.reply.id);
                incrementAttendeesCount();
                updateTable(response.totalResponders);
            } else {
                // add to waiting list instead
                var waitingListContainer = waitingListDiv.find("tbody");
                waitingListContainer.append(row);
                incrementWaitingListSize();
                if (isWaitingListVisible()) {
                    highlight(response.reply.id);
                } else {
                    displaySuccessMessage(AJS.I18n.getText("calendar3.event.success.added.to.waiting.list"));
                }
            }
            updateFooter();
        }

        /**
         * Fades out the specified element
         * @method fadeOut
         * @param {HTMLElement} element Element to fade out
         * @param {Function} callbackFn (optional) Function to run after fade out completes
         */
        function fadeOut(element, callbackFn) {
            var delay = 500;
            if (element.animate) {
                element.animate({opacity: '0'}, delay, 'linear', function() {
                    element.hide();
                    callbackFn && callbackFn();
                });
            } else {
                element.hide();
            }
        }

        /**
         * Fades in the specified element
         * @method fadeIn
         * @param {HTMLElement} element Element to fade in
         * @param {Function} callbackFn (optional) Function to run after fade out completes
         */
        function fadeIn(element, callbackFn) {
            var delay = 500;
            if (element.animate) {
                element.css({opacity: '0'}).show();
                element.animate({opacity: '1'}, delay, 'linear', function() {
                    callbackFn && callbackFn();
                });
            } else {
                element.show();
            }
        }

        /**
         * Briefly "highlights" (adds yellow background to) the specified element
         * @method highlight
         * @param id the reply id element Element to highlight
         */
        function highlight(id) {
            var element = $('#event-' + occurrence + '-reply-' + id),
                highlightLength = 3000; // length highlight will last, in milliseconds
            element.css({backgroundColor: '#ffd'});
            setTimeout(function() {
                element.css({backgroundColor: ''});
            }, highlightLength);
        }

        /**
         * Updates the number of attendees in the UI
         * @method incrementAttendeesCount
         */
        function incrementAttendeesCount() {
            var attendees = parseInt(attendeesCounterElement.val()) + 1;
            attendeesCounterElement.val(attendees);
            return attendees;
        }

        /**
         * Updates the number of attendees in the UI
         * @method decrementAttendeesCount
         */
        function decrementAttendeesCount() {
            var attendees = parseInt(attendeesCounterElement.val()) - 1;
            attendeesCounterElement.val(attendees);
            return attendees;
        }

        /**
         * Updates the number of attendees in the UI, and closes off replies if the event is full
         * @method updateAttendeesCount
         * @param {Number} count The current number of attendees
         */
        function updateAttendeesCount(count) {
            var text;

            if (replyLimit > 0) { // if there is a limit to # of attendees...
                var spotsLeft = replyLimit - count,
                    // close replies if full, or reopen if they have opened up
                    replyButton = $('button.event-attend', eventMacroContainer);
                if (spotsLeft <= 0) {
                    if (waitingListEnabled) {
                        // support waiting list
                        replyButton.removeAttr("disabled");
                        replyButton.attr("title", AJS.I18n.getText('calendar3.event.join.waiting.list.desc'));
                        replyButton.html(AJS.I18n.getText('calendar3.event.join.waiting.list.label'));
                    } else {
                        replyButton.attr("disabled", true);
                        replyButton.html(AJS.I18n.getText('calendar3.event.full'));
                    }
                } else {
                    replyButton.removeAttr("disabled");
                    replyButton.html(AJS.params['eventButtonReplyText-'+occurrence]);
                    if (waitingListEnabled) {
                        replyButton.removeAttr("title");
                    }
                }
                text = AJS.format(spotsLeft == 1 ? AJS.I18n.getText('calendar3.event.msg.spot.remaining') : AJS.I18n.getText('calendar3.event.msg.spots.remaining'), spotsLeft);
            } else {
                text = AJS.format(count == 1 ? AJS.I18n.getText('calendar3.event.msg.num.person.attending') : AJS.I18n.getText('calendar3.event.msg.num.people.attending'), count);
            }
            attendeesCountMsgElement.text(text);
        }

        /**
         * Updates the replies table after a reply is added or removed
         * @method updateTable
         */
        function updateTable(count) {
            updateAttendeesCount(count);
        }

        /**
         * Updates the footer after a reply is added or removed.
         * @method updateFooter
         */
        function updateFooter() {
            $(".attendees-report").remove();
            $(".event-footer").append('<span class="attendees-report">' + AJS.I18n.getText('calendar3.event.msg.refresh') + '</span>');
        }

        function getReplyById(id) {
            return $('#event-' + occurrence + '-reply-' + id);
        }
        /**
         * Fades out and removes a reply
         * @method fadeoutAndRemoveReply
         * @param {String} id Reply id
         */
        function fadeoutAndRemoveReply(id, response) {
            var reply = getReplyById(id);
            fadeOut(reply.add('#event_error_or_message-' + occurrence), function() {
                var isWaitingList = reply.data('isWaitingList');
                reply.remove();
                if (waitingListEnabled && response) {
                    // there is a promoted reply from the waiting list, remove it from waiting list and add it to reply table
                    var waitingListReply = getReplyById(response.id);
                    waitingListReply.remove();
                    var promotedReply = buildReplyElement(response);
                    replyListDiv.find('tbody').append(promotedReply);
                    highlight(waitingListReply.id);
                    decrementWaitingListSize();
                    // no change to the number of responders but other metrics might be changed
                } else if (isWaitingList) {
                    // it's deleting a waiting list item
                    decrementWaitingListSize();
                } else {
                    // a reply is deleted and no waiting list items to be promoted
                    var numOfAttendees = decrementAttendeesCount();
                    updateTable(numOfAttendees);
                }
                updateFooter();
            });
        }

        /**
         * Deletes a given reply and removes it from the dom
         * @method deleteReply
         * @param {Event} e Event object
         */
        function deleteReply(e) {
            var id = getReplyId(e.target);
            e.preventDefault();
            $.ajax({
                url: AJS.format(AJS.params['eventResponseUrl-' + occurrence], id),
                type: 'delete',
                success: function(response) {
                    fadeoutAndRemoveReply(id, response);
                },
                error: function(request) {
                    // Note that we can pass an invalid id and occurrence, and the server will still return a 200 response
                    displayErrorMessage(AJS.I18n.getText('calendar3.event.error.deleting.reply'));
                }
            });
        }

        /**
         * Posts a reply and handles success/failure
         * @method submitReply
         * @param {Event} e Event object
         */
        function submitReply(e) {
            var customColumns = $(".customColumn", $("#event_form-" + occurrence)),
                customCheckboxs = $(".customCheckbox", $("#event_form-" + occurrence)),
                customColumnsMap = {},
                customCheckboxsMap = {};

            e.preventDefault();
            var table = replyListDiv.find('table.event-table'),
                    nameTextField = $("#name-" + occurrence),
                    data = {
                        name: nameTextField.val(),
                        email: $("#email-" + occurrence).val(),
                        guests: $("#guests-" + occurrence).val(),
                        comment: $("#comment-" + occurrence).val()
                    };

            $.each(customColumns, function(i, column) {
                var name = $(column).attr("name");
                var value = $(column).val();
                customColumnsMap[name] = value;
            });

            $.each(customCheckboxs, function(i, column) {
                var name = $(column).attr("name");
                var value = $(column).is(':checked');
                customCheckboxsMap[name] = value;
            });

            data['customColumnsMap'] = customColumnsMap;
            data['customCheckboxsMap'] = customCheckboxsMap;

            if (!data.name) {
                nameTextField.addClass('error');
                $("name_error_box-" + occurrence).css({display: 'inline'});
                return false;
            }
            $.ajax({
                url: AJS.params['eventReplyUrl-' + occurrence],
                data: JSON.stringify(data),
                contentType: 'application/json',
                dataType: 'json',
                type: 'post',
                success: function(response) {
                    toggleFormVisibility();
                    if (hideReplies) {
                        var numOfAttendees = incrementAttendeesCount();
                        updateAttendeesCount(numOfAttendees);
                        displaySuccessMessage(AJS.I18n.getText("calendar3.event.success.reply.received"));
                    } else {
                        addReplyElement(response, table.find('tbody'));
                    }
                },
                error: function(request) {
                    if (request.responseText == "duplicate") {
                        displayErrorMessage(AJS.I18n.getText("calendar3.event.error.replying.duplicate"));
                    }
                    else {
                    // Note that we can pass an invalid id and occurrence, and the server will still return a 200 response
                    displayErrorMessage(AJS.I18n.getText("calendar3.event.error.replying"));
                    }
                }
            });
            return false;
        }

        /**
         * Toggles the "status" for a given reply
         * @method toggleStatus
         * @param {Event} e Event object
         */
        function toggleStatus(e) {
            var target = $(e.target),
                id = getReplyId(e.target),
                tr = target.closest('tr'),
                url = AJS.format(AJS.params['eventResponseUrl-' + occurrence], id),
                isChecked = target.prop('checked'),
                nameElement = tr.find('td.event-name'),
                name = nameElement.attr('data-name'),
                personalUrl = nameElement.attr('data-url'),
                date = tr.find('td.event-time').attr('data-time'),
                data = {
                    id : id,
                    name: tr.find('td.event-name').attr('data-name'),
                    confirm: isChecked
                };

            if (personalUrl) {
                data.email = personalUrl;
            }
            if (date) {
                data.date = date;
            }

            $.ajax({
                url: url,
                contentType: 'application/json',
                dataType: 'json',
                data: JSON.stringify(data),
                type: 'put',
                error: function(request) {
                    displayErrorMessage(AJS.I18n.getText('calendar3.event.error.toggling.status'));
                }
            });
        }

        /**
         * Shows or hides the submit reply form
         * @method toggleFormVisibility
         * @param {Event} e Event object
         */
        function toggleFormVisibility(e, callbackFn) {
            var formContainer = $('#event-input-box-' + occurrence);
            e && e.preventDefault();
            if (formContainer.is(':visible')) {
                formContainer.slideUp(350, callbackFn);
                enableSubmitButton();
            } else {
                formContainer.slideDown(400, callbackFn);
                disableSubmitButton();
            }
        }

        function enableSubmitButton() {
            $('button.event-attend', eventMacroContainer).enable();
        }

        function disableSubmitButton() {
            $('button.event-attend', eventMacroContainer).disable();
        }

        function isWaitingListVisible() {
            return waitingListDiv.is(':visible');
        }

        function toggleWaitingListVisibility(e) {
            e && e.preventDefault();
            if (waitingListDiv.is(':visible')) {
                waitingListDiv.slideUp(350);
            } else {
                waitingListDiv.slideDown(400);
            }
        }

        function incrementWaitingListSize() {
            waitingListSize++;
            updateWaitingListSize();
        }

        function decrementWaitingListSize() {
            if (waitingListSize > 0) {
                waitingListSize--;
            }
            updateWaitingListSize();
        }

        function getExpandWaitingListLink() {
            return $("#expand-waiting-list", eventMacroContainer);
        }

        function getWaitingListFooter() {
            return $("#event-footer-waiting-list", eventMacroContainer);
        }

        function updateWaitingListSize() {
            if (waitingListSize == 0) {
                getWaitingListFooter().hide();
                // hide the waiting list if it's currently being shown
                if (isWaitingListVisible()) {
                    toggleWaitingListVisibility();
                }
            } else {
                getWaitingListFooter().show();
                getExpandWaitingListLink().text(
                        AJS.format(waitingListSize == 1 ?
                                AJS.I18n.getText('calendar3.event.waiting.list.expand.one') :
                                AJS.I18n.getText('calendar3.event.waiting.list.expand.n'), waitingListSize));
            }
        }

        AJS.toInit(function() {
            $('button.event-attend', eventMacroContainer).click(toggleFormVisibility);
            $('form.event-form', eventMacroContainer).submit(submitReply);
            $('a.event-cancel', eventMacroContainer).click(function(e) {
                enableSubmitButton();
                e.preventDefault();
                $(e.target).closest('div.event-input-box').slideUp();
            });
            $('a.event-email-list-fetcher', eventMacroContainer).click(function(e) {
                e.preventDefault();
                $.ajax({
                    url: AJS.params['eventReplyUrl-' + occurrence]+'?type=emails',
                    success: function(response){
                        window.location = "mailto:" + (response.responseText || response);
                    }
                });
            });
            $('input.event-status-checkbox', eventMacroContainer).live('click', toggleStatus);
            $('a.event-delete', eventMacroContainer).live('click', deleteReply);

            $(".number-input", eventMacroContainer).keydown(function(event) {
                // Allow: backspace, delete, tab, escape, and enter
                if ( event.keyCode == 46 || event.keyCode == 8 || event.keyCode == 9 || event.keyCode == 27 || event.keyCode == 13 ||
                    // Allow: Ctrl+A
                    (event.keyCode == 65 && event.ctrlKey === true) ||
                    // Allow: home, end, left, right
                    (event.keyCode >= 35 && event.keyCode <= 39)) {
                    // let it happen, don't do anything
                    return;
                }
                else {
                    // Ensure that it is a number and stop the keypress
                    if (event.shiftKey || (event.keyCode < 48 || event.keyCode > 57) && (event.keyCode < 96 || event.keyCode > 105 )) {
                        event.preventDefault();
                    }
                }
            });
            updateWaitingListSize();
            getExpandWaitingListLink().click(function(event) {
                toggleWaitingListVisibility(event);
            });
        });

        return {
            /**
             * Returns the unique "occurrence" corresponding to this event macro
             * @method getOccurrence
             * @return {Number} Integer representing the occurrence of this macro on page
             */
            getOccurrence: function() {
                return occurrence;
            }
        };

    };

    //Get each macro from the dom
    $(".event-macro-container").each(function(i, element) {
        //Get the occurrence id from the element id
        eventMacro(element.id.split('-')[3]);
    });
});