require([
    "jquery",
    "underscore",
    "backbone",
    "cq",
    "cq/util/analytics",
    "cq/widget/loading",
    "cq/widget/topicwatchers",
    "cq/widget/topicwatchers-view"
], function (
    $,
    _,
    Backbone,
    CQ,
    analytics,
    loading,
    TopicWatchers,
    TopicWatchersView) {

    var EditTopicWatchers = TopicWatchers.extend({
        updateWatcher: function(watcher, method) {
            // Do nothing
        }
    });

    var EditTopicWatchersView = TopicWatchersView.extend({
        render: function() {
            var template = CQ.Templates.widget.topicWatchers.watchers({watchers: this.collection.toJSON()});
            this.$el.html(template);

            return this;
        },

        notifyWhenUserAlreadyWatching: function() {
            // Do nothing
        }
    });

    var EditTopicDialogModel = CQ.Model.extend({
        initialize: function(data, options) {
            if (data.topic) {
                this.id = data.topic.id;
                this.attributes["name"] = data.topic.name;
                this.attributes["description"] = data.topic.description;
                this.attributes["featured"] = data.topic.featured;
            }

            this.topicWatchers = data.topicWatchers;
            this.topicWatchers.on("add", this.addWatcher, this);
            this.topicWatchers.on("remove", this.removeWatcher, this);
            this.topicWatchers.on("reset", this.resetWatchers, this);
        },

        urlRoot: function() {
            return AJS.contextPath() + "/rest/questions/1.0/topic/";
        },

        resetWatchers: function() {
            this.attributes["watchers"] = this.topicWatchers.pluck("userKey");
        },

        addWatcher: function(watcher) {
            var watchers = _.clone(this.get("watchers"));
            var userKey = watcher.get("userKey");

            if (watchers.indexOf(userKey) == -1) {
                watchers.push(userKey);
                this.set("watchers", watchers);
            }
        },

        removeWatcher: function(watcher) {
            var watchers = _.clone(this.get("watchers"));
            var userKey = watcher.get("userKey");

            var index = watchers.indexOf(userKey);
            if (index !== -1) {
                watchers.splice(index, 1);
                this.set("watchers", watchers);
            }
        }
    });

    var EditTopicDialogView = CQ.View.extend({
        events: {
            "keyup #cq-topic-name" : "onKeyDown"
        },

        initialize: function() {
        },

        save: function() {
            this.model.set("name", this.normalizeName(this.$("#cq-topic-name").val()));
            this.model.set("description", this.$("#cq-topic-description").val());
            this.model.set("featured", this.$("#cq-topic-featured").is(":checked"));
        },

        onKeyDown: function() {
            var name = this.normalizeName(this.$("#cq-topic-name").val());
            if (name.length > 0) {
                this.$(".cq-topic-representation").html(CQ.Templates.widget.editTopic.topic({topicName: name}))
            } else {
                this.$(".cq-topic-representation").empty();
            }
        },

        normalizeName: function(name) {
            return name.replace(/\s+/g, '-').toLowerCase();
        }
    });

    function EditTopicDialog(topic) {
        this.adminMode = AJS.Meta.get("cq-is-admin");
        this.createMode = (topic === undefined);
        this.topic = topic;
        this.logoSelected = false;

        this.topicWatchers = new EditTopicWatchers([], { topicId: !this.createMode ? topic.id : -1 });
        this.model = new EditTopicDialogModel({topic: topic, topicWatchers: this.topicWatchers});

        var offsetX, offsetY, width;
        var $dialogBody = $(CQ.Templates.widget.editTopic.dialogBody({topic: topic}));
        var $watchersBody = $(CQ.Templates.widget.editTopic.watchersBody({watchers: []}));
        var self = this;

        this.buildPostUrl = function() {
            var scaledWidth = this.getPreviewImage().width();
            var params = "?width=" + this.cropProperties.w + "&offsetX=" + this.cropProperties.x + "&offsetY=" + this.cropProperties.y + "&scaledWidth=" + scaledWidth;
            return AJS.contextPath() + "/rest/questions/1.0/topic/" + this.topic.id + "/logo" + params;
        };

        this.show = function() {
            this.view = new EditTopicDialogView({model: self.model, el: $("#cq-edit-topic-dialog")});

            dialog.gotoPage(0);
            dialog.gotoPanel(0);
            dialog.show();
        };

        this.hide = function (dialog) {
            dialog.remove();
        };

        this.clearErrors = function() {
            $("#cq-edit-topic-dialog").find(".error").remove();
        };

        this.saveButtonHandler = function (dialog) {
            var metadataDfd;
            var logoDfd;
            var $saveButton = $("#cq-edit-topic-dialog").find(".cq-save-button");
            this.clearErrors();

            this.view.save();
            if (this.createMode) {
                metadataDfd = this.saveMetadata();

                metadataDfd.then(function(data) {
                    self.topic = data;
                    logoDfd = self.saveLogo();

                    $.when(logoDfd).done(function () {
                        // redirect to questions in topic page
                        window.location.pathname = AJS.contextPath() + self.topic.url;
                    });
                }).fail(function (jqXhr, status, errorThrown) {
                    // show errors
                    if (jqXhr.status === 400) {
                        var error = JSON.parse(jqXhr.responseText);
                        var $field = $("#cq-edit-topic-dialog").find("[name='" + error.field + "']");
                        _.each(error.messages, function (message) {
                            $field.after(aui.form.fieldError({
                                message: message
                            }));
                        });
                        dialog.prevPage();
                    }
                }).always(function() {
                    $saveButton.removeAttr("disabled")
                });

                // disable button while saving
                $saveButton.attr("disabled", "disabled");

            } else {
                metadataDfd = this.saveMetadata();
                logoDfd = this.saveLogo();

                // just hide dialog if nothing changed
                if (!metadataDfd && !logoDfd) {
                    dialog.hide();
                    return;
                }

                // save metadata and logo and reload on success.
                var dfd = $.when(metadataDfd, logoDfd).done(function () {
                    window.location.reload();
                });

                // disable button while saving
                $saveButton.attr("disabled", "disabled");
                dfd.always(function() {
                    $saveButton.removeAttr("disabled");
                })
            }

        };

        this.saveMetadata = function() {
            return this.model.save();
        };

        this.saveLogo = function () {
            if (!this.logoSelected) {
                return false;
            }

            // use our own deferred because .ajaxSubmit doesn't return one
            var dfd = new $.Deferred();

            $dialogBody.find("form").ajaxSubmit({
                url: this.buildPostUrl(),
                dataType: "text",
                success: function() {
                    dfd.resolve();
                },
                error: function() {
                    dfd.reject();
                }
            });

            dfd.fail(function() {
                $dialogBody.find(".cq-preview-image-wrapper").html(CQ.Templates.widget.editTopic.uploadError());
            });

            return dfd;
        };

        this.showLogoImage = function(src) {
            $dialogBody.find(".cq-preview-image-wrapper").html(CQ.Templates.widget.editTopic.previewImage({src: src}));

            var $previewImage = this.getPreviewImage();
            $previewImage.on("load", function () {
                // setup cropping for images bigger than 48px
                // setTimeout for jcrop IE8 bug: https://github.com/tapmodo/Jcrop/issues/112
                setTimeout(function() {
                    if ($previewImage.width() > 48 && $previewImage.height() > 48) {
                        $previewImage.Jcrop({
                            aspectRatio: 1,
                            minSize: [48, 48],
                            allowSelect: true,
                            drawBorders: false,
                            onSelect: _.bind(function (c) {
                                this.cropProperties = c;
                            }, self),
                            onRelease: _.bind(self.resetCropProperties, self)
                        });
                    }
                }, 0);

                self.resetCropProperties();
                self.logoSelected = true;
            });
        };

        this.readImageWithFileReader = function(e) {
            var files = e.target.files;
            var file = files[0];

            // only process image files
            if (!file.type.match('image.*')) {
                $dialogBody.find(".cq-preview-image-wrapper").html(CQ.Templates.widget.editTopic.fileNotSupported());
                this.logoSelected = false;
            } else {
                var reader = new FileReader();
                reader.onload = function (e) {
                    // show preview
                    var data = e.target.result;
                    self.showLogoImage(data);
                };

                // Read in the image file as a data URL.
                reader.readAsDataURL(file);
            }
        };

        /**
         * Posts the image to the server and gets back a download link for IE.
         */
        this.getImageFromServer = function (e) {
            $dialogBody.find("form").ajaxSubmit({
                dataType: "text",
                url: AJS.contextPath() + "/rest/questions/1.0/topic/tempLogo",
                success: function(response) {
                    response = $.parseJSON(response);
                    if (response && response.downloadPath) {
                        self.showLogoImage(AJS.contextPath() + response.downloadPath);
                    }
                }
            });
        };

        this.resetCropProperties = function() {
            this.cropProperties = {
                w: -1,
                x: -1,
                y: -1
            };
        };

        this.getPreviewImage = function() {
            return $dialogBody.find(".cq-preview-image-wrapper img");
        };

        this.nextButtonHandler = function(dialog) {

            dialog.nextPage();

            var $watchersContainer = $(".cq-edit-topic-watchers");
            this.topicWatchers.on("reset", function () {
                var topicWatchersView = new EditTopicWatchersView({
                    collection: self.topicWatchers
                });

                $watchersContainer.html(topicWatchersView.render().el);
                topicWatchersView.$("input").focus();

                Confluence.Binder.autocompleteUserOrGroup($watchersContainer);
            });

            if (!this.createMode) {
                this.topicWatchers.fetch({reset: true});
                $watchersContainer.append(CQ.Templates.widget.editTopic.spinner());
            } else {
                this.topicWatchers.reset();
            }
        };

        // build dialog
        var dialog = new AJS.Dialog({
            width: 840,
            height: 500,
            id: "cq-edit-topic-dialog"
        });

        var titleFormat = this.adminMode ? "{0} - {1}" : "{0}";
        var modeTitle = (this.createMode ? AJS.I18n.getText("cq.topic.create.header") : AJS.I18n.getText("cq.topic.edit.header"));
        dialog.addHeader(AJS.format(titleFormat, modeTitle, AJS.I18n.getText("cq.topic.edit.header.step1")));
        dialog.addPanel("SinglePanel", $dialogBody);
        dialog.get("panel:0").setPadding(0);

        // Next page
        if (this.adminMode) {
            // Only admin can add watchers to a topic

            dialog.addButton(AJS.I18n.getText("cq.topic.edit.next"), _.bind(this.nextButtonHandler, this), "cq-next-button");
            dialog.addLink(AJS.I18n.getText("cq.topic.edit.cancel"), _.bind(this.hide, this), "#");

            dialog.addPage();

            dialog.addHeader(AJS.format(titleFormat, modeTitle, AJS.I18n.getText("cq.topic.edit.header.step2")));

            dialog.addPanel("SinglePanel2", $watchersBody);
            dialog.get("panel:0").setPadding(0);

            // add "Previous" button to page 1
            dialog.addButton(AJS.I18n.getText("cq.topic.edit.previous"), function(dialog) {
                dialog.prevPage();
            });
        }

        dialog.addButton(AJS.I18n.getText("cq.topic.edit.done"), _.bind(this.saveButtonHandler, this), "cq-save-button");
        dialog.addLink(AJS.I18n.getText("cq.topic.edit.cancel"), _.bind(this.hide, this), "#");

        if (!this.adminMode) {
            var $featuredCheckBox = $dialogBody.find("#cq-topic-featured");
            var $featuredDiv = $featuredCheckBox.parent();
            $featuredCheckBox.attr('disabled', 'disabled');
            $featuredDiv.prop('title', AJS.I18n.getText("cq.topic.edit.form.featured.description")).tooltip();
        }

        // change handler implementation depends on browser capability
        $dialogBody.find("#cq-logo-input").on("change", function (e) {
            if (window.FileReader) {
                self.readImageWithFileReader(e);
            } else {
                self.getImageFromServer(e);
            }
        });

        // make the submit a proper aui-button (so we can disable it during save)
        dialog.popup.element.find(".cq-save-button.button-panel-button").removeClass("button-panel-button").addClass("aui-button");
    }

    function initEditTopic() {
        $(document).on("click", ".cq-edit-topic-link", function() {
            var topicId = $(this).data("topic-id");
            var dfd = $.ajax({
                dataType: "json",
                cache: false,
                url: AJS.contextPath() + "/rest/questions/1.0/topic/" + topicId
            });

            dfd.done(function (topic) {
                var dialog = new EditTopicDialog(topic);
                dialog.show();
            });

            loading.getPageLoadingIndicator().showUntilResolved(dfd);
        });

        $(".cq-create-topic-button, #cq-create-topic").on("click", function() {
            var dialog = new EditTopicDialog();
            dialog.show();

            return false;
        });

        // remove old dialog when you somehow manage to hide instead of removing it
        AJS.bind("hide.dialog", function (e, data) {
            if (data.dialog.id === "cq-edit-topic-dialog") {
                data.dialog.remove();
            }
        });
    }

    $(function() {
        initEditTopic();
    });
});
