define('cq-quick-edit/quick-edit', [
    'ajs',
    'jquery'
], function(
    AJS,
    $
) {
    /**
     * Quick edit plugin entry point
     */
    "use strict";

    /**
     * Start resource loading
     * @returns {$.Deferred} Promise to keep tracking of resource loading
     */
    function getResourcesDeferred(){
        var loadResources = new $.Deferred();
        AJS.CQ.EditorLoader.load(function(){
            setTimeout(function(){
                // the only reason for defering resolution to the next event loop
                // is to be able to reproduce a timeout when automated testing by setting loadingTimeout to 0.
                loadResources.resolve();
            },0);

        }, function(){
            loadResources.reject();
        });
        return loadResources;
    }

    /**
     * Currently, the editor preloading template gets loaded from a velocity template,
     * which contains all the necessary elements to create a full page editor (including things like
     * the editor precursor, breadcrumbs, etc)
     *
     * Ideally, we would want those elements not to be added to the DOM in the first place. Until we change that,
     * let's fix this by hiding everything that is not "default editor" related, and let's allow every
     * specific editor customisation to show them if necessary.
     */
    function hideEverythingNonEditorEssential(){
        $('#editor-precursor').hide();
        $('#rte-savebar').find('.toolbar-split-left').hide();
    }

    /**
     * Default pre initialisation.
     * This will be called after fetching editor resources and hence before tinymce is initialised
     *
     * @param {object} options.$container Editor main container (jquery object)
     * @param {object} options.content Content retrieved by the REST end-point, if any.
     * @param {object} options.$form Form (jquery object)
     * @param {function} options.replayBufferedKeys  Key buffer unblocker function (will replay typed keys)
     */
    function preInitialise (options){
        $(".cq-editor-prompt", options.$container).hide();

        var $preloadContainer = AJS.CQ.EditorLoader.getPreloadContainer();

        if (options.content && options.content.title) {
            setContentTitle($preloadContainer, options.content.title);
        }
        $(".cq-editor-body", options.$container).append($preloadContainer.children());

        hideEverythingNonEditorEssential();
    }

    /**
     * Set the latest page's title to a preloaded container.
     * When page is viewed, its editor container is also loaded but with an old page title.
     * If a user clicks edit, the latest page's title must be loaded.
     * @param $preloadContainer preloaded editor container
     * @param title the latest page's title
     */
    function setContentTitle ($preloadContainer, title) {
        var $title = $preloadContainer.find("#content-title");
        $title.val(title);
    }

    /**
     * If provided, set editor content.
     * @param editor
     * @param initialContent the initialContent to load into the editor, e.g. the current state of an existing page.
     */
    function setEditorContent(editor, initialContent, replayBufferedKeys) {
        if (initialContent) {
            // set initial content
            editor.setContent(initialContent);
            // CONFDEV-19832 - Since we are setting the editor content we also need to updated its start value
            // startContent is what tinymce uses with isDirty to determine if changes occurred
            editor.startContent = editor.getContent({ format: 'raw', no_events : 1 });

            //Ensure first undo step doesn't blow away the content we just set.  This isn't related to the startContent.
            editor.undoManager.clear();
        }
        replayBufferedKeys();
        editor.undoManager.add();
    }

    /**
     * Called after editor initialisation
     *
     * @param {object} options.editor Editor instance
     * @param {object} options.$container Editor main container (jquery object)
     * @param {object} options.content Content retrieved by the REST end-point, if any.
     * @param {object} options.$form Form (jquery object)
     * @param {function} options.replayBufferedKeys  Key buffer unblocker function (will replay typed keys)
     */
    function postInitialise(options) {
        var editorContent = options.content ? options.content.editorContent : '';
        setEditorContent (options.editor, editorContent, options.replayBufferedKeys);
        triggerEditorReadyEvents();
    }

    var handlers = [];

    function triggerEditorReadyEvents() {
        AJS.trigger('quickedit.success');
        AJS.trigger('quickedit.visible');
        AJS.trigger("add-bindings.keyboardshortcuts");
        AJS.trigger("active.dynamic.rte");
    }

    var QuickEdit = {

        /**
         * Register a quick edit handler (top-level, reply handler, etc)
         * @param handler
         */

        register: function(handler){
            handlers.push(handler);
        },

        /**
         * Disable all handlers.
         * When activate a top-level comment, for example, we may want to disable the comment reply handlers
         * @param handler
         */

        disableHandlers: function(){
            $.each(handlers, function(i, current){
                return current.disable();
            });
        },

        /**
         * Enables all handlers
         * @param handler
         */

        enableHandlers: function(){
            $.each(handlers, function(i, current){
                return current.enable();
            });
        },


        /**
         * An object that binds actions to the save bar as necessary
         */

        SaveBarBinder: {
            bind: function(saveHandler, cancelHandler) {
                if (saveHandler){

                }

                if (cancelHandler) {

                }
            }
        },

        /**
         * Activates the editor
         *
         * @param options.fetchContent {object} Deferred object
         * @param options.fallbackUrl {object} Url to fall back to in case of failure. NOTE: this option is deprecated. To be removed in the next major version (5.8 or 6.0). Please use the promise returned to bind custom action if the editor fails to load instead.
         * @param options.$container {object} The container containing the necessary structure to activate the editor within.
         * @param options.saveHandler {object} Save handler.
         * @param options.cancelHandler {object} Cancel handler.
         * @param options.preActivate {object} (optional) Pre-activation handler. Gets executed before the call to fetch resources.
         * @param options.preInitialise {object} (optional) Pre-initialisation handler. Gets executed after the resources are fetched but before the editor gets loaded.
         * @param options.postInitialise {object} (optional) Post-initialisation handler. Gets executed after editor has loaded.
         * @param options.toolbar {object} (optional) Toolbar initialisation options.
         *        Ex: toolbar : { Style : false }
         *        or, when  we implement more granular options: toolbar : { Style : { Bold: true, Italic: false } }
         * @param options.plugins {Array} (optional) List of additional plugins to load
         * @param options.excludePlugins {Array} List of plugins to be excluded from the editor.
         *        Plugins in this list will override plugins from the parameter before.
         * @param options.postDeactivate {function} (optional) callback function execute after destroy editor.
         * @param options.timeoutResources {number} (optional) timeout for editor resources to be loaded (milliseconds)
         * @param options.additionalResources {Array} (optional) additional resources to include via WRM.require
         *
         * @returns promise tracking editor activation
         */
        activateEditor: function(options) {
            function doActivate() {
                var replayBufferedKeys;
                var $deferredInitialisation = new $.Deferred();

                if (AJS.Rte && AJS.Rte.getEditor()) {
                    AJS.debug('there is already an editor open');
                    return $deferredInitialisation.reject('there is already an editor open');
                }

                if (!options.$container) {
                    AJS.logError("activateEditor could not be initialsed: bad arguments", options);
                    return $deferredInitialisation.reject('bad parameters');
                }

                // start capturing typed keys until editor shows up so we can replay them
                replayBufferedKeys = AJS.CQ.BlockAndBuffer.block($(document));

                options.preActivate && options.preActivate();

                /**
                 * Called when all the necessary deferred objects to load the editor get resolved
                 *
                 * @param resources {object} object returned by resource loader
                 * @param content {object} Content, if any
                 */
                function loadEditor (){
                    var initOptions = {
                        $container: options.$container,
                        content: options.content,
                        replayBufferedKeys: replayBufferedKeys
                    };

                    options.preInitialise && options.preInitialise(initOptions);

                    // default, common preInitialise. Goes after custom preinitialisation
                    preInitialise(initOptions);

                    var onInit = function() { // editor ready!
                        // custom and default post-initialisation
                        initOptions.editor = AJS.Rte.getEditor();
                        postInitialise(initOptions);
                        options.postInitialise && options.postInitialise(initOptions);

                        QuickEdit.SaveBarBinder.bind(options.saveHandler, options.cancelHandler);

                        AJS.trigger('rte-quick-edit-ready');
                        AJS.unbind('rte-ready', onInit);
                        $deferredInitialisation.resolve();
                    };

                    AJS.bind("rte-ready", onInit);
                    AJS.bind("rte-destroyed", options.postDeactivate || function() {});

                    AJS.Rte.BootstrapManager.initialise({
                        plugins: options.plugins,
                        toolbar: options.toolbar,
                        excludePlugins: options.excludePlugins,
                        isQuickEdit: true
                    });
                }

                function handleErrorActivatingEditor(e){
                    $deferredInitialisation.reject(e);
                    AJS.logError('Error loading page quick edit. Falling back to normal edit URL...');

                    if (options.fallbackUrl) {
                        AJS.log('This parameter is deprecated. To be removed in the next major version (5.8 or 6.0). Please use the promise returned to bind custom action if the editor fails to load instead.');
                        window.location = options.fallbackUrl;
                    }
                }

                /**
                 * .-
                 *  Main
                 *  Loads the editor when the dependent deferred objects get resolved
                 *  -.
                 */

                // Returns a deferred object form the WRM.require promise so it can be rejected
                function getWRMDeferred(resources) {
                    var deferred = new $.Deferred();
                    WRM.require(resources).done(function(data){
                        deferred.resolve(data);
                    }).fail(function(e){
                        deferred.reject(e);
                    });
                    return deferred;
                }
                // CQ-2327: We don't need to use the timeoutFn for the following deferred objects
                // because we don't ever want to fall back to the slow editor
                $.when(
                    getResourcesDeferred(),
                    options.fetchContent || $.Deferred().resolve(),
                    options.additionalResources ? getWRMDeferred(options.additionalResources) : $.Deferred().resolve()
                )
                .done(loadEditor)
                .fail(handleErrorActivatingEditor);

                return $deferredInitialisation.promise();
            }

            if (options.closeAnyExistingEditor && AJS.Rte && AJS.Rte.getEditor()) {
                var $deferredActivation = new $.Deferred();

                // Deactive all existing editor
                this.deactivateEditor()
                    .done(function() {
                        // when it's done, activate new editor
                        doActivate().done(function() {
                            $deferredActivation.resolve();
                        }).fail(function(e) {
                            $deferredActivation.reject(e);
                        });
                    })
                    .fail(function() {
                        AJS.debug('Could not deactivate current editor.');
                        $deferredActivation.reject('Could not deactivate current editor.');
                    });

                return $deferredActivation;
            } else {
                return doActivate();
            }
        },

        /**
         * Close and deactivate the editor.
         * @return promise tracking editor deactivation
         */
        deactivateEditor : function () {
            tinymce.execCommand('mceRemoveControl', true, 'wysiwygTextarea');

            // ensure cleanup
            var $preloadContainer = AJS.CQ.EditorLoader.getPreloadContainer().empty();

            // wrapping the editor back
            return AJS.CQ.EditorLoader.getEditorPreloadMarkup().done(function(markup){

                var editorWrapper = $(markup).find(".cq-editor");
                $preloadContainer.append(editorWrapper);
                AJS.loadTemplateScripts(editorWrapper);
                $preloadContainer.hide();

                AJS.trigger("rte-destroyed");
                AJS.unbind("rte-destroyed");
            });
        }
    };

    return QuickEdit;

});

require('cq/module-exporter').exportModuleAsGlobal('cq-quick-edit/quick-edit', 'AJS.CQ.QuickEdit');
