define("cq/util/util", ["jquery", "underscore", "ajs", "exports"], function ($, _, AJS, exports) {

    /**
     * Creates and returns a new function that waits for function execution until the function
     * has not been called for wait seconds.
     *
     * @param fn
     * @param wait
     * @param context
     * @return {Function}
     */
    exports.wait = function(fn, wait, context) {
        var last, timer, args;

        var startTimer = function() {
            timer = setTimeout(function () {
                fn.apply(context, args);
            }, wait);
        };

        return function() {
            args = arguments;
            var now = new Date();
            if (last && now - last < wait) {
                clearTimeout(timer);
            }
            startTimer();
            last = now;
        };
    };


    /**
     * Returns the aui version used in the form of:
     *
     * {
     *  'major' : 5,
     *  'minor' : 3,
     *  'patch' : 0
     * }
     * @returns {*}
     */
    function getAuiVersion() {
        function toInt(num) { return parseInt(num, 10); }

        var versionArray = _.map(AJS.version.split("."), toInt);

        // Add patch version when left out (i.e. 5.3)
        if (versionArray.length == 2) {
            versionArray.push(0);
        }

        var versionObject = _.object(["major", "minor", "patch"], versionArray);


        return versionObject;
    }

    var _version = getAuiVersion();

    var auiVersion = {
        get: function () {
            return _version;
        },
        equals: function (major, minor, patch) {
            minor = minor || 0;
            patch = patch || 0;
            var self = this.get();

            return self.major === major && self.minor === minor && self.patch === patch;
        },
        isLessThan: function (major, minor, patch) {
            minor = minor || 0;
            patch = patch || 0;
            var self = this.get();

            return major > self.major || minor > self.minor || patch > self.patch;
        }
    };

    exports.auiVersion = auiVersion;


    // from: http://www.regexguru.com/2008/11/detecting-urls-in-a-block-of-text/
    // if you change the regex, also change it in UrlUtils.java
    var URL_REGEX = /\b(?:(?:https?|ftp|file):\/\/|www\.|ftp\.)(?:\([-A-Z0-9+&@#\/%=~_|$?!:;,.]*\)|[-A-Z0-9+&@#\/%=~_|$?!:;,.])*(?:\([-A-Z0-9+&@#\/%=~_|$?!:;,.]*\)|[A-Z0-9+&@#\/%=~_|$])/ig;
    exports.linkify = function (text) {
        return text.replace(URL_REGEX, function (match) {
            var href = match;

            // prepend http:// to www. urls
            if (href.indexOf("www.") === 0) {
                href = "http://" + href;
            }

            return "<a href=\"" + href + "\" rel=\"nofollow\">" + match + "</a>";
        });
    };

    exports.isValidEmail = function (email) {
        // http://blog.trojanhunter.com/2012/09/26/the-best-regex-to-validate-an-email-address/
        return /[-0-9a-zA-Z.+_]+@[-0-9a-zA-Z.+_]+\.[a-zA-Z]{2,4}/.test(email);
    };

    exports.namespace = function(namespace) {
        var parts = namespace.split(".");
        var last = window;
        _.each(parts, function (part) {
            last[part] = {};
            last = last[part];
        });
    };

    //If you change this please also check com.atlassian.confluence.plugins.questions.soy.FormatIntegerFunction()
    exports.formatInteger = function(val) {
        if (val >= 999501) {
            return parseFloat((val / 1000000).toFixed(1)) + "M";
        }

        if (val >= 10000) {
            return parseFloat((val / 1000).toFixed(0)) + "k";
        }

        if (val >= 1000) {
            return parseFloat((val / 1000).toFixed(1)) + "k";
        }

        return val;
    };

    exports.replaceHighlightPlaceholders = function (text) {
        var regExp = new RegExp("@@@hl@@@(.+?)@@@endhl@@@", "g");

        return text.replace(regExp, function (match, p1) {
            return "<span class='highlight'>" + p1 + "</span>";
        });
    };

    // support storage mechanism
    var cookieStorage = function() {
        return {
            getItem: function(key) {
                $.cookie(key)
            },

            setItem: function(key, value, expire) {
                $.cookie(key, value, expire)
            }
        }
    }();

    var id = "confluence-questions",
            cqStorageManger = Confluence.storageManager ? Confluence.storageManager(id) : (localStorage ? localStorage : cookieStorage),
            user = AJS.Meta ? AJS.Meta.get("remote-user") : "",
            namespace = "confluence" + "." +  user + "." + id;

    var getKeyStore = function(key) {
        return Confluence.storageManager ? key : namespace + "." + key;
    };

    exports.getItem = function(key) {
        return cqStorageManger.getItem(getKeyStore(key));
    };

    exports.setItem = function(key, value, expire) {
        cqStorageManger.setItem(getKeyStore(key), value, expire);
    };

    exports.isIE = function () {
        return window.navigator.userAgent.indexOf('Trident') > -1;
    };

});