/**
 * Multi purpose events tracker. Dispatches analytics events to registered mappers like GA or AA.
 */
define("cq/util/analytics", [ "underscore", "uri" ], function (_, Uri) {

    /**
     * Sanitize URL for analytics:
     * - Don't track question titles or even question ids
     * - Don't track topic names
     * - Aggregate all question views into one trackable URL (-> /questions/id)
     * - Remove context path
     *
     * @param [path]
     */
    function sanitizePath(path) {
        // URL CASES:
        // /questions/2523394/inline-topic-test
        // /questions/2523394/inline-topic-test2
        // /confluence/questions/2523394
        // /confluence/questions/2523394/edit
        // /confluence/questions/2523394/delete
        // /confluence/questions/2523394/answers/121243
        // /confluence/questions/2523394/answers/121243/edit
        // /confluence/questions/2523394/answers/121243/delete
        // /confluence/questions/topics
        // /confluence/questions/topics/124124/foobar
        // /confluence/questions/topics/foobar/filter/unanswered
        // /confluence/questions/users?username=foobar


        var uri = new Uri(path || document.location.pathname + document.location.search);
        var path = uri.path();

        // remove context path
        var questionPathIndex = path.indexOf("/questions");
        if (questionPathIndex !== -1) {
            path = path.substring(questionPathIndex);
        }

        // remove question ids from url
        // /questions/2523394/inline-topic-test2 -> /questions/id/inline-topic-test2
        path = path.replace(/(\d)+/g, "id");

        // remove friendly name from view question
        // regex inspired by: http://stackoverflow.com/questions/1051989/regex-for-alphanumeric-but-at-least-one-character
        // /questions/id/inline-topic-test2 -> /questions/id
        path = path.replace(/(id)(\/\w*\-[\w\-\.]+)/, "$1");


        // replace topic name with 'id' for topic filter
        // - questions/topics/id/foobar -> questions/topics/id
        path = path.replace(/(topics\/id)(\/\w+)/, "topics/id");

        // delete query params we don't want to track
        uri.deleteQueryParam("flashId");
        uri.deleteQueryParam("title");
        uri.deleteQueryParam("username");

        // deal with *.action urls that you land on if you have to log in before viewing a page
        if (path.indexOf(".action") !== -1) {
            path = path.replace("cq/viewquestion.action", "questions/id");
            path = path.replace("questions.action", "questions");

            uri.deleteQueryParam("id");
            uri.deleteQueryParam("questionTitle");
        }

        uri.setPath(path);
        return uri.toString();
    }

    function elementToString(elem) {
        var str = elem.nodeName.toLowerCase();
        if (elem.id) {
            str += "#" + elem.id;
        } else {
            if (elem.classList.length) {
                str += "." + _.toArray(elem.classList).join(".");
            }
            if (elem.href) {
                str += "[href=" + elem.getAttribute("href") + "]";
            }
        }
        return str;
    }

    var mappers = [];

    var analytics = {
        addMapper: function (mapper) {
            mappers.push(mapper);
        },

        trackPageview: function (page) {
            _.each(mappers, function (mapper) {
                try {
                    mapper.pageview(page);
                } catch (e) {
                    console.log("Error while tracking pageview:", page, e);
                }
            });
        },

        trackEvent: function (noun, verb, data) {
            _.each(mappers, function (mapper) {
                try {
                    mapper.event(noun, verb, data);
                } catch (e) {
                    console.log("Error while tracking event:", noun, verb, e);
                }
            });
        },

        trackClick: function (element, context) {
            context = context || "element";
            var selector = elementToString(element);
            this.trackEvent(context, "click", {selector: selector});
        },

        sanitizePath: sanitizePath
    };

    return analytics;
});

// track pageview on load
require([ "cq/util/analytics", "jquery"], function (analytics, $) {
    /*
     * A setTimeout -- why?
     *
     * Here we are battling a race condition with almond.js. In aa.js we are saying 'require([ "cq/util/analytics" ], callback)
     * to register atlassian analytics as an analytics mapper. In this module we are requiring 'analytics' and call trackPageview()
     * on DOM ready.
     *
     * Now, almond.js has an internal setTimeout in their code to simulate the async require. What can happen is that DOM ready is fired
     * BEFORE aa.js has registered itself due to the timeout and when we call trackPageview() it's basically a no op because analytics.js
     * doesn't know about any mappers yet.
     *
     * This only seems to happen on fast instances that render quickly and don't process much JS on page load (e.g. questions page on OD instances).
     *
     * We can workaround this issue by pushing our call to trackPageview into a setTimeout as well, therefore hopefully executing at the right time.
     * This is sort of ugly, but a quick fix.
     *
     * Note that this is not a bug in almond.js, since it only simulates the behaviour that would happen in 'real' AMD environments as well.
     * Instead we should come up with a solution that makes sure that our mappers are registered before we track a pageView.
     */
    $(function () {
        setTimeout(function () {
            analytics.trackPageview();
        }, 16)
    });
});